package com.newrelic.agent.browser;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import com.newrelic.agent.ConnectionListener;
import com.newrelic.agent.IRPMService;
import com.newrelic.agent.config.AgentConfig;
import com.newrelic.agent.service.AbstractService;
import com.newrelic.agent.service.ServiceFactory;

public class BrowserServiceImpl extends AbstractService implements BrowserService, ConnectionListener {

    private final ConcurrentMap<String, IBrowserConfig> browserConfigs = new ConcurrentHashMap<String, IBrowserConfig>();
    private volatile IBrowserConfig defaultBrowserConfig = null;
    private final String defaultAppName;

    public BrowserServiceImpl() {
        super(BrowserService.class.getSimpleName());
        defaultAppName = ServiceFactory.getConfigService().getDefaultAgentConfig().getApplicationName();
    }

    @Override
    protected void doStart() throws Exception {
        ServiceFactory.getRPMServiceManager().addConnectionListener(this);
    }

    @Override
    protected void doStop() throws Exception {
        ServiceFactory.getRPMServiceManager().removeConnectionListener(this);
    }

    @Override
    public IBrowserConfig getBrowserConfig(String appName) {
        if (appName == null || appName.equals(defaultAppName)) {
            return defaultBrowserConfig;
        }
        return browserConfigs.get(appName);
    }

    @Override
    public boolean isEnabled() {
        return true;
    }

    @Override
    public void connected(IRPMService rpmService, AgentConfig agentConfig) {
        String appName = rpmService.getApplicationName();
        IBrowserConfig browserConfig = BrowserConfigFactory.createBrowserConfig(appName, agentConfig);
        if (appName == null || appName.equals(defaultAppName)) {
            defaultBrowserConfig = browserConfig;
        } else {
            if (browserConfig == null) {
                browserConfigs.remove(appName);
            } else {
                browserConfigs.put(appName, browserConfig);
            }
        }
    }

    @Override
    public void disconnected(IRPMService rpmService) {
        // do nothing
    }

}
