package com.newrelic.agent.attributes;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;
import com.newrelic.agent.config.AgentConfig;
import com.newrelic.agent.config.AgentConfigListener;
import com.newrelic.agent.service.AbstractService;
import com.newrelic.agent.service.ServiceFactory;

import java.util.Map;

/**
 * Determines which attributes can be sent to the various destinations. Current destinations are transaction traces,
 * error collector, transaction events, and browser monitoring.
 */
public class AttributesService extends AbstractService implements AgentConfigListener {
    private final boolean enabled;
    private final String defaultAppName;
    /**
     * The default filter to use. This is always used when auto app naming is not set.
     */
    private volatile AttributesFilter defaultFilter;
    private final Map<String, AttributesFilter> appNamesToFilters;
    private final Map<String, Object> requestUriDummyMap = ImmutableMap.<String, Object>of(AttributeNames.REQUEST_URI, "");

    public AttributesService() {
        super(AttributesService.class.getSimpleName());
        AgentConfig config = ServiceFactory.getConfigService().getDefaultAgentConfig();
        enabled = config.getAttributesConfig().isEnabledRoot();
        defaultAppName = config.getApplicationName();
        defaultFilter = new AttributesFilter(config);
        appNamesToFilters = Maps.<String, AttributesFilter>newConcurrentMap();
        ServiceFactory.getConfigService().addIAgentConfigListener(this);
    }

    @Override
    public boolean isEnabled() {
        return enabled;
    }

    @Override
    protected void doStart() throws Exception {
    }

    @Override
    protected void doStop() throws Exception {
        ServiceFactory.getConfigService().removeIAgentConfigListener(this);
    }

    public boolean captureRequestParams(String appName) {
        return getFilter(appName).captureRequestParams();
    }

    public boolean captureMessageParams(String appName) {
        return getFilter(appName).captureMessageParams();
    }

    public boolean isAttributesEnabledForErrors(String appName) {
        return getFilter(appName).isAttributesEnabledForErrors();
    }

    public boolean isAttributesEnabledForEvents(String appName) {
        return getFilter(appName).isAttributesEnabledForEvents();
    }

    public boolean isAttributesEnabledForTraces(String appName) {
        return getFilter(appName).isAttributesEnabledForTraces();
    }

    public boolean isAttributesEnabledForBrowser(String appName) {
        return getFilter(appName).isAttributesEnabledForBrowser();
    }

    /**
     * Filter attributes based on top-level rules only.
     *
     * Please make sure you're using the correct API.
     *
     * There are more specific APIs:
     *
     * {@link #filterTraceAttributes(String, Map)},
     * {@link #filterErrorAttributes(String, Map)}
     * {@link #filterEventAttributes(String, Map)}
     * {@link #filterBrowserAttributes(String, Map)}
     *
     * @param appName Application name.
     * @param values Values to filter.
     * @return Map of filtered values.
     */
    public Map<String, ? extends Object> filterAttributes(String appName, Map<String, ? extends Object> values) {
        return getFilter(appName).filterAttributes(values);
    }

    public Map<String, ? extends Object> filterErrorAttributes(String appName, Map<String, ? extends Object> values) {
        return getFilter(appName).filterErrorAttributes(values);
    }

    public Map<String, ? extends Object> filterEventAttributes(String appName, Map<String, ? extends Object> values) {
        return getFilter(appName).filterEventAttributes(values);
    }

    public Map<String, ? extends Object> filterTraceAttributes(String appName, Map<String, ? extends Object> values) {
        return getFilter(appName).filterTraceAttributes(values);
    }

    public Map<String, ? extends Object> filterBrowserAttributes(String appName, Map<String, ? extends Object> values) {
        return getFilter(appName).filterBrowserAttributes(values);
    }

    private AttributesFilter getFilter(String appName) {
        if (appName == null || appName.equals(defaultAppName)) {
            return defaultFilter;
        } else {
            AttributesFilter filter = appNamesToFilters.get(appName);
            return (filter == null) ? defaultFilter : filter;
        }
    }

    @Override
    public void configChanged(String appName, AgentConfig agentConfig) {
        // create a new filter any time the configuration has changed for the application name
        if (appName != null) {
            if (appName.equals(defaultAppName)) {
                defaultFilter = new AttributesFilter(agentConfig);
            } else {
                appNamesToFilters.put(appName, new AttributesFilter(agentConfig));
            }
        }
    }

    public String filterRequestUri(String appName, String destination, String uri) {
        Map<String, ?> filteredUriMap = getFilter(appName).filterAttributesForDestination(requestUriDummyMap, destination);
        return filteredUriMap.containsKey(AttributeNames.REQUEST_URI) ? uri : null;
    }
}
