package com.newrelic.agent;

import com.google.common.collect.MapMaker;
import com.newrelic.agent.attributes.AttributeSender;
import com.newrelic.agent.attributes.CustomSpanAttributeSender;
import com.newrelic.agent.bridge.NoOpDistributedTracePayload;
import com.newrelic.agent.bridge.Span;
import com.newrelic.agent.service.ServiceFactory;
import com.newrelic.agent.service.analytics.SpanCategory;
import com.newrelic.agent.service.analytics.SpanEvent;
import com.newrelic.agent.trace.TransactionGuidFactory;
import com.newrelic.agent.tracing.DistributedTraceUtil;
import com.newrelic.agent.util.TimeConversion;
import com.newrelic.api.agent.DistributedTracePayload;
import com.newrelic.api.agent.ExternalParameters;
import com.newrelic.api.agent.tracing.SpanProxy;

import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;

/**
 * This is the Span implementation for Transaction-less Spans only. Transaction Segment Spans are created at the end of a Transaction by creating SpanEvents
 * directly without using this class as an intermediary.
 */
public class SpanImpl implements Span {

    private static final String applicationId = ServiceFactory.getDistributedTraceService().getApplicationId();

    private final String appName;
    private final String guid;
    private final AtomicReference<Float> priority;
    private final SpanCategory category;
    private final boolean sampled;
    private final Map<String, Object> userAttributes;

    private final AttributeSender attributeSender = new CustomSpanAttributeSender(this);
    private final AtomicReference<Boolean> finished = new AtomicReference<>(false);
    private final AtomicReference<SpanProxy> spanProxy = new AtomicReference<>(new SpanProxy());

    private volatile String name;
    private volatile long timestamp;
    private volatile String traceId;
    private volatile long duration;
    private volatile ExternalParameters externalParameters;

    public SpanImpl(String name) {
        this.appName = ServiceFactory.getConfigService().getDefaultAgentConfig().getApplicationName();
        this.guid = TransactionGuidFactory.generateGuid();
        this.timestamp = System.currentTimeMillis();
        this.name = name;
        this.category = SpanCategory.generic;
        this.traceId = this.guid;
        this.priority = DistributedTraceUtil.getPriority();
        this.sampled = DistributedTraceUtil.isSampledPriority(priority.get());
        this.userAttributes = new LazyMapImpl<>(new MapMaker().initialCapacity(8).concurrencyLevel(4));
    }

    @Override
    public Span reportAsExternal(ExternalParameters externalParameters) {
        this.externalParameters = externalParameters;
        return this;
    }

    @Override
    public DistributedTracePayload createDistributedTracePayload() {
        if (ServiceFactory.getConfigService().getDefaultAgentConfig().getDistributedTracingConfig().isEnabled()) {
            spanProxy.get().setTimestamp(timestamp);
            DistributedTracePayloadImpl payload = (DistributedTracePayloadImpl) spanProxy.get()
                    .createDistributedTracePayload(traceId, priority.get(), guid, null);

            // may be null if we didn't receive a priority in the payload
            if (payload != null && payload.priority != null) {
                this.priority.set(payload.priority);
            }

            return payload == null ? NoOpDistributedTracePayload.INSTANCE : payload;
        } else {
            return NoOpDistributedTracePayload.INSTANCE;
        }
    }

    @Override
    public boolean acceptDistributedTracePayload(String payload) {
        if (ServiceFactory.getConfigService().getDefaultAgentConfig().getDistributedTracingConfig().isEnabled()) {
            spanProxy.get().setTimestamp(timestamp);
            boolean accept = spanProxy.get().acceptDistributedTracePayload(payload);
            if (accept) {
                this.traceId = spanProxy.get().getInboundDistributedTracePayload().traceId;
                this.priority.set(spanProxy.get().getInboundDistributedTracePayload().priority);
            }
            return accept;
        } else {
            Agent.LOG.log(Level.FINE, "Not accepting payload, distributed tracing disabled");
            return false;
        }
    }

    @Override
    public boolean acceptDistributedTracePayload(DistributedTracePayload payload) {
        if (ServiceFactory.getConfigService().getDefaultAgentConfig().getDistributedTracingConfig().isEnabled()) {
            spanProxy.get().setTimestamp(timestamp);
            boolean accept = spanProxy.get().acceptDistributedTracePayload(payload);
            if (accept) {
                this.traceId = spanProxy.get().getInboundDistributedTracePayload().traceId;
                this.priority.set(spanProxy.get().getInboundDistributedTracePayload().priority);
            }
            return accept;
        } else {
            Agent.LOG.log(Level.FINE, "Not accepting payload, distributed tracing disabled");
            return false;
        }
    }

    //atm, these can return null if param fails verification
    @Override
    public Span addCustomParameter(String key, String value) {
        attributeSender.addAttribute(key, value, "Span.addCustomParameter");
        return this;
    }

    @Override
    public Span addCustomParameter(String key, Number value) {
        attributeSender.addAttribute(key, value, "Span.addCustomParameter");
        return this;
    }

    @Override
    public Span addCustomParameter(String key, Boolean value) {
        attributeSender.addAttribute(key, value, "Span.addCustomParameter");
        return this;
    }

    @Override
    public Span addCustomParameters(Map<String, Object> params) {
        attributeSender.addAttributes(params, "Span.addCustomParameters");
        return this;
    }

    @Override
    public Span setName(String name) {
        this.name = name;
        return this;
    }

    @Override
    public void setTimestamp(long timestamp) {
        if (timestamp >= 0) {
            this.timestamp = timestamp;
        } else {
            if (Agent.LOG.isFinestEnabled()) {
                Agent.LOG.log(Level.FINEST, "Invalid timestamp: {0} set for span: {1}. Using existing " +
                        "timestamp: {2}.", timestamp, this, this.timestamp);
            }
        }
    }

    @Override
    public void finish() {
        finish(System.currentTimeMillis());
    }

    @Override
    public void finish(long finishMillis) {
        if (finished.compareAndSet(false, true)) {
            duration = finishMillis - timestamp;

            SpanEvent.SpanEventBuilder builder = SpanEvent.builder()
                    .setName(name)
                    .setTraceId(traceId)
                    .setTimestamp(timestamp)
                    .setGuid(guid)
                    .setAppName(appName)
                    .setDurationInSeconds((float) duration / TimeConversion.MILLISECONDS_PER_SECOND)
                    .setCategory(category)
                    .setExternalParameterAttributes(externalParameters)
                    .setUserAttributes(userAttributes);

            DistributedTracePayloadImpl payload = spanProxy.get().getInboundDistributedTracePayload();
            if (payload == null) {
                // acceptDistributedTracePayload not called so this must be a root (and the start of a trace)
                builder.setSampled(sampled)
                        .setParentId(null)
                        .setPriority(priority.get())
                        .setIsRootSpanEvent(true)
                        .setDecider(true);
            } else {
                // acceptDistributedTracePayload was called, this will be a root if the applicationId is
                // different between this service and the inbound payload
                builder.setSampled(payload.sampled)
                        .setParentId(payload.guid)
                        .setPriority(payload.priority)
                        .setIsRootSpanEvent(!payload.applicationId.equals(applicationId));
            }

            ServiceFactory.getServiceManager().getSpanEventsService().storeEvent(builder.build());
        }
    }

    public Map<String, Object> getUserAttributes() {
        return userAttributes;
    }

    public long getStartTime() {
        return timestamp;
    }

    @Override
    public String toString() {
        return "SpanImpl{" +
                "guid='" + guid + '\'' +
                ", name='" + name + '\'' +
                ", traceId='" + traceId + '\'' +
                '}';
    }

}
