package com.newrelic.agent;

import com.google.common.base.Charsets;
import com.newrelic.agent.bridge.AgentBridge;
import com.newrelic.agent.bridge.DistributedTracePayload;
import com.newrelic.agent.service.ServiceFactory;
import com.newrelic.agent.tracing.DistributedTraceService;
import com.newrelic.agent.tracing.DistributedTraceUtil;
import com.newrelic.api.agent.NewRelic;
import com.newrelic.org.apache.axis.encoding.Base64;
import org.json.simple.JSONArray;
import org.json.simple.JSONObject;
import org.json.simple.parser.JSONParser;

import java.util.logging.Level;

import static com.newrelic.agent.tracing.DistributedTraceUtil.ACCOUNT_ID;
import static com.newrelic.agent.tracing.DistributedTraceUtil.APPLICATION_ID;
import static com.newrelic.agent.tracing.DistributedTraceUtil.APP_PARENT_TYPE;
import static com.newrelic.agent.tracing.DistributedTraceUtil.DATA;
import static com.newrelic.agent.tracing.DistributedTraceUtil.GUID;
import static com.newrelic.agent.tracing.DistributedTraceUtil.PARENT_TYPE;
import static com.newrelic.agent.tracing.DistributedTraceUtil.PRIORITY;
import static com.newrelic.agent.tracing.DistributedTraceUtil.SAMPLED;
import static com.newrelic.agent.tracing.DistributedTraceUtil.TIMESTAMP;
import static com.newrelic.agent.tracing.DistributedTraceUtil.TRACE_ID;
import static com.newrelic.agent.tracing.DistributedTraceUtil.TRUSTED_ACCOUNT_KEY;
import static com.newrelic.agent.tracing.DistributedTraceUtil.TX;
import static com.newrelic.agent.tracing.DistributedTraceUtil.VERSION;

public class DistributedTracePayloadImpl implements DistributedTracePayload {

    public final long timestamp;
    public final String parentType;
    public final String accountId;
    public final String trustKey;
    public final String applicationId;
    public final String guid;
    public final String traceId;
    public final Float priority;
    public final Boolean sampled;
    public final String txnId;

    public static DistributedTracePayloadImpl createDistributedTracePayload(DistributedTracePayloadImpl inboundPayload, String traceId, String guid,
            String txnId, float priority) {
        DistributedTraceService distributedTraceService = ServiceFactory.getDistributedTraceService();
        String accountId = distributedTraceService.getAccountId();
        if (accountId == null) {
            AgentBridge.getAgent().getLogger().log(Level.FINER, "Not creating distributed trace payload due to null accountId.");
            return null;
        }

        String trustKey = distributedTraceService.getTrustKey();
        String applicationId = distributedTraceService.getApplicationId();
        long timestamp = System.currentTimeMillis();
        String parentType = APP_PARENT_TYPE;

        boolean sampled = DistributedTraceUtil.isSampledPriority(priority);

        return new DistributedTracePayloadImpl(timestamp, parentType, accountId, trustKey, applicationId, guid, traceId, txnId,
                priority, sampled);
    }

    private DistributedTracePayloadImpl(long timestamp, String parentType, String accountId, String trustKey, String applicationId,
            String guid, String traceId, String txnId, Float priority, Boolean sampled) {
        this.timestamp = timestamp;
        this.parentType = parentType;
        this.accountId = accountId;
        this.trustKey = trustKey;
        this.applicationId = applicationId;
        this.guid = guid;
        this.txnId = txnId;
        this.traceId = traceId;
        this.priority = priority;
        this.sampled = sampled;
    }

    @Override
    @SuppressWarnings("unchecked")
    public String text() {
        DistributedTraceService distributedTraceService = ServiceFactory.getDistributedTraceService();

        JSONObject payload = new JSONObject();

        JSONArray catVersion = new JSONArray();
        catVersion.add(distributedTraceService.getMajorSupportedCatVersion());
        catVersion.add(distributedTraceService.getMinorSupportedCatVersion());
        payload.put(VERSION, catVersion); // version [major, minor]

        JSONObject data = new JSONObject();

        data.put(TIMESTAMP, timestamp);
        data.put(PARENT_TYPE, parentType);
        data.put(ACCOUNT_ID, accountId);

        if (!accountId.equals(trustKey)) {
            data.put(TRUSTED_ACCOUNT_KEY, trustKey);
        }

        data.put(APPLICATION_ID, applicationId);

        if (guid != null) {
            // span events is enabled
            data.put(GUID, guid);
        }

        data.put(TRACE_ID, traceId);
        data.put(PRIORITY, priority);
        data.put(SAMPLED, sampled);

        if (txnId != null) {
            data.put(TX, txnId);
        }

        payload.put(DATA, data);
        return payload.toJSONString();
    }

    @Override
    public String httpSafe() {
        return Base64.encode(text().getBytes(Charsets.UTF_8));
    }

    public static DistributedTracePayloadImpl parseDistributedTracePayload(DistributedTracePayloadImpl outboundPayloadData, String payload) {
        if (payload == null) {
            AgentBridge.getAgent().getLogger().log(Level.FINER, "Incoming distributed trace payload is null.");
            NewRelic.incrementCounter(MetricNames.SUPPORTABILITY_ACCEPT_PAYLOAD_IGNORED_NULL);
            return null;
        }

        // record supportability error if someone called createDistributedTracePayload already
        if (outboundPayloadData != null) {
            Agent.LOG.log(Level.WARNING, "Error: createDistributedTracePayload was called before acceptDistributedTracePayload. Ignoring Call");
            NewRelic.incrementCounter(MetricNames.SUPPORTABILITY_ACCEPT_PAYLOAD_CREATE_BEFORE_ACCEPT);
            return null;
        }

        if (!payload.trim().isEmpty()) {
            payload = payload.trim();
            char firstChar = payload.charAt(0);
            if (firstChar != '{' && firstChar != '[') {
                // This must be base64 encoded, decode it
                payload = new String(Base64.decode(payload), Charsets.UTF_8);
            }
        }

        DistributedTraceService distributedTraceService = ServiceFactory.getDistributedTraceService();
        JSONParser parser = new JSONParser();
        try {
            JSONObject object = (JSONObject) parser.parse(payload);

            // ignore payload if major version is higher than our own
            JSONArray version = (JSONArray) object.get(VERSION);
            final Long majorVersion = (Long) version.get(0);
            int majorSupportedVersion = distributedTraceService.getMajorSupportedCatVersion();
            if (majorVersion > majorSupportedVersion) {
                AgentBridge.getAgent().getLogger().log(Level.FINER,
                        "Incoming distributed trace payload major version: {0} is newer than supported agent"
                                + " version: {1}. Ignoring payload.", majorVersion, majorSupportedVersion);
                NewRelic.incrementCounter(MetricNames.SUPPORTABILITY_ACCEPT_PAYLOAD_IGNORED_MAJOR_VERSION);
                return null;
            }

            JSONObject data = (JSONObject) object.get(DATA);

            // ignore payload if accountId isn't trusted
            String payloadAccountId = (String) data.get(ACCOUNT_ID);

            // ignore payload if isn't trusted
            String payloadTrustKey = (String) data.get(TRUSTED_ACCOUNT_KEY);
            String trustKey = distributedTraceService.getTrustKey();

            if (payloadAccountId == null) {
                AgentBridge.getAgent().getLogger().log(Level.FINER, "Invalid payload {0}. Payload missing accountId.", data);
                NewRelic.incrementCounter(MetricNames.SUPPORTABILITY_ACCEPT_PAYLOAD_IGNORED_PARSE_EXCEPTION);
                return null;
            }

            String applicationId = (String) data.get(APPLICATION_ID);
            if (applicationId == null) {
                AgentBridge.getAgent().getLogger().log(Level.FINER, "Incoming distributed trace payload is missing application id");
                NewRelic.incrementCounter(MetricNames.SUPPORTABILITY_ACCEPT_PAYLOAD_IGNORED_PARSE_EXCEPTION);
                return null;
            }

            // If payload doesn't have a tk, use accountId
            boolean isTrustedAccountKey = trustKey.equals(payloadTrustKey == null ? payloadAccountId : payloadTrustKey);
            if (!isTrustedAccountKey) {
                AgentBridge.getAgent().getLogger().log(Level.FINER,
                        "Incoming distributed trace payload trustKey: {0} does not match trusted account key: {1}."
                                + " Ignoring payload.", payloadTrustKey, trustKey);
                NewRelic.incrementCounter(MetricNames.SUPPORTABILITY_ACCEPT_PAYLOAD_IGNORED_UNTRUSTED_ACCOUNT);
                return null;
            }

            long timestamp = (Long) data.get(TIMESTAMP);
            if (timestamp <= 0) {
                AgentBridge.getAgent().getLogger().log(Level.FINER, "Invalid payload {0}. Payload missing keys.", data);
                NewRelic.incrementCounter(MetricNames.SUPPORTABILITY_ACCEPT_PAYLOAD_IGNORED_PARSE_EXCEPTION);
                return null;
            }

            String parentType = (String) data.get(PARENT_TYPE);
            if (parentType == null) {
                AgentBridge.getAgent().getLogger().log(Level.FINER, "Incoming distributed trace payload is missing type");
                NewRelic.incrementCounter(MetricNames.SUPPORTABILITY_ACCEPT_PAYLOAD_IGNORED_PARSE_EXCEPTION);
                return null;
            }

            String traceId = (String) data.get(TRACE_ID);
            if (traceId == null) {
                AgentBridge.getAgent().getLogger().log(Level.FINER, "Incoming distributed trace payload is missing traceId");
                NewRelic.incrementCounter(MetricNames.SUPPORTABILITY_ACCEPT_PAYLOAD_IGNORED_PARSE_EXCEPTION);
                return null;
            }

            String guid = (String) data.get(GUID);
            String txnId = (String) data.get(TX);
            if (guid == null && txnId == null) {
                // caller has span events disabled and there's no transaction?
                // they must be using txn-less api, but no spans?
                AgentBridge.getAgent().getLogger().log(Level.FINER, "Incoming distributed trace payload is missing traceId");
                NewRelic.incrementCounter(MetricNames.SUPPORTABILITY_ACCEPT_PAYLOAD_IGNORED_PARSE_EXCEPTION);
                return null;
            }

            Number priorityNumber = (Number) data.get(PRIORITY);
            Float priority = priorityNumber != null ? priorityNumber.floatValue() : null;
            Boolean sampled = (Boolean) data.get(SAMPLED);

            DistributedTracePayloadImpl distributedTracePayload = new DistributedTracePayloadImpl(timestamp,
                    parentType, payloadAccountId, payloadTrustKey, applicationId, guid, traceId, txnId, priority, sampled);

            if (Agent.LOG.isFinestEnabled()) {
                Agent.LOG.log(Level.FINEST, "Parsed distributed trace payload: {0}", distributedTracePayload);
            }

            return distributedTracePayload;
        } catch (Exception e) {
            Agent.LOG.log(Level.FINEST, e, "Failed to parse distributed trace payload");
            NewRelic.incrementCounter(MetricNames.SUPPORTABILITY_ACCEPT_PAYLOAD_IGNORED_PARSE_EXCEPTION);
            return null;
        }
    }

    @Override
    public String toString() {
        return "DistributedTracePayloadImpl{" +
                "timestamp=" + timestamp +
                ", parentType='" + parentType + '\'' +
                ", accountId='" + accountId + '\'' +
                ", trustKey='" + trustKey + '\'' +
                ", applicationId='" + applicationId + '\'' +
                ", guid='" + guid + '\'' +
                ", traceId='" + traceId + '\'' +
                ", txnId='" + txnId + '\'' +
                ", sampled=" + sampled +
                ", priority=" + priority +
                '}';
    }

}
