/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.zuul.netty.connectionpool;

import com.netflix.client.config.IClientConfig;
import com.netflix.spectator.api.Counter;
import com.netflix.spectator.api.Timer;
import com.netflix.zuul.discovery.DiscoveryResult;
import com.netflix.zuul.exception.OutboundErrorType;
import com.netflix.zuul.netty.connectionpool.ConnectionPoolConfig;
import com.netflix.zuul.netty.connectionpool.DefaultClientChannelManager;
import com.netflix.zuul.netty.connectionpool.IConnectionPool;
import com.netflix.zuul.netty.connectionpool.NettyClientConnectionFactory;
import com.netflix.zuul.netty.connectionpool.OriginConnectException;
import com.netflix.zuul.netty.connectionpool.PooledConnection;
import com.netflix.zuul.netty.connectionpool.PooledConnectionFactory;
import com.netflix.zuul.passport.CurrentPassport;
import com.netflix.zuul.passport.PassportState;
import io.netty.channel.ChannelFuture;
import io.netty.channel.EventLoop;
import io.netty.util.concurrent.Promise;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.util.Deque;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PerServerConnectionPool
implements IConnectionPool {
    private static final Logger LOG = LoggerFactory.getLogger(PerServerConnectionPool.class);
    private final ConcurrentHashMap<EventLoop, Deque<PooledConnection>> connectionsPerEventLoop = new ConcurrentHashMap();
    private final DiscoveryResult server;
    private final SocketAddress serverAddr;
    private final NettyClientConnectionFactory connectionFactory;
    private final PooledConnectionFactory pooledConnectionFactory;
    private final ConnectionPoolConfig config;
    private final IClientConfig niwsClientConfig;
    private final Counter createNewConnCounter;
    private final Counter createConnSucceededCounter;
    private final Counter createConnFailedCounter;
    private final Counter requestConnCounter;
    private final Counter reuseConnCounter;
    private final Counter connTakenFromPoolIsNotOpen;
    private final Counter maxConnsPerHostExceededCounter;
    private final Timer connEstablishTimer;
    private final AtomicInteger connsInPool;
    private final AtomicInteger connsInUse;
    private final AtomicInteger connCreationsInProgress;

    public PerServerConnectionPool(DiscoveryResult server, SocketAddress serverAddr, NettyClientConnectionFactory connectionFactory, PooledConnectionFactory pooledConnectionFactory, ConnectionPoolConfig config, IClientConfig niwsClientConfig, Counter createNewConnCounter, Counter createConnSucceededCounter, Counter createConnFailedCounter, Counter requestConnCounter, Counter reuseConnCounter, Counter connTakenFromPoolIsNotOpen, Counter maxConnsPerHostExceededCounter, Timer connEstablishTimer, AtomicInteger connsInPool, AtomicInteger connsInUse) {
        this.server = server;
        this.serverAddr = Objects.requireNonNull(serverAddr, "serverAddr");
        this.connectionFactory = connectionFactory;
        this.pooledConnectionFactory = pooledConnectionFactory;
        this.config = config;
        this.niwsClientConfig = niwsClientConfig;
        this.createNewConnCounter = createNewConnCounter;
        this.createConnSucceededCounter = createConnSucceededCounter;
        this.createConnFailedCounter = createConnFailedCounter;
        this.requestConnCounter = requestConnCounter;
        this.reuseConnCounter = reuseConnCounter;
        this.connTakenFromPoolIsNotOpen = connTakenFromPoolIsNotOpen;
        this.maxConnsPerHostExceededCounter = maxConnsPerHostExceededCounter;
        this.connEstablishTimer = connEstablishTimer;
        this.connsInPool = connsInPool;
        this.connsInUse = connsInUse;
        this.connCreationsInProgress = new AtomicInteger(0);
    }

    @Override
    public ConnectionPoolConfig getConfig() {
        return this.config;
    }

    public IClientConfig getNiwsClientConfig() {
        return this.niwsClientConfig;
    }

    @Override
    public boolean isAvailable() {
        return true;
    }

    private void onAcquire(PooledConnection conn, CurrentPassport passport) {
        passport.setOnChannel(conn.getChannel());
        this.removeIdleStateHandler(conn);
        conn.setInUse();
        if (LOG.isDebugEnabled()) {
            LOG.debug("PooledConnection acquired: " + conn.toString());
        }
    }

    protected void removeIdleStateHandler(PooledConnection conn) {
        DefaultClientChannelManager.removeHandlerFromPipeline("idleStateHandler", conn.getChannel().pipeline());
    }

    @Override
    public Promise<PooledConnection> acquire(EventLoop eventLoop, CurrentPassport passport, AtomicReference<? super InetAddress> selectedHostAddr) {
        this.requestConnCounter.increment();
        this.server.incrementActiveRequestsCount();
        Promise promise = eventLoop.newPromise();
        PooledConnection conn = this.tryGettingFromConnectionPool(eventLoop);
        if (conn != null) {
            conn.startRequestTimer();
            conn.incrementUsageCount();
            conn.getChannel().read();
            this.onAcquire(conn, passport);
            this.initPooledConnection(conn, (Promise<PooledConnection>)promise);
            selectedHostAddr.set(PerServerConnectionPool.getSelectedHostString(this.serverAddr));
        } else {
            this.tryMakingNewConnection(eventLoop, (Promise<PooledConnection>)promise, passport, selectedHostAddr);
        }
        return promise;
    }

    public PooledConnection tryGettingFromConnectionPool(EventLoop eventLoop) {
        PooledConnection conn;
        Deque<PooledConnection> connections = this.getPoolForEventLoop(eventLoop);
        while ((conn = connections.poll()) != null) {
            conn.setInPool(false);
            if (this.isValidFromPool(conn)) {
                this.reuseConnCounter.increment();
                this.connsInUse.incrementAndGet();
                this.connsInPool.decrementAndGet();
                return conn;
            }
            this.connTakenFromPoolIsNotOpen.increment();
            this.connsInPool.decrementAndGet();
            conn.close();
        }
        return null;
    }

    protected boolean isValidFromPool(PooledConnection conn) {
        return conn.isActive() && conn.getChannel().isOpen();
    }

    protected void initPooledConnection(PooledConnection conn, Promise<PooledConnection> promise) {
        promise.setSuccess((Object)conn);
    }

    protected Deque<PooledConnection> getPoolForEventLoop(EventLoop eventLoop) {
        Deque<PooledConnection> pool = this.connectionsPerEventLoop.get(eventLoop);
        if (pool == null) {
            pool = new ConcurrentLinkedDeque<PooledConnection>();
            this.connectionsPerEventLoop.putIfAbsent(eventLoop, pool);
        }
        return pool;
    }

    protected void tryMakingNewConnection(EventLoop eventLoop, Promise<PooledConnection> promise, CurrentPassport passport, AtomicReference<? super InetAddress> selectedHostAddr) {
        int maxConnectionsPerHost = this.config.maxConnectionsPerHost();
        int openAndOpeningConnectionCount = this.server.getOpenConnectionsCount() + this.connCreationsInProgress.get();
        if (maxConnectionsPerHost != -1 && openAndOpeningConnectionCount >= maxConnectionsPerHost) {
            this.maxConnsPerHostExceededCounter.increment();
            promise.setFailure((Throwable)new OriginConnectException("maxConnectionsPerHost=" + maxConnectionsPerHost + ", connectionsPerHost=" + openAndOpeningConnectionCount, OutboundErrorType.ORIGIN_SERVER_MAX_CONNS));
            LOG.warn("Unable to create new connection because at MaxConnectionsPerHost! maxConnectionsPerHost=" + maxConnectionsPerHost + ", connectionsPerHost=" + openAndOpeningConnectionCount + ", host=" + this.server.getServerId() + "origin=" + this.config.getOriginName());
            return;
        }
        try {
            this.createNewConnCounter.increment();
            this.connCreationsInProgress.incrementAndGet();
            passport.add(PassportState.ORIGIN_CH_CONNECTING);
            selectedHostAddr.set(PerServerConnectionPool.getSelectedHostString(this.serverAddr));
            ChannelFuture cf = this.connectToServer(eventLoop, passport, this.serverAddr);
            if (cf.isDone()) {
                this.handleConnectCompletion(cf, promise, passport);
            } else {
                cf.addListener(future -> {
                    try {
                        this.handleConnectCompletion((ChannelFuture)future, promise, passport);
                    }
                    catch (Throwable e) {
                        if (!promise.isDone()) {
                            promise.setFailure(e);
                        }
                        LOG.warn("Error creating new connection! origin=" + this.config.getOriginName() + ", host=" + this.server.getServerId());
                    }
                });
            }
        }
        catch (Throwable e) {
            promise.setFailure(e);
        }
    }

    protected ChannelFuture connectToServer(EventLoop eventLoop, CurrentPassport passport, SocketAddress serverAddr) {
        return this.connectionFactory.connect(eventLoop, serverAddr, passport);
    }

    protected void handleConnectCompletion(ChannelFuture cf, Promise<PooledConnection> callerPromise, CurrentPassport passport) {
        this.connCreationsInProgress.decrementAndGet();
        if (cf.isSuccess()) {
            passport.add(PassportState.ORIGIN_CH_CONNECTED);
            this.server.incrementOpenConnectionsCount();
            this.createConnSucceededCounter.increment();
            this.connsInUse.incrementAndGet();
            this.createConnection(cf, callerPromise, passport);
        } else {
            this.server.incrementSuccessiveConnectionFailureCount();
            this.server.addToFailureCount();
            this.server.decrementActiveRequestsCount();
            this.createConnFailedCounter.increment();
            callerPromise.setFailure((Throwable)new OriginConnectException(cf.cause().getMessage(), OutboundErrorType.CONNECT_ERROR));
        }
    }

    protected void createConnection(ChannelFuture cf, Promise<PooledConnection> callerPromise, CurrentPassport passport) {
        PooledConnection conn = this.pooledConnectionFactory.create(cf.channel());
        conn.incrementUsageCount();
        conn.startRequestTimer();
        conn.getChannel().read();
        this.onAcquire(conn, passport);
        callerPromise.setSuccess((Object)conn);
    }

    @Override
    public boolean release(PooledConnection conn) {
        if (conn == null) {
            return false;
        }
        if (conn.isInPool()) {
            return false;
        }
        EventLoop eventLoop = conn.getChannel().eventLoop();
        Deque<PooledConnection> connections = this.getPoolForEventLoop(eventLoop);
        CurrentPassport passport = CurrentPassport.fromChannel(conn.getChannel());
        int poolWaterline = this.config.perServerWaterline();
        if (poolWaterline > -1 && connections.size() >= poolWaterline) {
            conn.close();
            conn.setInPool(false);
            return false;
        }
        if (connections.offer(conn)) {
            conn.setInPool(true);
            this.connsInPool.incrementAndGet();
            passport.add(PassportState.ORIGIN_CH_POOL_RETURNED);
            return true;
        }
        conn.close();
        conn.setInPool(false);
        return false;
    }

    @Override
    public boolean remove(PooledConnection conn) {
        if (conn == null) {
            return false;
        }
        if (!conn.isInPool()) {
            return false;
        }
        EventLoop eventLoop = conn.getChannel().eventLoop();
        Deque<PooledConnection> connections = this.getPoolForEventLoop(eventLoop);
        if (connections.remove(conn)) {
            conn.setInPool(false);
            this.connsInPool.decrementAndGet();
            return true;
        }
        return false;
    }

    @Override
    public void shutdown() {
        for (Deque<PooledConnection> connections : this.connectionsPerEventLoop.values()) {
            for (PooledConnection conn : connections) {
                conn.close();
            }
        }
    }

    @Override
    public int getConnsInPool() {
        return this.connsInPool.get();
    }

    @Override
    public int getConnsInUse() {
        return this.connsInUse.get();
    }

    @Nullable
    private static InetAddress getSelectedHostString(SocketAddress addr) {
        if (addr instanceof InetSocketAddress) {
            return ((InetSocketAddress)addr).getAddress();
        }
        return null;
    }
}

