/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.evcache;

import com.google.common.base.Optional;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.CacheStats;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.netflix.archaius.api.Property;
import com.netflix.evcache.EVCacheException;
import com.netflix.evcache.EVCacheImpl;
import com.netflix.evcache.EVCacheKey;
import com.netflix.evcache.metrics.EVCacheMetricsFactory;
import com.netflix.evcache.util.EVCacheConfig;
import com.netflix.spectator.api.BasicTag;
import com.netflix.spectator.api.Counter;
import com.netflix.spectator.api.Gauge;
import com.netflix.spectator.api.Id;
import com.netflix.spectator.api.Tag;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import net.spy.memcached.transcoders.Transcoder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EVCacheInMemoryCache<T> {
    private static final Logger log = LoggerFactory.getLogger(EVCacheInMemoryCache.class);
    private final Property<Integer> _cacheDuration;
    private final Property<Integer> _refreshDuration;
    private final Property<Integer> _exireAfterAccessDuration;
    private final Property<Integer> _cacheSize;
    private final Property<Integer> _poolSize;
    private final String appName;
    private final Map<String, Counter> counterMap = new ConcurrentHashMap<String, Counter>();
    private final Map<String, Gauge> gaugeMap = new ConcurrentHashMap<String, Gauge>();
    private LoadingCache<EVCacheKey, Optional<T>> cache;
    private ExecutorService pool = null;
    private final Transcoder<T> tc;
    private final EVCacheImpl impl;
    private final Id sizeId;
    private ReentrantReadWriteLock.WriteLock writeLock = new ReentrantReadWriteLock().writeLock();
    private CacheStats previousStats = null;

    public EVCacheInMemoryCache(String appName, Transcoder<T> tc, EVCacheImpl impl) {
        this.appName = appName;
        this.tc = tc;
        this.impl = impl;
        this._cacheDuration = EVCacheConfig.getInstance().getPropertyRepository().get(appName + ".inmemory.expire.after.write.duration.ms", Integer.class).orElseGet(appName + ".inmemory.cache.duration.ms").orElse((Object)0);
        this._cacheDuration.subscribe(i -> this.setupCache());
        this._exireAfterAccessDuration = EVCacheConfig.getInstance().getPropertyRepository().get(appName + ".inmemory.expire.after.access.duration.ms", Integer.class).orElse((Object)0);
        this._exireAfterAccessDuration.subscribe(i -> this.setupCache());
        this._refreshDuration = EVCacheConfig.getInstance().getPropertyRepository().get(appName + ".inmemory.refresh.after.write.duration.ms", Integer.class).orElse((Object)0);
        this._refreshDuration.subscribe(i -> this.setupCache());
        this._cacheSize = EVCacheConfig.getInstance().getPropertyRepository().get(appName + ".inmemory.cache.size", Integer.class).orElse((Object)100);
        this._cacheSize.subscribe(i -> this.setupCache());
        this._poolSize = EVCacheConfig.getInstance().getPropertyRepository().get(appName + ".thread.pool.size", Integer.class).orElse((Object)5);
        this._poolSize.subscribe(i -> this.initRefreshPool());
        ArrayList<Tag> tags = new ArrayList<Tag>(3);
        tags.addAll(impl.getTags());
        tags.add((Tag)new BasicTag("evc.metric", "size"));
        this.sizeId = EVCacheMetricsFactory.getInstance().getId("internal.evc.client.inmemorycache", tags);
        this.setupCache();
        this.setupMonitoring(appName);
    }

    private void initRefreshPool() {
        ExecutorService oldPool = this.pool;
        this.writeLock.lock();
        try {
            ThreadFactory factory = new ThreadFactoryBuilder().setDaemon(true).setNameFormat("EVCacheInMemoryCache-%d").build();
            this.pool = Executors.newFixedThreadPool((Integer)this._poolSize.get(), factory);
            if (oldPool != null) {
                oldPool.shutdown();
            }
        }
        finally {
            this.writeLock.unlock();
        }
    }

    private void setupCache() {
        try {
            CacheBuilder builder = CacheBuilder.newBuilder().recordStats();
            if ((Integer)this._cacheSize.get() > 0) {
                builder = builder.maximumSize((long)((Integer)this._cacheSize.get()).intValue());
            }
            if ((Integer)this._exireAfterAccessDuration.get() > 0) {
                builder = builder.expireAfterAccess((long)((Integer)this._exireAfterAccessDuration.get()).intValue(), TimeUnit.MILLISECONDS);
            } else if ((Integer)this._cacheDuration.get() > 0) {
                builder = builder.expireAfterWrite((long)((Integer)this._cacheDuration.get()).intValue(), TimeUnit.MILLISECONDS);
            }
            if ((Integer)this._refreshDuration.get() > 0) {
                builder = builder.refreshAfterWrite((long)((Integer)this._refreshDuration.get()).intValue(), TimeUnit.MILLISECONDS);
            }
            this.initRefreshPool();
            LoadingCache newCache = builder.build(new CacheLoader<EVCacheKey, Optional<T>>(){

                public Optional<T> load(EVCacheKey key) throws EVCacheException, DataNotFoundException {
                    try {
                        return Optional.fromNullable(EVCacheInMemoryCache.this.impl.doGet(key, EVCacheInMemoryCache.this.tc));
                    }
                    catch (EVCacheException e) {
                        log.error("EVCacheException while loading key -> " + key, (Throwable)e);
                        throw e;
                    }
                    catch (Exception e) {
                        log.error("EVCacheException while loading key -> " + key, (Throwable)e);
                        throw new EVCacheException("key : " + key + " could not be loaded", e);
                    }
                }

                public ListenableFuture<Optional<T>> reload(final EVCacheKey key, final Optional<T> oldValue) {
                    ListenableFutureTask task = ListenableFutureTask.create((Callable)new Callable<Optional<T>>(){

                        @Override
                        public Optional<T> call() {
                            try {
                                Optional t = this.load(key);
                                if (t == null) {
                                    EVCacheMetricsFactory.getInstance().increment("EVCacheInMemoryCache-" + EVCacheInMemoryCache.this.appName + "-Reload-NotFound");
                                    return oldValue;
                                }
                                EVCacheMetricsFactory.getInstance().increment("EVCacheInMemoryCache-" + EVCacheInMemoryCache.this.appName + "-Reload-Success");
                                return t;
                            }
                            catch (EVCacheException e) {
                                log.error("EVCacheException while reloading key -> " + key, (Throwable)e);
                                EVCacheMetricsFactory.getInstance().increment("EVCacheInMemoryCache-" + EVCacheInMemoryCache.this.appName + "-Reload-Fail");
                                return oldValue;
                            }
                        }
                    });
                    EVCacheInMemoryCache.this.pool.execute((Runnable)task);
                    return task;
                }
            });
            if (this.cache != null) {
                newCache.putAll((Map)this.cache.asMap());
            }
            LoadingCache<EVCacheKey, Optional<T>> currentCache = this.cache;
            this.cache = newCache;
            if (currentCache != null) {
                currentCache.invalidateAll();
                currentCache.cleanUp();
            }
        }
        catch (Exception e) {
            log.error(e.getMessage(), (Throwable)e);
        }
    }

    private long getSize() {
        long size = this.cache.size();
        CacheStats stats = this.cache.stats();
        if (this.previousStats != null) {
            try {
                this.getCounter("hits").increment(stats.hitCount() - this.previousStats.hitCount());
                this.getCounter("miss").increment(stats.missCount() - this.previousStats.missCount());
                this.getCounter("evictions").increment(stats.evictionCount() - this.previousStats.evictionCount());
                this.getCounter("requests").increment(stats.requestCount() - this.previousStats.requestCount());
                this.getCounter("loadExceptionCount").increment(stats.loadExceptionCount() - this.previousStats.loadExceptionCount());
                this.getCounter("loadCount").increment(stats.loadCount() - this.previousStats.loadCount());
                this.getCounter("loadSuccessCount").increment(stats.loadSuccessCount() - this.previousStats.loadSuccessCount());
                this.getCounter("totalLoadTime-ms").increment((stats.totalLoadTime() - this.previousStats.totalLoadTime()) / 1000000L);
                this.getGauge("hitrate").set(stats.hitRate());
                this.getGauge("loadExceptionRate").set(stats.loadExceptionRate());
                this.getGauge("averageLoadTime-ms").set(stats.averageLoadPenalty() / 1000000.0);
            }
            catch (Exception e) {
                log.error("Error while reporting stats", (Throwable)e);
            }
        }
        this.previousStats = stats;
        return size;
    }

    private void setupMonitoring(String appName) {
        EVCacheMetricsFactory.getInstance().getRegistry().gauge(this.sizeId, (Object)this, EVCacheInMemoryCache::getSize);
    }

    private Counter getCounter(String name) {
        Counter counter = this.counterMap.get(name);
        if (counter != null) {
            return counter;
        }
        ArrayList<Tag> tags = new ArrayList<Tag>(3);
        tags.addAll(this.impl.getTags());
        tags.add((Tag)new BasicTag("evc.metric", name));
        counter = EVCacheMetricsFactory.getInstance().getCounter("internal.evc.client.inmemorycache", tags);
        this.counterMap.put(name, counter);
        return counter;
    }

    private Gauge getGauge(String name) {
        Gauge gauge = this.gaugeMap.get(name);
        if (gauge != null) {
            return gauge;
        }
        ArrayList<Tag> tags = new ArrayList<Tag>(3);
        tags.addAll(this.impl.getTags());
        tags.add((Tag)new BasicTag("evc.metric", name));
        Id id = EVCacheMetricsFactory.getInstance().getId("internal.evc.client.inmemorycache", tags);
        gauge = EVCacheMetricsFactory.getInstance().getRegistry().gauge(id);
        this.gaugeMap.put(name, gauge);
        return gauge;
    }

    public T get(EVCacheKey key) throws ExecutionException {
        if (this.cache == null) {
            return null;
        }
        Optional val = (Optional)this.cache.get((Object)key);
        if (!val.isPresent()) {
            return null;
        }
        if (log.isDebugEnabled()) {
            log.debug("GET : appName : " + this.appName + "; Key : " + key + "; val : " + val);
        }
        return (T)val.get();
    }

    public void put(EVCacheKey key, T value) {
        if (this.cache == null) {
            return;
        }
        this.cache.put((Object)key, (Object)Optional.fromNullable(value));
        if (log.isDebugEnabled()) {
            log.debug("PUT : appName : " + this.appName + "; Key : " + key + "; val : " + value);
        }
    }

    public void delete(String key) {
        if (this.cache == null) {
            return;
        }
        this.cache.invalidate((Object)key);
        if (log.isDebugEnabled()) {
            log.debug("DEL : appName : " + this.appName + "; Key : " + key);
        }
    }

    public Map<EVCacheKey, Optional<T>> getAll() {
        if (this.cache == null) {
            return Collections.emptyMap();
        }
        return this.cache.asMap();
    }

    public static final class DataNotFoundException
    extends EVCacheException {
        private static final long serialVersionUID = 1800185311509130263L;

        public DataNotFoundException(String message) {
            super(message);
        }
    }
}

