/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.evcache;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.CacheStats;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.netflix.config.ChainedDynamicProperty;
import com.netflix.config.DynamicIntProperty;
import com.netflix.evcache.EVCacheException;
import com.netflix.evcache.EVCacheImpl;
import com.netflix.evcache.metrics.EVCacheMetricsFactory;
import com.netflix.evcache.util.EVCacheConfig;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import net.spy.memcached.transcoders.Transcoder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EVCacheInMemoryCache<T> {
    private static final Logger log = LoggerFactory.getLogger(EVCacheInMemoryCache.class);
    private final ChainedDynamicProperty.IntProperty _cacheDuration;
    private final DynamicIntProperty _refreshDuration;
    private final DynamicIntProperty _exireAfterAccessDuration;
    private final DynamicIntProperty _cacheSize;
    private final DynamicIntProperty _poolSize;
    private final String appName;
    private LoadingCache<String, T> cache;
    private ExecutorService pool = null;
    private final Transcoder<T> tc;
    private final EVCacheImpl impl;
    private ReentrantReadWriteLock.WriteLock writeLock = new ReentrantReadWriteLock().writeLock();

    public EVCacheInMemoryCache(String appName, Transcoder<T> tc, EVCacheImpl impl) {
        this.appName = appName;
        this.tc = tc;
        this.impl = impl;
        this._cacheDuration = EVCacheConfig.getInstance().getChainedIntProperty(appName + ".inmemory.cache.duration.ms", appName + ".inmemory.expire.after.write.duration.ms", 0);
        this._cacheDuration.addCallback(new Runnable(){

            @Override
            public void run() {
                EVCacheInMemoryCache.this.setupCache();
            }
        });
        this._exireAfterAccessDuration = EVCacheConfig.getInstance().getDynamicIntProperty(appName + ".inmemory.expire.after.access.duration.ms", 0);
        this._exireAfterAccessDuration.addCallback(new Runnable(){

            @Override
            public void run() {
                EVCacheInMemoryCache.this.setupCache();
            }
        });
        this._refreshDuration = EVCacheConfig.getInstance().getDynamicIntProperty(appName + ".inmemory.refresh.after.write.duration.ms", 0);
        this._refreshDuration.addCallback(new Runnable(){

            @Override
            public void run() {
                EVCacheInMemoryCache.this.setupCache();
            }
        });
        this._cacheSize = EVCacheConfig.getInstance().getDynamicIntProperty(appName + ".inmemory.cache.size", 100);
        this._cacheSize.addCallback(new Runnable(){

            @Override
            public void run() {
                EVCacheInMemoryCache.this.setupCache();
            }
        });
        this._poolSize = EVCacheConfig.getInstance().getDynamicIntProperty(appName + ".thread.pool.size", 5);
        this._poolSize.addCallback(new Runnable(){

            @Override
            public void run() {
                EVCacheInMemoryCache.this.initRefreshPool();
            }
        });
        this.setupCache();
        this.setupMonitoring(appName);
    }

    private void initRefreshPool() {
        ExecutorService oldPool = this.pool;
        this.writeLock.lock();
        try {
            ThreadFactory factory = new ThreadFactoryBuilder().setDaemon(true).setNameFormat("EVCacheInMemoryCache-%d").build();
            this.pool = Executors.newFixedThreadPool(this._poolSize.get(), factory);
            if (oldPool != null) {
                oldPool.shutdown();
            }
        }
        finally {
            this.writeLock.unlock();
        }
    }

    private void setupCache() {
        try {
            CacheBuilder builder = CacheBuilder.newBuilder().recordStats();
            if (this._cacheSize.get() > 0) {
                builder = builder.maximumSize((long)this._cacheSize.get());
            }
            if (this._exireAfterAccessDuration.get() > 0) {
                builder = builder.expireAfterAccess((long)this._exireAfterAccessDuration.get(), TimeUnit.MILLISECONDS);
            } else if ((Integer)this._cacheDuration.get() > 0) {
                builder = builder.expireAfterWrite((long)((Integer)this._cacheDuration.get()).intValue(), TimeUnit.MILLISECONDS);
            }
            if (this._refreshDuration.get() > 0) {
                builder = builder.refreshAfterWrite((long)this._refreshDuration.get(), TimeUnit.MILLISECONDS);
            }
            this.initRefreshPool();
            LoadingCache newCache = builder.build(new CacheLoader<String, T>(){

                public T load(String key) throws EVCacheException {
                    try {
                        return EVCacheInMemoryCache.this.impl.doGet(key, EVCacheInMemoryCache.this.tc);
                    }
                    catch (EVCacheException e) {
                        log.error("EVCacheException while loading key -> " + key, (Throwable)e);
                        throw e;
                    }
                }

                public ListenableFuture<T> reload(final String key, final T prev) {
                    ListenableFutureTask task = ListenableFutureTask.create((Callable)new Callable<T>(){

                        @Override
                        public T call() {
                            try {
                                Object t = this.load(key);
                                if (t == null) {
                                    EVCacheMetricsFactory.getInstance().increment("EVCacheInMemoryCache-" + EVCacheInMemoryCache.this.appName + "-Reload-NotFound");
                                } else {
                                    EVCacheMetricsFactory.getInstance().increment("EVCacheInMemoryCache-" + EVCacheInMemoryCache.this.appName + "-Reload-Success");
                                }
                                return t;
                            }
                            catch (EVCacheException e) {
                                log.error("EVCacheException while reloading key -> " + key, (Throwable)e);
                                EVCacheMetricsFactory.getInstance().increment("EVCacheInMemoryCache-" + EVCacheInMemoryCache.this.appName + "-Reload-Fail");
                                return prev;
                            }
                        }
                    });
                    EVCacheInMemoryCache.this.pool.execute((Runnable)task);
                    return task;
                }
            });
            if (this.cache != null) {
                newCache.putAll((Map)this.cache.asMap());
            }
            LoadingCache<String, T> currentCache = this.cache;
            this.cache = newCache;
            if (currentCache != null) {
                currentCache.invalidateAll();
                currentCache.cleanUp();
            }
        }
        catch (Exception e) {
            log.error(e.getMessage(), (Throwable)e);
        }
    }

    private long getSize() {
        long size = this.cache.size();
        CacheStats stats = this.cache.stats();
        EVCacheMetricsFactory.getInstance().getRegistry().counter("EVCacheInMemoryCache-" + this.appName + "-hits").increment(stats.hitCount());
        EVCacheMetricsFactory.getInstance().getRegistry().counter("EVCacheInMemoryCache-" + this.appName + "-miss").increment(stats.missCount());
        EVCacheMetricsFactory.getInstance().getRegistry().counter("EVCacheInMemoryCache-" + this.appName + "-evictions").increment(stats.evictionCount());
        EVCacheMetricsFactory.getInstance().getRegistry().counter("EVCacheInMemoryCache-" + this.appName + "-requests").increment(stats.requestCount());
        EVCacheMetricsFactory.getInstance().getRegistry().gauge("EVCacheInMemoryCache-" + this.appName + "-hitrate", (Number)stats.hitRate());
        EVCacheMetricsFactory.getInstance().getRegistry().counter("EVCacheInMemoryCache-" + this.appName + "-loadExceptionCount").increment(stats.loadExceptionCount());
        EVCacheMetricsFactory.getInstance().getRegistry().counter("EVCacheInMemoryCache-" + this.appName + "-loadCount").increment(stats.loadCount());
        EVCacheMetricsFactory.getInstance().getRegistry().counter("EVCacheInMemoryCache-" + this.appName + "-loadSuccessCount").increment(stats.loadSuccessCount());
        EVCacheMetricsFactory.getInstance().getRegistry().counter("EVCacheInMemoryCache-" + this.appName + "-totalLoadTime-ms").increment(this.cache.stats().totalLoadTime() / 1000000L);
        EVCacheMetricsFactory.getInstance().getRegistry().gauge("EVCacheInMemoryCache-" + this.appName + "-loadExceptionRate", (Number)stats.loadExceptionRate());
        EVCacheMetricsFactory.getInstance().getRegistry().gauge("EVCacheInMemoryCache-" + this.appName + "-averageLoadTime-ms", (Number)(stats.averageLoadPenalty() / 1000000.0));
        return size;
    }

    private void setupMonitoring(String appName) {
        EVCacheMetricsFactory.getInstance().getRegistry().gauge("EVCacheInMemoryCache-" + appName + "-size", (Object)this, EVCacheInMemoryCache::getSize);
    }

    public T get(String key) throws ExecutionException {
        if (this.cache == null) {
            return null;
        }
        Object val = this.cache.get((Object)key);
        if (log.isDebugEnabled()) {
            log.debug("GET : appName : " + this.appName + "; Key : " + key + "; val : " + val);
        }
        return (T)val;
    }

    public void put(String key, T value) {
        if (this.cache == null) {
            return;
        }
        this.cache.put((Object)key, value);
        if (log.isDebugEnabled()) {
            log.debug("PUT : appName : " + this.appName + "; Key : " + key + "; val : " + value);
        }
    }

    public void delete(String key) {
        if (this.cache == null) {
            return;
        }
        this.cache.invalidate((Object)key);
        if (log.isDebugEnabled()) {
            log.debug("DEL : appName : " + this.appName + "; Key : " + key);
        }
    }

    public Map<String, T> getAll() {
        if (this.cache == null) {
            return Collections.emptyMap();
        }
        return this.cache.asMap();
    }
}

