/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.evcache;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.CacheStats;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.netflix.config.ChainedDynamicProperty;
import com.netflix.config.DynamicIntProperty;
import com.netflix.evcache.EVCacheException;
import com.netflix.evcache.EVCacheImpl;
import com.netflix.evcache.metrics.EVCacheMetricsFactory;
import com.netflix.evcache.util.EVCacheConfig;
import com.netflix.servo.DefaultMonitorRegistry;
import com.netflix.servo.MonitorRegistry;
import com.netflix.servo.annotations.DataSourceType;
import com.netflix.servo.monitor.Monitor;
import com.netflix.servo.monitor.MonitorConfig;
import com.netflix.servo.tag.Tag;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import net.spy.memcached.transcoders.Transcoder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EVCacheInMemoryCache<T> {
    private static final Logger log = LoggerFactory.getLogger(EVCacheInMemoryCache.class);
    private final ChainedDynamicProperty.IntProperty _cacheDuration;
    private final DynamicIntProperty _refreshDuration;
    private final DynamicIntProperty _exireAfterAccessDuration;
    private final DynamicIntProperty _cacheSize;
    private final DynamicIntProperty _poolSize;
    private final String appName;
    private LoadingCache<String, T> cache;
    private ExecutorService pool = null;
    private final Transcoder<T> tc;
    private final EVCacheImpl impl;
    private ReentrantReadWriteLock.WriteLock writeLock = new ReentrantReadWriteLock().writeLock();

    public EVCacheInMemoryCache(String appName, Transcoder<T> tc, EVCacheImpl impl) {
        this.appName = appName;
        this.tc = tc;
        this.impl = impl;
        Runnable setupCache = new Runnable(){

            @Override
            public void run() {
                EVCacheInMemoryCache.this.setupCache();
            }
        };
        this._cacheDuration = EVCacheConfig.getInstance().getChainedIntProperty(appName + ".inmemory.cache.duration.ms", appName + ".inmemory.expire.after.write.duration.ms", 0, setupCache);
        this._exireAfterAccessDuration = EVCacheConfig.getInstance().getDynamicIntProperty(appName + ".inmemory.expire.after.access.duration.ms", 0);
        this._exireAfterAccessDuration.addCallback(setupCache);
        this._refreshDuration = EVCacheConfig.getInstance().getDynamicIntProperty(appName + ".inmemory.refresh.after.write.duration.ms", 0);
        this._refreshDuration.addCallback(setupCache);
        this._cacheSize = EVCacheConfig.getInstance().getDynamicIntProperty(appName + ".inmemory.cache.size", 100);
        this._cacheSize.addCallback(setupCache);
        this._poolSize = EVCacheConfig.getInstance().getDynamicIntProperty(appName + ".thread.pool.size", 5);
        this._poolSize.addCallback(new Runnable(){

            @Override
            public void run() {
                EVCacheInMemoryCache.this.initRefreshPool();
            }
        });
        this.setupCache();
        this.setupMonitoring(appName);
    }

    private void initRefreshPool() {
        ExecutorService oldPool = this.pool;
        this.writeLock.lock();
        try {
            ThreadFactory factory = new ThreadFactoryBuilder().setDaemon(true).setNameFormat("EVCacheInMemoryCache-%d").build();
            this.pool = Executors.newFixedThreadPool(this._poolSize.get(), factory);
            if (oldPool != null) {
                oldPool.shutdown();
            }
        }
        finally {
            this.writeLock.unlock();
        }
    }

    private void register(Monitor<?> monitor) {
        MonitorRegistry registry = DefaultMonitorRegistry.getInstance();
        if (registry.isRegistered(monitor)) {
            registry.unregister(monitor);
        }
        registry.register(monitor);
    }

    private MonitorConfig getMonitorConfig(String appName, String metric, Tag tag) {
        MonitorConfig.Builder builder = MonitorConfig.builder((String)("EVCacheInMemoryCache-" + appName + "-" + metric)).withTag(tag).withTag(EVCacheMetricsFactory.OWNER);
        return builder.build();
    }

    private void setupCache() {
        try {
            CacheBuilder builder = CacheBuilder.newBuilder().recordStats();
            if (this._cacheSize.get() > 0) {
                builder = builder.maximumSize((long)this._cacheSize.get());
            }
            if (this._exireAfterAccessDuration.get() > 0) {
                builder = builder.expireAfterAccess((long)this._exireAfterAccessDuration.get(), TimeUnit.MILLISECONDS);
            } else if ((Integer)this._cacheDuration.get() > 0) {
                builder = builder.expireAfterWrite((long)((Integer)this._cacheDuration.get()).intValue(), TimeUnit.MILLISECONDS);
            }
            if (this._refreshDuration.get() > 0) {
                builder = builder.refreshAfterWrite((long)this._refreshDuration.get(), TimeUnit.MILLISECONDS);
            }
            this.initRefreshPool();
            LoadingCache newCache = builder.build(new CacheLoader<String, T>(){

                public T load(String key) throws EVCacheException {
                    try {
                        Object t = EVCacheInMemoryCache.this.impl.doGet(key, EVCacheInMemoryCache.this.tc);
                        if (t == null) {
                            throw new DataNotFoundException("Data for key : " + key + " could not be loaded as it was not found in EVCache");
                        }
                        return t;
                    }
                    catch (DataNotFoundException e) {
                        throw e;
                    }
                    catch (EVCacheException e) {
                        log.error("EVCacheException while loading key -> {}", (Object)key, (Object)e);
                        throw e;
                    }
                    catch (Exception e) {
                        log.error("Exception while loading key -> {}", (Object)key, (Object)e);
                        throw new EVCacheException("key : " + key + " could not be loaded", e);
                    }
                }

                public ListenableFuture<T> reload(final String key, final T prev) {
                    ListenableFutureTask task = ListenableFutureTask.create((Callable)new Callable<T>(){

                        @Override
                        public T call() {
                            try {
                                Object t = this.load(key);
                                if (t == null) {
                                    EVCacheMetricsFactory.increment(EVCacheInMemoryCache.this.appName, null, null, "EVCacheInMemoryCache-" + EVCacheInMemoryCache.this.appName + "-Reload-NotFound");
                                    return prev;
                                }
                                EVCacheMetricsFactory.increment(EVCacheInMemoryCache.this.appName, null, null, "EVCacheInMemoryCache-" + EVCacheInMemoryCache.this.appName + "-Reload-Success");
                                return t;
                            }
                            catch (EVCacheException e) {
                                log.error("EVCacheException while reloading key -> {}", (Object)key, (Object)e);
                                EVCacheMetricsFactory.increment(EVCacheInMemoryCache.this.appName, null, null, "EVCacheInMemoryCache-" + EVCacheInMemoryCache.this.appName + "-Reload-Fail");
                                return prev;
                            }
                        }
                    });
                    EVCacheInMemoryCache.this.pool.execute((Runnable)task);
                    return task;
                }
            });
            if (this.cache != null) {
                newCache.putAll((Map)this.cache.asMap());
            }
            LoadingCache<String, T> currentCache = this.cache;
            this.cache = newCache;
            if (currentCache != null) {
                currentCache.invalidateAll();
                currentCache.cleanUp();
            }
        }
        catch (Exception e) {
            log.error(e.getMessage(), (Throwable)e);
        }
    }

    private LoadingCache<String, T> getCache() {
        return this.cache;
    }

    private CacheStats getStats() {
        return this.cache.stats();
    }

    private void setupMonitoring(final String appName) {
        this.register(new Monitor<Number>(){
            final MonitorConfig config;
            {
                this.config = EVCacheInMemoryCache.this.getMonitorConfig(appName, "size", (Tag)DataSourceType.COUNTER);
            }

            public Number getValue() {
                if (EVCacheInMemoryCache.this.getCache() == null) {
                    return 0L;
                }
                return EVCacheInMemoryCache.this.getCache().size();
            }

            public Number getValue(int pollerIndex) {
                return this.getValue();
            }

            public MonitorConfig getConfig() {
                return this.config;
            }
        });
        this.register(new Monitor<Number>(){
            final MonitorConfig config;
            {
                this.config = EVCacheInMemoryCache.this.getMonitorConfig(appName, "requests", (Tag)DataSourceType.COUNTER);
            }

            public Number getValue() {
                if (EVCacheInMemoryCache.this.getCache() == null) {
                    return 0L;
                }
                return EVCacheInMemoryCache.this.getStats().requestCount();
            }

            public Number getValue(int pollerIndex) {
                return this.getValue();
            }

            public MonitorConfig getConfig() {
                return this.config;
            }
        });
        this.register(new Monitor<Number>(){
            final MonitorConfig config;
            {
                this.config = EVCacheInMemoryCache.this.getMonitorConfig(appName, "hitrate", (Tag)DataSourceType.GAUGE);
            }

            public Number getValue() {
                if (EVCacheInMemoryCache.this.getCache() == null) {
                    return 0L;
                }
                return EVCacheInMemoryCache.this.getStats().hitRate() * 100.0;
            }

            public Number getValue(int pollerIndex) {
                return this.getValue();
            }

            public MonitorConfig getConfig() {
                return this.config;
            }
        });
        this.register(new Monitor<Number>(){
            final MonitorConfig config;
            {
                this.config = EVCacheInMemoryCache.this.getMonitorConfig(appName, "hits", (Tag)DataSourceType.COUNTER);
            }

            public Number getValue() {
                if (EVCacheInMemoryCache.this.getCache() == null) {
                    return 0L;
                }
                return (double)EVCacheInMemoryCache.this.getStats().hitCount();
            }

            public Number getValue(int pollerIndex) {
                return this.getValue();
            }

            public MonitorConfig getConfig() {
                return this.config;
            }
        });
        this.register(new Monitor<Number>(){
            final MonitorConfig config;
            {
                this.config = EVCacheInMemoryCache.this.getMonitorConfig(appName, "miss", (Tag)DataSourceType.COUNTER);
            }

            public Number getValue() {
                if (EVCacheInMemoryCache.this.getCache() == null) {
                    return 0L;
                }
                return (double)EVCacheInMemoryCache.this.getStats().missCount();
            }

            public Number getValue(int pollerIndex) {
                return this.getValue();
            }

            public MonitorConfig getConfig() {
                return this.config;
            }
        });
        this.register(new Monitor<Number>(){
            final MonitorConfig config;
            {
                this.config = EVCacheInMemoryCache.this.getMonitorConfig(appName, "evictions", (Tag)DataSourceType.COUNTER);
            }

            public Number getValue() {
                if (EVCacheInMemoryCache.this.getCache() == null) {
                    return 0L;
                }
                return (double)EVCacheInMemoryCache.this.getStats().evictionCount();
            }

            public Number getValue(int pollerIndex) {
                return this.getValue();
            }

            public MonitorConfig getConfig() {
                return this.config;
            }
        });
        this.register(new Monitor<Number>(){
            final MonitorConfig config;
            {
                this.config = EVCacheInMemoryCache.this.getMonitorConfig(appName, "loadExceptionCount", (Tag)DataSourceType.COUNTER);
            }

            public Number getValue() {
                if (EVCacheInMemoryCache.this.getCache() == null) {
                    return 0L;
                }
                return (double)EVCacheInMemoryCache.this.getStats().loadExceptionCount();
            }

            public Number getValue(int pollerIndex) {
                return this.getValue();
            }

            public MonitorConfig getConfig() {
                return this.config;
            }
        });
        this.register(new Monitor<Number>(){
            final MonitorConfig config;
            {
                this.config = EVCacheInMemoryCache.this.getMonitorConfig(appName, "loadCount", (Tag)DataSourceType.COUNTER);
            }

            public Number getValue() {
                if (EVCacheInMemoryCache.this.getCache() == null) {
                    return 0L;
                }
                return (double)EVCacheInMemoryCache.this.getStats().loadCount();
            }

            public Number getValue(int pollerIndex) {
                return this.getValue();
            }

            public MonitorConfig getConfig() {
                return this.config;
            }
        });
        this.register(new Monitor<Number>(){
            final MonitorConfig config;
            {
                this.config = EVCacheInMemoryCache.this.getMonitorConfig(appName, "loadSuccessCount", (Tag)DataSourceType.COUNTER);
            }

            public Number getValue() {
                if (EVCacheInMemoryCache.this.getCache() == null) {
                    return 0L;
                }
                return (double)EVCacheInMemoryCache.this.getStats().loadSuccessCount();
            }

            public Number getValue(int pollerIndex) {
                return this.getValue();
            }

            public MonitorConfig getConfig() {
                return this.config;
            }
        });
        this.register(new Monitor<Number>(){
            final MonitorConfig config;
            {
                this.config = EVCacheInMemoryCache.this.getMonitorConfig(appName, "totalLoadTime-ms", (Tag)DataSourceType.COUNTER);
            }

            public Number getValue() {
                if (EVCacheInMemoryCache.this.getCache() == null) {
                    return 0L;
                }
                return (double)(EVCacheInMemoryCache.this.getStats().totalLoadTime() / 1000000L);
            }

            public Number getValue(int pollerIndex) {
                return this.getValue();
            }

            public MonitorConfig getConfig() {
                return this.config;
            }
        });
        this.register(new Monitor<Number>(){
            final MonitorConfig config;
            {
                this.config = EVCacheInMemoryCache.this.getMonitorConfig(appName, "loadExceptionRate", (Tag)DataSourceType.GAUGE);
            }

            public Number getValue() {
                if (EVCacheInMemoryCache.this.getCache() == null) {
                    return 0L;
                }
                return EVCacheInMemoryCache.this.getStats().loadExceptionRate() * 100.0;
            }

            public Number getValue(int pollerIndex) {
                return this.getValue();
            }

            public MonitorConfig getConfig() {
                return this.config;
            }
        });
        this.register(new Monitor<Number>(){
            final MonitorConfig config;
            {
                this.config = EVCacheInMemoryCache.this.getMonitorConfig(appName, "averageLoadTime-ms", (Tag)DataSourceType.GAUGE);
            }

            public Number getValue() {
                if (EVCacheInMemoryCache.this.getCache() == null) {
                    return 0L;
                }
                return EVCacheInMemoryCache.this.getStats().averageLoadPenalty() / 1000000.0;
            }

            public Number getValue(int pollerIndex) {
                return this.getValue();
            }

            public MonitorConfig getConfig() {
                return this.config;
            }
        });
    }

    public T get(String key) throws ExecutionException {
        if (this.cache == null) {
            return null;
        }
        Object val = this.cache.get((Object)key);
        if (log.isDebugEnabled()) {
            log.debug("GET : appName : " + this.appName + "; Key : " + key + "; val : " + val);
        }
        return (T)val;
    }

    public void put(String key, T value) {
        if (this.cache == null) {
            return;
        }
        this.cache.put((Object)key, value);
        if (log.isDebugEnabled()) {
            log.debug("PUT : appName : " + this.appName + "; Key : " + key + "; val : " + value);
        }
    }

    public void delete(String key) {
        if (this.cache == null) {
            return;
        }
        this.cache.invalidate((Object)key);
        if (log.isDebugEnabled()) {
            log.debug("DEL : appName : " + this.appName + "; Key : " + key);
        }
    }

    public Map<String, T> getAll() {
        if (this.cache == null) {
            return Collections.emptyMap();
        }
        return this.cache.asMap();
    }

    public static final class DataNotFoundException
    extends EVCacheException {
        private static final long serialVersionUID = 1800185311509130263L;

        public DataNotFoundException(String message) {
            super(message);
        }
    }
}

