/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.concurrency.limits.strategy;

import com.netflix.concurrency.limits.MetricRegistry;
import com.netflix.concurrency.limits.Strategy;
import com.netflix.concurrency.limits.internal.EmptyMetricRegistry;
import com.netflix.concurrency.limits.internal.Preconditions;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;

public class LookupPartitionStrategy<T>
implements Strategy<T> {
    private static final String PARTITION_TAG_NAME = "partition";
    private final Map<String, Partition> partitions;
    private final Partition unknownPartition;
    private final Function<T, String> lookup;
    private int busy = 0;
    private int limit = 0;

    public static <T> Builder<T> newBuilder(Function<T, String> lookup) {
        return new Builder<T>(lookup);
    }

    private LookupPartitionStrategy(Builder<T> builder) {
        Preconditions.checkArgument(!builder.partitions.isEmpty(), "No partitions specified");
        Preconditions.checkArgument(builder.partitions.values().stream().map(Partition::getPercent).reduce(0.0, Double::sum) <= 1.0, "Sum of percentages must be <= 1.0");
        this.partitions = new HashMap<String, Partition>(builder.partitions);
        this.partitions.forEach((name, partition) -> partition.createMetrics(builder.registry));
        this.unknownPartition = new Partition("unknown", 0.0);
        this.unknownPartition.createMetrics(builder.registry);
        this.lookup = builder.lookup;
        builder.registry.registerGauge("limit", this::getLimit, new String[0]);
    }

    @Override
    public synchronized Strategy.Token tryAcquire(T type) {
        Partition partition = this.partitions.getOrDefault(this.lookup.apply(type), this.unknownPartition);
        if (this.busy >= this.limit && partition.isLimitExceeded()) {
            return Strategy.Token.newNotAcquired(this.busy);
        }
        ++this.busy;
        partition.acquire();
        return Strategy.Token.newAcquired(this.busy, () -> this.releasePartition(partition));
    }

    private synchronized void releasePartition(Partition partition) {
        --this.busy;
        partition.release();
    }

    @Override
    public synchronized void setLimit(int newLimit) {
        if (this.limit != newLimit) {
            this.limit = newLimit;
            this.partitions.forEach((name, partition) -> partition.updateLimit(newLimit));
        }
    }

    public synchronized int getLimit() {
        return this.limit;
    }

    synchronized int getBinBusyCount(String key) {
        return Optional.ofNullable(this.partitions.get(key)).orElseThrow(() -> new IllegalArgumentException("Invalid group " + key)).busy;
    }

    synchronized int getBinLimit(String key) {
        return Optional.ofNullable(this.partitions.get(key)).orElseThrow(() -> new IllegalArgumentException("Invalid group " + key)).limit;
    }

    synchronized int getBusyCount() {
        return this.busy;
    }

    public String toString() {
        return "LookupPartitionedStrategy [partitions=" + this.partitions + ", unknownPartition=" + this.unknownPartition + ", limit=" + this.limit + "]";
    }

    private static class Partition {
        private final double percent;
        private final String name;
        private MetricRegistry.SampleListener busyDistribution;
        private int limit;
        private int busy;

        public Partition(String name, double pct) {
            this.name = name;
            this.percent = pct;
        }

        public void createMetrics(MetricRegistry registry) {
            this.busyDistribution = registry.registerDistribution("inflight", LookupPartitionStrategy.PARTITION_TAG_NAME, this.name);
            registry.registerGauge("limit.partition", this::getLimit, LookupPartitionStrategy.PARTITION_TAG_NAME, this.name);
        }

        public void updateLimit(int totalLimit) {
            this.limit = (int)Math.max(1.0, Math.ceil((double)totalLimit * this.percent));
        }

        public boolean isLimitExceeded() {
            return this.busy >= this.limit;
        }

        public void acquire() {
            ++this.busy;
            this.busyDistribution.addSample(this.busy);
        }

        public void release() {
            --this.busy;
        }

        public int getLimit() {
            return this.limit;
        }

        public double getPercent() {
            return this.percent;
        }

        public String toString() {
            return "Partition [pct=" + this.percent + ", limit=" + this.limit + ", busy=" + this.busy + "]";
        }
    }

    public static class Builder<T> {
        private final Function<T, String> lookup;
        private final Map<String, Partition> partitions = new LinkedHashMap<String, Partition>();
        private MetricRegistry registry = EmptyMetricRegistry.INSTANCE;

        protected Builder(Function<T, String> lookup) {
            this.lookup = lookup;
        }

        public Builder<T> metricRegistry(MetricRegistry registry) {
            this.registry = registry;
            return this;
        }

        public Builder<T> assign(String group, Double percent) {
            this.partitions.put(group, new Partition(group, percent));
            return this;
        }

        public LookupPartitionStrategy<T> build() {
            return new LookupPartitionStrategy(this);
        }

        public boolean hasPartitions() {
            return !this.partitions.isEmpty();
        }
    }
}

