/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.concurrency.limits.limit;

import com.netflix.concurrency.limits.Limit;
import com.netflix.concurrency.limits.MetricRegistry;
import com.netflix.concurrency.limits.internal.EmptyMetricRegistry;
import com.netflix.concurrency.limits.internal.Preconditions;
import com.netflix.concurrency.limits.limit.Measurement;
import com.netflix.concurrency.limits.limit.MinimumMeasurement;
import com.netflix.concurrency.limits.limit.functions.SquareRootFunction;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.function.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class GradientLimit
implements Limit {
    private static final int DISABLED = -1;
    private static final Logger LOG = LoggerFactory.getLogger(GradientLimit.class);
    private volatile double estimatedLimit;
    private Measurement rttNoLoad = new MinimumMeasurement();
    private final int maxLimit;
    private final Function<Integer, Integer> queueSize;
    private final double smoothing;
    private final long minRttThreshold;
    private final double rttTolerance;
    private final MetricRegistry.SampleListener minRttSampleListener;
    private final MetricRegistry.SampleListener minWindowRttSampleListener;
    private final MetricRegistry.SampleListener queueSizeSampleListener;
    private final Supplier<Integer> resetRttCounterSupplier;
    private int resetRttCounter;

    public static Builder newBuilder() {
        return new Builder();
    }

    public static GradientLimit newDefault() {
        return GradientLimit.newBuilder().build();
    }

    private GradientLimit(Builder builder) {
        this.estimatedLimit = builder.initialLimit;
        this.maxLimit = builder.maxConcurrency;
        this.queueSize = builder.queueSize;
        this.smoothing = builder.smoothing;
        this.minRttThreshold = builder.minRttThreshold;
        this.rttTolerance = builder.rttTolerance;
        this.resetRttCounterSupplier = builder.resetRttCounterSupplier;
        this.resetRttCounter = this.resetRttCounterSupplier.get();
        this.minRttSampleListener = builder.registry.registerDistribution("min_rtt", new String[0]);
        this.minWindowRttSampleListener = builder.registry.registerDistribution("min_window_rtt", new String[0]);
        this.queueSizeSampleListener = builder.registry.registerDistribution("queue_size", new String[0]);
    }

    @Override
    public synchronized void update(Limit.SampleWindow sample) {
        double newLimit;
        long rtt = sample.getCandidateRttNanos();
        this.minWindowRttSampleListener.addSample(rtt);
        Preconditions.checkArgument(rtt > 0L, "rtt must be >0 but got " + rtt);
        if (rtt < this.minRttThreshold) {
            return;
        }
        double queueSize = this.queueSize.apply((int)this.estimatedLimit).intValue();
        this.queueSizeSampleListener.addSample(queueSize);
        if (this.resetRttCounter != -1 && this.resetRttCounter-- <= 0) {
            LOG.debug("Probe for a new noload RTT");
            this.resetRttCounter = this.resetRttCounterSupplier.get();
            this.estimatedLimit = Math.max(queueSize, this.estimatedLimit / 2.0);
            this.rttNoLoad.reset();
        }
        if (this.rttNoLoad.add(rtt)) {
            LOG.debug("New MinRTT {}", (Object)rtt);
        }
        this.minRttSampleListener.addSample(this.rttNoLoad.get());
        double gradient = Math.max(0.5, Math.min(1.0, this.rttTolerance * (double)this.rttNoLoad.get() / (double)rtt));
        if (sample.didDrop()) {
            newLimit = this.estimatedLimit / 2.0;
        } else {
            if (this.estimatedLimit - (double)sample.getMaxInFlight() > queueSize) {
                return;
            }
            newLimit = this.estimatedLimit * gradient + queueSize;
        }
        newLimit = Math.max(queueSize, Math.min((double)this.maxLimit, newLimit));
        if (newLimit < this.estimatedLimit) {
            newLimit = this.estimatedLimit * (1.0 - this.smoothing) + this.smoothing * newLimit;
        }
        if ((int)newLimit != (int)this.estimatedLimit && LOG.isDebugEnabled()) {
            LOG.debug("New limit={} minRtt={} ms winRtt={} ms queueSize={} gradient={} resetCounter={}", new Object[]{(int)this.estimatedLimit, (double)TimeUnit.NANOSECONDS.toMicros(this.rttNoLoad.get()) / 1000.0, (double)TimeUnit.NANOSECONDS.toMicros(rtt) / 1000.0, queueSize, gradient, this.resetRttCounter});
        }
        this.estimatedLimit = newLimit;
    }

    @Override
    public int getLimit() {
        return (int)this.estimatedLimit;
    }

    public long getRttNoLoad() {
        return this.rttNoLoad.get();
    }

    public String toString() {
        return "GradientLimit [limit=" + (int)this.estimatedLimit + ", rtt_noload=" + (double)TimeUnit.MICROSECONDS.toMillis(this.rttNoLoad.get()) / 1000.0 + " ms]";
    }

    public static class Builder {
        private int initialLimit = 100;
        private int maxConcurrency = 1000;
        private long minRttThreshold = TimeUnit.MICROSECONDS.toNanos(1L);
        private double smoothing = 0.2;
        private Function<Integer, Integer> queueSize = SquareRootFunction.create(4);
        private MetricRegistry registry = EmptyMetricRegistry.INSTANCE;
        private double rttTolerance = 1.0;
        private Supplier<Integer> resetRttCounterSupplier;

        private Builder() {
            this.probeNoLoadRtt(1000, 2000);
        }

        public Builder minRttThreshold(long minRttTreshold, TimeUnit units) {
            this.minRttThreshold = units.toNanos(minRttTreshold);
            return this;
        }

        public Builder initialLimit(int initialLimit) {
            this.initialLimit = initialLimit;
            return this;
        }

        public Builder rttTolerance(double rttTolerance) {
            Preconditions.checkArgument(rttTolerance >= 1.0, "Tolerance must be >= 1.0");
            this.rttTolerance = rttTolerance;
            return this;
        }

        public Builder maxConcurrency(int maxConcurrency) {
            this.maxConcurrency = maxConcurrency;
            return this;
        }

        public Builder queueSize(int queueSize) {
            this.queueSize = ignore -> queueSize;
            return this;
        }

        public Builder queueSize(Function<Integer, Integer> queueSize) {
            this.queueSize = queueSize;
            return this;
        }

        public Builder smoothing(double smoothing) {
            this.smoothing = smoothing;
            return this;
        }

        public Builder metricRegistry(MetricRegistry registry) {
            this.registry = registry;
            return this;
        }

        public Builder probeNoLoadRtt(int minUpdates, int maxUpdates) {
            Preconditions.checkArgument(minUpdates < maxUpdates, "minUpdates must be < maxUpdates");
            this.resetRttCounterSupplier = () -> ThreadLocalRandom.current().nextInt(minUpdates, maxUpdates);
            return this;
        }

        public GradientLimit build() {
            return new GradientLimit(this);
        }
    }
}

