/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.concurrency.limits.strategy;

import com.netflix.concurrency.limits.MetricRegistry;
import com.netflix.concurrency.limits.Strategy;
import com.netflix.concurrency.limits.internal.EmptyMetricRegistry;
import com.netflix.concurrency.limits.internal.Preconditions;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.function.Predicate;

public final class PredicatePartitionStrategy<T>
implements Strategy<T> {
    private static final String PARTITION_TAG_NAME = "partition";
    private final List<Partition<T>> partitions;
    private int busy = 0;
    private int limit = 0;

    public static <T> Builder<T> newBuilder() {
        return new Builder();
    }

    private PredicatePartitionStrategy(Builder<T> builder) {
        Preconditions.checkArgument(builder.partitions.stream().map(Partition::getPercent).reduce(0.0, Double::sum) <= 1.0, "Sum of percentages must be <= 1.0");
        this.partitions = new ArrayList<Partition<T>>(builder.partitions);
        this.partitions.forEach(partition -> partition.createMetrics(builder.registry));
        builder.registry.registerGuage("limit", this::getLimit, new String[0]);
    }

    @Override
    public synchronized Optional<Strategy.Token> tryAcquire(T type) {
        for (Partition<T> partition : this.partitions) {
            if (!((Partition)partition).predicate.test(type)) continue;
            if (this.busy >= this.limit && partition.isLimitExceeded()) {
                return Optional.empty();
            }
            ++this.busy;
            partition.acquire();
            return Optional.of(() -> this.release(partition));
        }
        return Optional.empty();
    }

    private synchronized void release(Partition<T> partition) {
        --this.busy;
        partition.release();
    }

    @Override
    public synchronized void setLimit(int newLimit) {
        if (this.limit != newLimit) {
            this.limit = newLimit;
            this.partitions.forEach(partition -> partition.updateLimit(newLimit));
        }
    }

    public synchronized int getLimit() {
        return this.limit;
    }

    synchronized int getBinBusyCount(int index) {
        Preconditions.checkArgument(index >= 0 && index < this.partitions.size(), "Invalid bin index " + index);
        return ((Partition)this.partitions.get(index)).busy;
    }

    synchronized int getBinLimit(int index) {
        Preconditions.checkArgument(index >= 0 && index < this.partitions.size(), "Invalid bin index " + index);
        return ((Partition)this.partitions.get(index)).limit;
    }

    public synchronized int getBusyCount() {
        return this.busy;
    }

    public String toString() {
        int maxLen = 10;
        return "PercentageStrategy [" + this.partitions.subList(0, Math.min(this.partitions.size(), 10)) + "]";
    }

    private static class Partition<T> {
        private final double percent;
        private final Predicate<T> predicate;
        private final String name;
        private MetricRegistry.SampleListener inflightDistribution;
        private int limit;
        private int busy;

        public Partition(String name, double pct, Predicate<T> predicate) {
            this.name = name;
            this.percent = pct;
            this.predicate = predicate;
        }

        public void createMetrics(MetricRegistry registry) {
            this.inflightDistribution = registry.registerDistribution("inflight", PredicatePartitionStrategy.PARTITION_TAG_NAME, this.name);
            registry.registerGuage("limit.partition", this::getLimit, PredicatePartitionStrategy.PARTITION_TAG_NAME, this.name);
        }

        public void updateLimit(int totalLimit) {
            this.limit = (int)Math.max(1.0, Math.ceil((double)totalLimit * this.percent));
        }

        public boolean isLimitExceeded() {
            return this.busy >= this.limit;
        }

        public void acquire() {
            ++this.busy;
            this.inflightDistribution.addSample(this.busy);
        }

        public void release() {
            --this.busy;
        }

        public int getLimit() {
            return this.limit;
        }

        public double getPercent() {
            return this.percent;
        }

        public String toString() {
            return "Partition [pct=" + this.percent + ", limit=" + this.limit + ", busy=" + this.busy + "]";
        }
    }

    public static class Builder<T> {
        private final List<Partition<T>> partitions = new ArrayList<Partition<T>>();
        private MetricRegistry registry = EmptyMetricRegistry.INSTANCE;

        public Builder<T> metricRegistry(MetricRegistry registry) {
            this.registry = registry;
            return this;
        }

        public Builder<T> add(String name, Double pct, Predicate<T> predicate) {
            this.partitions.add(new Partition<T>(name, pct, predicate));
            return this;
        }

        public PredicatePartitionStrategy<T> build() {
            return new PredicatePartitionStrategy(this);
        }

        public boolean hasPartitions() {
            return !this.partitions.isEmpty();
        }
    }
}

