package com.netease.nimlib.log.core;

import android.text.TextUtils;
import android.util.Log;

import com.netease.nimlib.log.sdk.util.FileUtils;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.io.UnsupportedEncodingException;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.Arrays;

/**
 * Created by huangjun on 2017/8/24.
 * 头四个字节记录当前写入的有效数据偏移
 * 需要输入mapped File 和 destFile
 * mappedFile每写满一定的数据，就flush到destFile中，然后再重头开始写
 * 每次open和close的时候都将已有 mapped file 的数据提交到 dest file 中
 */

public class SimpleMMapWriter {
    private static final String TAG = "SimpleMMapWriter";
    private static final boolean OUTPUT_LOG = false;
    private static final int K = 1024;
    private static final int HEADER_LENGTH = 4;
    private static final int COPY_BUFFER_SIZE = 32 * K;
    private static final int MAX_PRE_WRITE_LINES = 100;

    private final int MAX_BUFFER_SIZE; // 128K
    private final int FLUSH_BUFFER_OFFSET; // 64K

    private RandomAccessFile out;
    private MappedByteBuffer mappedByteBuffer;
    private File mappedFile;
    private File destFile;
    private int linesCount = 0;

    public SimpleMMapWriter() {
        this(0, 0);
    }

    public SimpleMMapWriter(final int maxLength, final int flushOffset) {
        MAX_BUFFER_SIZE = (maxLength > 0 && maxLength > flushOffset) ? maxLength : 128 * K;
        FLUSH_BUFFER_OFFSET = (flushOffset > 0 && flushOffset < maxLength) ? flushOffset : 64 * K;
    }

    public boolean open(final String mappedFilePath, final String destFilePath) {
        if (TextUtils.isEmpty(mappedFilePath) || TextUtils.isEmpty(destFilePath)) {
            return false;
        }

        if (checkValid()) {
            close();
        }

        try {
            // dest file, mapped file data flush to dest file
            destFile = FileUtils.getFile(destFilePath);
            if (destFile == null) {
                log("dest file path invalid, path=" + destFilePath);
                return false;
            }

            // mapped file
            mappedFile = FileUtils.getFile(mappedFilePath);
            if (mappedFile == null) {
                log("mapped file path invalid, path=" + mappedFilePath);
                return false;
            }

            // open
            log("try to open mapped file, path=" + mappedFile.getCanonicalPath());
            out = new RandomAccessFile(mappedFile, "rw");
            if (out.length() <= 0) {
                out.setLength(MAX_BUFFER_SIZE);
            }

            // map
            mappedByteBuffer = out.getChannel().map(FileChannel.MapMode.READ_WRITE, 0, MAX_BUFFER_SIZE);

            // offset
            int offset = moveToWriteOffset();

            // wipe the remaining data which may be the dirty content
            writeEOF(offset, MAX_BUFFER_SIZE - 1);
            mappedByteBuffer.position(offset);

            // flush remain data to dest file
            offset = flush(offset);

            // log
            log("open file success, path=" + mappedFile.getCanonicalPath() + ", offset=" + offset + ", length=" + mappedFile.length());
        } catch (IOException e) {
            e.printStackTrace();
            log("open file error, e=" + e.getMessage());
        }

        return true;
    }

    public void write(String content) {
        if (TextUtils.isEmpty(content)) {
            return;
        }

        // check status
        if (!checkValid()) {
            log("SimpleMappedByteBuffer is invalid when do write");
            return;
        }

        if (mappedFile == null) {
            log("mapped file is null, write failed!");
            return;
        }

        if (destFile == null) {
            log("dest file is null, write failed!");
            return;
        }

        byte[] bytes = null;
        try {
            bytes = content.getBytes("UTF8");
            for (int i = 0; i < bytes.length; i++) {
                if (bytes[i] == 0x00) {
                    bytes[i] = 0x20;
                }
            }
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        }

        if (bytes == null) {
            log("write content null");
            return;
        }

        // write
        mappedByteBuffer.put(bytes);

        // save offset
        saveWriteOffset();

        // flush to mapped file
        checkFlushToMappedFile();

        // log
        log("write position " + mappedByteBuffer.position() + "/" + MAX_BUFFER_SIZE + ", add " + bytes.length);

        // check position, may be need flush to dest file
        int offset = mappedByteBuffer.position();
        if (offset >= FLUSH_BUFFER_OFFSET) {
            log("mapped buffer should flush to dest file, position=" + offset + "/" + MAX_BUFFER_SIZE);
            flush(offset);
        }
    }

    public void close() {
        // force write
        if (mappedByteBuffer != null) {
            // commit all mapped data to dest file
            forceFlush();
            mappedByteBuffer.clear();
            mappedByteBuffer = null;
        }

        // file channel close
        FileUtils.close(out);

        // log
        log("file close success");
    }

    /**
     * 文件裁剪
     */
    private int flush(int offset) {
        // check status
        if (!checkValid()) {
            log("SimpleMappedByteBuffer is invalid when do flush");
            return offset;
        }

        if (offset <= HEADER_LENGTH) {
            log("invalid offset when do flush, offset=" + offset);
            return offset;
        }

        if (destFile == null || !destFile.exists()) {
            log("dest file is not exists when do flush");
            return offset;
        }

        // flush current mapped file's data to dest file
        mappedByteBuffer.position(HEADER_LENGTH);
        int size = offset - HEADER_LENGTH;
        byte[] bytes = new byte[size];
        mappedByteBuffer.get(bytes);
        boolean flushSuccess = FileUtils.appendFile(bytes, destFile.getAbsolutePath());
        if (!flushSuccess) {
            // failed, no need to continue, protect buffer position
            log("flush to dest file failed");
            return offset;
        }

        // reset mapped file head
        mappedByteBuffer.position(0);
        int writePos = saveWriteOffset();

        // wipe remain space
        writeEOF(writePos, MAX_BUFFER_SIZE - 1);

        // force
        mappedByteBuffer.force();

        // offset
        mappedByteBuffer.position(writePos);

        // log
        log("flush file success, new offset=" + writePos);

        return writePos;
    }

    private void writeEOF(int startOffset, int end) {
        if (startOffset < HEADER_LENGTH) {
            return;
        }

        mappedByteBuffer.position(startOffset);

        final byte b = 0x00;
        byte buf[] = null;
        int writeLength;
        int remainLength;
        while (mappedByteBuffer.position() <= end) {
            remainLength = end - mappedByteBuffer.position() + 1;
            writeLength = remainLength < COPY_BUFFER_SIZE ? remainLength : COPY_BUFFER_SIZE;

            // zero memory
            if (buf == null || buf.length != writeLength) {
                buf = new byte[writeLength];
                Arrays.fill(buf, b);
            }

            // write
            mappedByteBuffer.put(buf);
        }

        log("write EOF from " + startOffset + " to " + end);
    }

    private int moveToWriteOffset() {
        mappedByteBuffer.position(0);
        int offset = mappedByteBuffer.getInt();
        if (offset >= HEADER_LENGTH && offset < MAX_BUFFER_SIZE) {
            mappedByteBuffer.position(offset);
            return offset;
        } else {
            mappedByteBuffer.position(0);
            return saveWriteOffset();
        }
    }

    private int saveWriteOffset() {
        int offset = mappedByteBuffer.position();
        if (offset < HEADER_LENGTH) {
            offset = HEADER_LENGTH;
        }

        mappedByteBuffer.position(0);
        mappedByteBuffer.putInt(offset);
        mappedByteBuffer.position(offset);

        return offset;
    }

    public void forceFlush() {
        if (!checkValid()) {
            return;
        }

        // commit all mapped data to dest file
        flush(mappedByteBuffer.position());

        log("force flush to dest file");
    }

    private void checkFlushToMappedFile() {
        if (++linesCount >= MAX_PRE_WRITE_LINES && checkValid()) {
            mappedByteBuffer.force();
            linesCount = 0;
            log("flush to mapped file");
        }
    }

    private boolean checkValid() {
        return out != null && mappedByteBuffer != null;
    }

    /**
     * ******************************* log ********************************
     */

    private void log(String content) {
        if (OUTPUT_LOG) {
            Log.i(TAG, content);
        }
    }
}
