package com.netcore.android.notification

import android.app.NotificationManager
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.graphics.Bitmap
import android.os.Bundle
import android.view.View
import android.widget.RemoteViews
import androidx.core.app.NotificationCompat
import com.netcore.android.R
import com.netcore.android.event.SMTEventRecorder
import com.netcore.android.logger.SMTLogger
import com.netcore.android.mediadownloader.SMTMediaDownloadManager
import com.netcore.android.notification.models.SMTNotificationData
import com.netcore.android.utility.SMTCommonUtility
import org.json.JSONObject


/**
 * Copyright © 2019 Netcore. All rights reserved.
 *
 * Base abstract notification generator class having common methods
 * All other notification generator classes Extends this class
 *
 * @author Netcore
 * @version 1.0
 * @since 26-02-2019
 */

internal class SMTRichImagePNGenerator : SMTBaseNotificationGenerator() {

    lateinit var notifModel: SMTNotificationData
    private val TAG = SMTRichImagePNGenerator::class.java.simpleName

    /**
     * Create the notification based on the Payload and notify to Notification manager to display it
     * @param context - App context
     * @param notifData - Extras passed through the pending intent
     */
    fun handle(context: Context, notifData: SMTNotificationData, sourceType: Int) {

        val notifOptions = SMTCommonUtility.getNotificationOptions(context)

        this.notifModel = notifData

//        notifModel.notificationId = SMTCommonUtility.getRandomId()

        if (notifModel.notificationId == 0) {
            notifModel.notificationId = SMTCommonUtility.getRandomId()
            updateNotificationId(context, notifModel)
        }
//        updateNotificationId(context, notifModel)

        if (notifModel.mMediaUrl.isNullOrEmpty() || notifModel.mIsScheduledPN == 1) {
            // show scheduled Notification
            showNotification(context, notifModel, notifOptions)
        } else {
            // Download media
            SMTMediaDownloadManager().downloadMedia(context, notifModel, object : SMTMediaDownloadManager.MediaDownloadListener {

                override fun onDownloadSuccess(notification: SMTNotificationData) {
                    notifModel = notification
                    showNotification(context, notification, notifOptions)
                }

                override fun onDownloadFailed(notification: SMTNotificationData) {
                    showNotification(context, notification, notifOptions)
                }
            })
        }

    }

    /**
     * Shows ntofication in system tray
     * @param context - App context
     * @param notificationParcel - Extras passed through the pending intent
     *
     */
    private fun showNotification(context: Context, notificationParcel: SMTNotificationData, notifOptions: SMTNotificationOptions) {
        notificationManager = context.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager

        val title = notificationParcel.mTitle ?: ""
        val message = notificationParcel.mMessage ?: ""
        val subtitle = notificationParcel.mSubtitle ?: ""

        var remotePicture: Bitmap? = null
        try {
            if (!notificationParcel.mMediaLocalPath.equals("", true)) {
                remotePicture = SMTCommonUtility.loadImageFromLocalStorage(notificationParcel.mMediaLocalPath!!)
            }
        } catch (e: Exception) {
            SMTLogger.e(TAG, e.message.toString())
        }

        val remoteView = RemoteViews(context.packageName, R.layout.notification_rich_image_layout)

        // Build the content of the notification
        val nBuilder = getNotificationBuilder(context,
                title,
                message,
                subtitle,
                createPendingIntent(context, notificationParcel),
                notificationParcel)


        remoteView.setTextViewText(R.id.img_title, SMTCommonUtility.parseHtml(title))
        remoteView.setTextViewText(R.id.img_message, SMTCommonUtility.parseHtml(notificationParcel.mMessage))
        remoteView.setTextViewText(R.id.img_big_text, SMTCommonUtility.parseHtml(notificationParcel.mMessage))

        if(notificationParcel.mStickyEnabled){
            remoteView.setViewVisibility(R.id.img_icon_close,View.VISIBLE)
            val closeIntent = SMTNotificationUtility.getInstance().getActionPendingIntent(context,notifModel,SMTNotificationConstants.IMAGE_NOTIF_OTHER_REGION_CLICKED_KEY, SMTNotificationConstants.NOTIF_STICKY_ACTION_CLOSE)
            remoteView.setOnClickPendingIntent(R.id.img_icon_close, closeIntent)
        }

        nBuilder.setStyle(NotificationCompat.DecoratedCustomViewStyle())
        nBuilder.setLargeIcon(null)
        // Set the notification style depend upon image
        if (remotePicture != null) {
            remoteView.setViewVisibility(R.id.img_big_text,View.GONE)
            remoteView.setImageViewBitmap(R.id.img_image, remotePicture);

        } else {
            remoteView.setViewVisibility(R.id.img_image,View.GONE)
            remoteView.setViewVisibility(R.id.img_message,View.GONE)
        }
        nBuilder.setCustomBigContentView(remoteView)

        // check and add action buttons
        checkAndAddActionButtons(context, notifModel, nBuilder)

        // notify to notification manager
        notificationManager?.notify(notificationParcel.notificationId, nBuilder.build())

    }


    /**
     * Notification Content Pending Intent created
     * @param context - App context
     * @param data - Parsed Notification data model
     * @return PendingIntent - receiver pending intent
     */
    private fun createPendingIntent(context: Context, data: SMTNotificationData): PendingIntent {
        val rand = System.currentTimeMillis().toInt()
        val bundle = Bundle()
        bundle.putString(SMTNotificationConstants.NOTIF_TYPE_KEY, SMTNotificationType.BIG_IMAGE.type)
        bundle.putString(SMTNotificationConstants.IMAGE_NOTIF_OTHER_REGION_CLICKED_KEY, SMTNotificationConstants.IMAGE_NOTIF_CLICKED)
        bundle.putParcelable(SMTNotificationConstants.NOTIFICATION_PARCEL, data)
        bundle.putBoolean(SMTNotificationConstants.NOTIF_STICKY_ENABLED,notifModel.mStickyEnabled)
        val launchIntent = Intent(context, SMTPNActionReceiver::class.java)
        launchIntent.putExtras(bundle)

        return PendingIntent.getBroadcast(context, rand, launchIntent, PendingIntent.FLAG_UPDATE_CURRENT)
    }

    /**
     * Handles the image click action
     * @param context App context
     * @param extras - Extras passed through the pending intent
     */
    fun handleImageClick(context: Context, extras: Bundle) {
        if (extras.containsKey(SMTNotificationConstants.IMAGE_NOTIF_OTHER_REGION_CLICKED_KEY)) {
            if (SMTNotificationConstants.IMAGE_NOTIF_CLICKED == extras.get(SMTNotificationConstants.IMAGE_NOTIF_OTHER_REGION_CLICKED_KEY)) {
                val parcel = extras.getParcelable<SMTNotificationData>(SMTNotificationConstants.NOTIFICATION_PARCEL)
                //Update DB record whether it has been clicked

                SMTEventRecorder.getInstance(context).recordNotificationClick(parcel?.mTrid
                        ?: "", parcel?.mPNMeta ?: JSONObject().toString(),
                        parcel?.mDeepLinkPath ?: "", parcel?.mSource
                        ?: 1, parcel?.mSmtAttributePayload ?: HashMap(), parcel?.mIsScheduledPN!!)

                SMTCommonUtility.handleNotificationClick(context, parcel.mDeepLinkPath
                        ?: "", parcel.mCustomPayload)

                // Delete the file from local file system
                if (parcel.mMediaLocalPath?.isNotEmpty() == true) {
                    SMTCommonUtility.deleteFile(parcel.mMediaLocalPath!!)
                }
            }
        }

    }

    /**
     * Handles the action on notification dismiss
     * @param context - App context
     * @param extras - Extras passed through the pending intent
     */
    override fun handleNotificationDismiss(context: Context?, extras: Bundle) {

        if (extras.containsKey(SMTNotificationConstants.NOTIFICATION_PARCEL)) {
            val parcel = extras.getParcelable(SMTNotificationConstants.NOTIFICATION_PARCEL) as SMTNotificationData
            if (parcel.mMediaLocalPath?.isNotEmpty() == true) {
                SMTCommonUtility.deleteFile(parcel.mMediaLocalPath!!)
            }
        }
    }
}