package com.netcore.android.inapp

import android.annotation.SuppressLint
import android.app.Activity
import android.content.Context
import android.graphics.Color
import android.graphics.drawable.ColorDrawable
import android.os.Build
import android.util.DisplayMetrics
import android.view.Gravity
import android.view.View
import android.view.WindowManager
import android.webkit.WebView
import android.widget.PopupWindow
import androidx.annotation.VisibleForTesting
import com.netcore.android.SMTEventParamKeys
import com.netcore.android.SMTInAppConst
import com.netcore.android.Smartech
import com.netcore.android.db.SMTDataBaseService
import com.netcore.android.event.SMTEventId
import com.netcore.android.event.SMTEventPayloadCreator
import com.netcore.android.event.SMTEventRecorder
import com.netcore.android.event.SMTEventType
import com.netcore.android.inapp.model.SMTInAppRule
import com.netcore.android.logger.SMTLogger
import com.netcore.android.network.SMTThreadPoolManager
import com.netcore.android.network.models.SMTSdkInitializeResponse
import com.netcore.android.preference.SMTPreferenceConstants
import com.netcore.android.preference.SMTPreferenceHelper
import com.netcore.android.utility.SMTCommonUtility
import org.json.JSONException
import org.json.JSONObject
import java.lang.ref.WeakReference
import java.net.URL
import java.util.*
import java.util.concurrent.TimeUnit


@Suppress("PrivatePropertyName")
/**
 * @author Netcore
 * created on 8/03/2019
 * @Description: Class which handles the InApp rule display
 *
 */
internal class SMTInAppHandler private constructor() : SMTInAppRuleActionListener {


    private val TAG = SMTInAppHandler::class.java.simpleName

    private var mPopupWindow: PopupWindow? = null

    private var isRecordInAppDismissEent = true

    companion object {
        @Volatile
        private var instance: SMTInAppHandler? = null

        @Synchronized
        fun getInstance(): SMTInAppHandler =
                SMTInAppHandler.instance ?: synchronized(SMTInAppHandler::class.java) {
                    SMTInAppHandler.instance
                            ?: SMTInAppHandler.buildInstance().also { SMTInAppHandler.instance = it }
                }


        private fun buildInstance(): SMTInAppHandler {
            return SMTInAppHandler()
        }
    }

    /**
     * Check the event payload against the applicable inapp rule
     * @param payloadMap payload of a event. System or custom event
     */
    fun checkRule(payloadMap: HashMap<String, Any>) {

        if (isInAppOptIn()) {
            /**
             * If activity is in foreground then show the in app notifications
             */
            if (SMTInAppUtility.isAppInForeground()) {
                val activity: Activity? = SMTInAppUtility.getForeGroundActivity() ?:
                /**
                 * current foreground activity is null. so returning here
                 */
                return

                val inAppRules: MutableList<SMTInAppRule>? = getInAppRules(payloadMap, activity!!)
                val identifiedRule = findARule(activity, inAppRules, payloadMap)
                if (identifiedRule != null) {
                    if (SMTCommonUtility.isNetworkAvailable(activity)) {
                        showInAppRule(activity, identifiedRule)
                    } else {
                        SMTLogger.internal(TAG, "Network is not available.")
                    }
                }


            } else {
                SMTLogger.e(TAG, "Application isn't in foreground so rejecting the InAPP request")
            }
        }
    }

    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    fun isInAppOptIn(): Boolean {
        SMTInAppUtility.getForeGroundActivity()?.let {
            return SMTPreferenceHelper.getAppPreferenceInstance(it.applicationContext, null).getBoolean(SMTPreferenceConstants.OPT_IN_OUT_IN_APP_MESSAGES, true)
        }

        return false
    }

    /**
     * Get all the matching in app rules for the given event and for the current time
     */
    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    fun getInAppRules(eventPayLoad: HashMap<String, Any>, activity: Activity): MutableList<SMTInAppRule>? {
        return SMTDataBaseService.getInstance(WeakReference(activity.applicationContext)).getInAppRules(eventPayLoad)
    }

    /**
     * Find a rule which matches the criteria considering the filters and usage
     */
    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    private fun findARule(context: Context, inAppRules: MutableList<SMTInAppRule>?, eventPayLoad: HashMap<String, Any>): SMTInAppRule? {

        var matchedRule: SMTInAppRule? = null
        if (inAppRules != null && inAppRules.size > 0) {
            val listIterator = inAppRules.listIterator()
            while (listIterator.hasNext()) {
                val inAppRule = listIterator.next()
                //println("inapp inAppRule.id: ${inAppRule.id}")
                // Save MID, CG and RandomValue in pref so we can send
                // these value in other events
                SMTPreferenceHelper.getAppPreferenceInstance(context, null).setString(SMTPreferenceConstants.SMT_MID, inAppRule.id)
                SMTPreferenceHelper.getAppPreferenceInstance(context, null).setString(SMTPreferenceConstants.SMT_CG, if (inAppRule.controlGroup > inAppRule.randomNumber) "1" else "0")
                SMTPreferenceHelper.getAppPreferenceInstance(context, null).setString(SMTPreferenceConstants.SMT_CG_RANDOM, inAppRule.randomNumber.toString())
                SMTPreferenceHelper.getAppPreferenceInstance(context, null).setString(SMTPreferenceConstants.SMT_CG_RULE, inAppRule.controlGroup.toString())

                //proceed with rule checking if the control group is less than randomNumber
                if (inAppRule.randomNumber > inAppRule.controlGroup) {
                    //println("inapp payload: $eventPayLoad")

                    //Visitor and its type check
                    if (SMTInAppUtility.checkTheVisitor(inAppRule)) {

                        var isShowRule =
                                SMTInAppUtility.checkRuleFilter(inAppRule.whereTo.trigger.filters, inAppRule.whereTo.trigger.filterType, eventPayLoad) && checkTheUsage(inAppRule)

                        if (isShowRule) {
                            val listIds = getListSegDataObject(context, SMTInAppConst.SMT_LIST_SEGMENT_LIST_IDS)
                            val segIds = getListSegDataObject(context, SMTInAppConst.SMT_LIST_SEGMENT_SEG_IDS)
                            isShowRule = SMTInAppUtility.checkListAndSegmentIds(inAppRule, listIds, segIds) && SMTInAppUtility.checkMultiEvents(context, inAppRule)
                        }

                        if (isShowRule) {
                            matchedRule = inAppRule
                            break
                        }
                    }

                } else {
                    // Don't show  InApp but send InAppViewed event
                    recordInAppEvent(SMTEventId.EVENT_INAPP_VIEWED, inAppRule)
                }

            }
        }
        return matchedRule
    }


    /**
     * Method to check if matched rule still valid to show, considering its usage statics
     * @param inAppRule Filter matched InAppRule object
     */
    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    private fun checkTheUsage(inAppRule: SMTInAppRule): Boolean {
        var isShow = true
        //println("inapp : check ${inAppRule.frequencyType}")
        when (inAppRule.frequencyType) {
            SMTInAppUtility.FREQUENCY_TYPE_DAY -> {

                if (SMTInAppUtility.isApplicableNow(inAppRule, Date())) {
                    //Check if the usage is for today is over or not
                    val currentDateTimeInMilli = SMTInAppUtility.getTodayInMilliSec()
                    isShow = if (inAppRule.frequencyTypeValue == currentDateTimeInMilli) {
                        isNotLimitReached(inAppRule)
                    } else {
                        //println("inapp : day :${inAppRule.frequencyTypeValue} --- $currentDateTimeInMilli")
                        // so the date has changed so don't compare the frequency directly show
                        true
                    }
                }
            }
            SMTInAppUtility.FREQUENCY_TYPE_SESSION -> {
                isShow = if (SMTInAppUtility.isCurrentTimeWithinTheRange(inAppRule, Date())) {
                    isNotLimitReached(inAppRule)
                } else {
                    false
                }
            }
            SMTInAppUtility.FREQUENCY_TYPE_CAMPAIGN -> {
                //if type is campaign then we need to just check if the usage is
                // less than the frequency because we already applied the from and to date filter
                // while querying the rules
                isShow = if (SMTInAppUtility.isCurrentTimeWithinTheRange(inAppRule, Date())) {
                    isNotLimitReached(inAppRule)
                } else {
                    false
                }
            }
            else -> {
                isShow = if (SMTInAppUtility.isApplicableNow(inAppRule, Date())) {
                    when (SMTInAppUtility.isCurrentTimeWithinTheRange(inAppRule, Date())) {
                        true -> {
                            isNotLimitReached(inAppRule)
                        }
                        false -> {
                            false
                        }
                    }
                } else {
                    false
                }

            }
        }
        SMTLogger.i(TAG, "InApp 2: $isShow")
        //println("inapp checkTheUsage: $isShow")
        return isShow

    }

    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    private fun isNotLimitReached(inAppRule: SMTInAppRule): Boolean {
        //println("inapp : freq ${inAppRule.frequency}---${inAppRule.frequency}")
        if (inAppRule.frequency.isNotEmpty() && Integer.parseInt(inAppRule.frequency) != 0) {
            //println("inapp : isNotLimitReached ${inAppRule.alreadyViewedCount} : ${Integer.parseInt(inAppRule.frequency)}")
            return inAppRule.alreadyViewedCount < Integer.parseInt(inAppRule.frequency)
        }
        return true
    }


    /**
     * Show In app Rule
     */
    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    fun showInAppRule(identifiedRule: SMTInAppRule) {
        SMTInAppUtility.getForeGroundActivity()?.runOnUiThread(kotlinx.coroutines.Runnable {
            showInAppPopupWindow(identifiedRule)
        })
    }

    private fun showInAppRule(context: Context, identifiedRule: SMTInAppRule) {
        try {
            val delay = SMTPreferenceHelper.getAppPreferenceInstance(context, null)
                    .getLong(SMTPreferenceConstants.SMT_INAPP_WAIT_TIME)

            //println("inapp delay : $delay")
            if (delay > 0) {
                val inAppPopupRunner = Runnable {
                    checkEventInDB(context, identifiedRule)
                }
                // Start running the thread to perform the task
                SMTThreadPoolManager.getIntance().schedule(inAppPopupRunner, delay, TimeUnit.MILLISECONDS)

            } else {
                SMTInAppUtility.getForeGroundActivity()?.runOnUiThread(kotlinx.coroutines.Runnable {
                    showInAppPopupWindow(identifiedRule)
                })
            }
        } catch (e: Exception) {
            SMTLogger.e(TAG, e.message.toString())
        }
    }

    private fun checkEventInDB(context: Context, identifiedRule: SMTInAppRule) {
        //Check whether event done within delay
        try {
            var isShow = true
            val pos = SMTPreferenceHelper.getAppPreferenceInstance(context, null)
                    .getInt(SMTPreferenceConstants.SMT_INAPP_ME_POS)
            val multiEventsRules = identifiedRule.whomTo.events.rules
            //println("inapp : pos $pos multiEventsRules: $multiEventsRules")
            if (pos >= 0 && multiEventsRules.size > 0) {
                val multiEventsRule = multiEventsRules[pos]

                // Check eventname present in event list table
                val eventPayLoad: HashMap<String, Any> = SMTEventPayloadCreator.createMultiEventPayload(multiEventsRule)
                val smtEventList = SMTDataBaseService.getInstance(WeakReference(context.applicationContext))
                        .getStoredEventData(eventPayLoad)
                val filters = multiEventsRule.filters

                if (!smtEventList.isNullOrEmpty() && filters.size > 0) {
                    val smtEvent = smtEventList[0]
                    //println("inapp : ${smtEvent.eventId} ---- ${smtEvent.eventName}")
                    //println("inapp : ${smtEvent.eventPayload}")
                    //println("inapp : ${smtEvent.eventDate}")
                    isShow = SMTInAppUtility.checkRuleFilter(filters, multiEventsRule.filterType,
                            SMTCommonUtility.jsonToHashMap(JSONObject(smtEvent.eventPayload)))
                    //println("inapp : $isShow")
                    isShow = !isShow
                    //println("inapp : $isShow")
                } else isShow = smtEventList.isNullOrEmpty()
            }
            //println("inapp : $isShow")

            //reseting delay and position of event
            SMTPreferenceHelper.getAppPreferenceInstance(context, null)
                    .setLong(SMTPreferenceConstants.SMT_INAPP_WAIT_TIME, 0)
            SMTPreferenceHelper.getAppPreferenceInstance(context, null)
                    .setInt(SMTPreferenceConstants.SMT_INAPP_ME_POS, -1)

            if (isShow) {
                SMTInAppUtility.getForeGroundActivity()?.runOnUiThread(kotlinx.coroutines.Runnable {
                    showInAppPopupWindow(identifiedRule)
                })
            }
        } catch (e: Exception) {
            SMTLogger.e(TAG, e.message.toString())
        }

    }

    /**
     * Method to show the in app popup window using the selected in app rule
     */
    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    private fun showInAppPopupWindow(identifiedRule: SMTInAppRule) {

        //If there is already inapp rule is showing then dismiss the that and show the new one
        if (mPopupWindow?.isShowing == true) {

            mPopupWindow?.dismiss()
        }

        val webView = createWebView(identifiedRule)
        if (webView != null) {

            setJavaScriptInterface(webView, identifiedRule)
            showPopUpWindow(identifiedRule, webView)
        }

    }

    fun dismiss() {
        mPopupWindow?.dismiss()
        mPopupWindow = null
    }

    /**
     * Method to create the PopupWindow instance using the rule and show it at the defined location
     */
    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    private fun showPopUpWindow(identifiedRule: SMTInAppRule, webView: WebView) {


        if (SMTInAppUtility.getForeGroundActivity() != null) {
            val displayMetrics = DisplayMetrics()
            SMTInAppUtility.getForeGroundActivity()?.windowManager!!.defaultDisplay.getMetrics(displayMetrics)
            val height = displayMetrics.heightPixels
            val width = displayMetrics.widthPixels

            try {
                when (Integer.parseInt(identifiedRule.whereTo.position)) {
                    SMTInAppConstants.WhereTo.FULL_SCREEN.value -> {
                        mPopupWindow = PopupWindow(webView, -1, -1, true)
                        mPopupWindow!!.setBackgroundDrawable(ColorDrawable(Color.TRANSPARENT))
                        mPopupWindow!!.showAtLocation(webView, Gravity.CENTER, 0, 0)

                    }
                    SMTInAppConstants.WhereTo.INTERSTITIAL.value -> {
                        mPopupWindow = PopupWindow(webView, width - (width / 10), height - (height / 10), true)
                        mPopupWindow!!.setBackgroundDrawable(ColorDrawable(Color.TRANSPARENT))
                        mPopupWindow!!.showAtLocation(webView, Gravity.CENTER, 0, 0)

                    }
                    SMTInAppConstants.WhereTo.HALF_INTERSTIAL.value -> {
                        mPopupWindow = PopupWindow(webView, width - (width / 10), height / 2, true)
                        mPopupWindow!!.setBackgroundDrawable(ColorDrawable(Color.TRANSPARENT))
                        mPopupWindow!!.showAtLocation(webView, Gravity.CENTER, 0, 0)

                    }
                    SMTInAppConstants.WhereTo.STICKY_HEADER.value -> {
                        mPopupWindow = PopupWindow(webView, -1, height / 5, true)
                        mPopupWindow!!.setBackgroundDrawable(ColorDrawable(Color.TRANSPARENT))
                        mPopupWindow!!.showAtLocation(webView, Gravity.TOP, 0, 0)

                    }
                    SMTInAppConstants.WhereTo.STICKY_FOOTER.value -> {
                        mPopupWindow = PopupWindow(webView, -1, height / 5, true)
                        mPopupWindow!!.setBackgroundDrawable(ColorDrawable(Color.TRANSPARENT))
                        mPopupWindow!!.showAtLocation(webView, Gravity.BOTTOM, 0, 0)

                    }
                    else -> {
                        mPopupWindow = PopupWindow(webView, -1, height / 5, true)
                        mPopupWindow!!.setBackgroundDrawable(ColorDrawable(Color.TRANSPARENT))
                        mPopupWindow!!.showAtLocation(webView, Gravity.BOTTOM, 0, 0)

                    }
                }
                mPopupWindow!!.setOnDismissListener {
                    if (isRecordInAppDismissEent) {
                        recordInAppEvent(SMTEventId.EVENT_INAPP_DISMISSED, identifiedRule)
                    }
                }

                dimBehind(mPopupWindow!!, 0.2f)
                updateUsageDetails(identifiedRule)
                recordInAppEvent(SMTEventId.EVENT_INAPP_VIEWED, identifiedRule)

            } catch (e: Exception) {
                SMTLogger.e(TAG, "InApp display issue.")
            }
        }

    }

    override fun recordInAppEvent(eventId: Int, inAppRule: SMTInAppRule, url: String?) {

        SMTInAppUtility.getForeGroundActivity()?.let {

            val payloadMap = hashMapOf<String, Any>()
            payloadMap[SMTEventParamKeys.SMT_MID] = inAppRule.id
            payloadMap[SMTEventParamKeys.SMT_CG] = if (inAppRule.controlGroup > inAppRule.randomNumber) {
                "1"
            } else {
                "0"
            }

            when (eventId) {
                SMTEventId.EVENT_INAPP_CLICKED -> {
                    payloadMap[SMTEventParamKeys.SMT_IN_APP_CLICK_LINK] = url!!
                }
                SMTEventId.EVENT_INAPP_VIEWED -> {
                    payloadMap[SMTEventParamKeys.SMT_CG_REPEAT] = if (inAppRule.alreadyViewedCount > 0) {
                        1
                    } else {
                        0
                    }
                    payloadMap[SMTEventParamKeys.SMT_CG_CONTROL_GROUP] = inAppRule.controlGroup
                    payloadMap[SMTEventParamKeys.SMT_CG_RANDOM_NO] = inAppRule.randomNumber
                }
                SMTEventId.EVENT_INAPP_DISMISSED -> {

                }

            }

            SMTEventRecorder.getInstance(it.applicationContext).recordEvent(eventId, SMTEventId.getEventName(eventId), payloadMap, SMTEventType.EVENT_TYPE_SYSTEM)
        }
    }

    /**
     * Method to update the In App rules table. update the viewed count
     */

    private fun updateUsageDetails(inAppRule: SMTInAppRule) {

        if (SMTInAppUtility.getForeGroundActivity() != null) {
            SMTDataBaseService.getInstance(WeakReference(SMTInAppUtility.getForeGroundActivity()!!.applicationContext)).updateInAppRuleUsage(inAppRule, SMTInAppUtility.getTodayInMilliSec())
        }

    }

    /**
     * Function which sets the
     */
    private fun setJavaScriptInterface(webView: WebView, inAppRule: SMTInAppRule) {
        if (SMTInAppUtility.getForeGroundActivity() != null) {
            val inAppHTMLListener = Smartech.getInstance(WeakReference(SMTInAppUtility.getForeGroundActivity()!!.applicationContext)).getInAppCustomHTMLListener()
            webView.addJavascriptInterface(SMTInAppJavaScriptInterface(inAppRule, this, inAppHTMLListener), "jse")
        }


    }

    override fun closeActionClicked(isRecordInAppDismissEent: Boolean) {
        this.isRecordInAppDismissEent = isRecordInAppDismissEent
        SMTInAppUtility.getForeGroundActivity()?.runOnUiThread(kotlinx.coroutines.Runnable {
            mPopupWindow?.dismiss()
        })
    }


    @SuppressLint("SetJavaScriptEnabled")
    private fun createWebView(inAppRule: SMTInAppRule): WebView? {
        SMTInAppUtility.getForeGroundActivity()?.let {

            val wv = WebView(it)
            wv.settings.javaScriptEnabled = true
            wv.settings.domStorageEnabled = true
            wv.settings.javaScriptCanOpenWindowsAutomatically = true
            wv.setBackgroundColor(0x00000000)
            wv.loadUrl(getUrl(inAppRule, it))

            return wv
        }
        return null

    }

    private fun getUrl(inAppRule: SMTInAppRule, activity: Activity): String {

        return when (shouldAppendInAppCreativeWithBaseURL(inAppRule.whatTo.url)) {
            true -> inAppRule.whatTo.url
            false -> "${
                SMTPreferenceHelper.getAppPreferenceInstance(activity.applicationContext, null)
                        .getString(SMTPreferenceConstants.SMT_BASE_URL_INAPP)
            }inapp?${inAppRule.whatTo.url}"
        }
    }

    private fun shouldAppendInAppCreativeWithBaseURL(url: String): Boolean {
        return try {
            URL(url).toURI()
            true
        } catch (e: java.lang.Exception) {
            false
        }
    }


    /**
     * Method to update the inapp rules
     */
    internal fun processInAppRules(initSdkResponse: SMTSdkInitializeResponse, context: Context) {
        SMTLogger.d(TAG, "Processing inapp rules")

        if (initSdkResponse.inAppRules != null && initSdkResponse.inAppRules!!.size > 0) {

            //Deleting the other in app rules
            val ruleIds = getAllRuleIds(initSdkResponse.inAppRules!!)

            SMTDataBaseService.getInstance(WeakReference(context.applicationContext)).deleteOtherInAppRules(ruleIds)


            // insert the new rules
            SMTDataBaseService.getInstance(WeakReference(context.applicationContext)).insertInAppRules(initSdkResponse.inAppRules!!)


        } else {
            // delete all inapp rules
            SMTDataBaseService.getInstance(WeakReference(context.applicationContext)).deleteOtherInAppRules(null)
        }

    }

    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    fun getAllRuleIds(inAppRules: ArrayList<SMTInAppRule>): String {
        val inQuery = StringBuilder()

        inQuery.append("(")
        var first = true
        for (item in inAppRules) {
            if (first) {
                first = false
                inQuery.append("'").append(item.id).append("'")
            } else {
                inQuery.append(", '").append(item.id).append("'")
            }
        }
        inQuery.append(")")

        return inQuery.toString()
    }

    fun resetUsageForSessionTypeRule(context: Context) {
        SMTDataBaseService.getInstance(WeakReference(context.applicationContext)).resetUsageForSessionTypeRule()

        // Reset CG, MID and RandomNo parameters
        SMTPreferenceHelper.getAppPreferenceInstance(context, null).setString(SMTPreferenceConstants.SMT_MID, "")
        SMTPreferenceHelper.getAppPreferenceInstance(context, null).setString(SMTPreferenceConstants.SMT_CG, "")
        SMTPreferenceHelper.getAppPreferenceInstance(context, null).setString(SMTPreferenceConstants.SMT_CG_RANDOM, "")
        SMTPreferenceHelper.getAppPreferenceInstance(context, null).setString(SMTPreferenceConstants.SMT_CG_RULE, "")
    }


    private fun dimBehind(popupWindow: PopupWindow, value: Float) {
        val container: View = if (popupWindow.background == null) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                popupWindow.contentView.parent as View
            } else {
                popupWindow.contentView
            }
        } else {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                popupWindow.contentView.parent.parent as View
            } else {
                popupWindow.contentView.parent as View
            }
        }
        val context = popupWindow.contentView.context
        val wm = context.getSystemService(Context.WINDOW_SERVICE) as WindowManager
        val p = container.layoutParams as WindowManager.LayoutParams
        p.flags = p.flags or WindowManager.LayoutParams.FLAG_DIM_BEHIND // add a flag here instead of clear others
        p.dimAmount = value
        wm.updateViewLayout(container, p)
    }

    fun getListSegDataObject(context: Context, key: String): List<String> {
        val inAppListSeg: String? = SMTPreferenceHelper.getAppPreferenceInstance(context, null).getString(SMTPreferenceConstants.SMT_LIST_SEGMENT_DATA, "")
        //println("inapp : list and segment prefs $inAppListSeg")
        var listSegDataList: List<String> = ArrayList()
        try {
            if (inAppListSeg != null && inAppListSeg != "") {
                val inAppListSegObject = JSONObject(inAppListSeg)
                val dataObject = inAppListSegObject.optJSONObject(SMTInAppConst.SMT_LIST_SEGMENT_DATA)
                listSegDataList = SMTCommonUtility.jsonArrayToStringList(dataObject.optJSONArray(key))
            }
        } catch (e: JSONException) {
            SMTLogger.e(TAG, e.message.toString())
        }
        return listSegDataList
    }
}