package com.netcore.android.event

import android.app.NotificationManager
import android.content.Context
import android.content.Intent
import android.nfc.Tag
import android.os.Bundle
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import com.netcore.android.SMTConfigConstants
import com.netcore.android.SMTEventParamKeys
import com.netcore.android.SMTHanselConst
import com.netcore.android.Smartech
import com.netcore.android.db.SMTDataBaseService
import com.netcore.android.db.SMTNotificationTable
import com.netcore.android.inapp.SMTInAppHandler
import com.netcore.android.inapp.SMTInAppUtility
import com.netcore.android.inbox.utility.SMTInboxMessageStatus
import com.netcore.android.logger.SMTLogger
import com.netcore.android.notification.SMTNotificationConstants
import com.netcore.android.notification.SMTNotificationType
import com.netcore.android.notification.models.SMTNotificationData
import com.netcore.android.preference.SMTPreferenceConstants
import com.netcore.android.preference.SMTPreferenceHelper
import com.netcore.android.utility.SMTCommonUtility
import com.netcore.android.utility.SMTGWSource
import com.netcore.android.utility.SMTInfo
import com.netcore.android.workmgr.SMTWorkerScheduler

import org.json.JSONObject
import java.lang.ref.WeakReference


@Suppress("UNUSED_PARAMETER", "RedundantIf")
/**
 * Copyright © 2019 Netcore. All rights reserved.
 *
 * Singleton class that records the events to Notification, InApp and Event Table
 *
 * @author Netcore
 * @version 1.0
 * @since 06-03-2019
 */
internal class SMTEventRecorder private constructor(val context: Context) {

    companion object {
        private var mDbService: SMTDataBaseService? = null
        private var mSmtInfo: SMTInfo? = null

        @Volatile
        private var INSTANCE: SMTEventRecorder? = null

        /**
         * Getting instance of the class
         */
        fun getInstance(context: Context): SMTEventRecorder =
                INSTANCE ?: synchronized(SMTEventRecorder::class.java) {
                    INSTANCE ?: buildInstance(context).also { INSTANCE = it }
                }

        private fun buildInstance(context: Context): SMTEventRecorder {

            mSmtInfo = SMTInfo.getInstance(WeakReference(context))
            mDbService = SMTDataBaseService.getInstance(WeakReference(context))

            return SMTEventRecorder(context)
        }
    }

    /**
     * Records notification delivery to Event and Notificaiton table as well
     * @param trid - notification transaction id
     * @param payload - notification payload
     * @param source - notificaiton source type
     * @param notifModel - notificaiton model
     *
     */
    fun recordNotificationDelivery(trid: String, payload: String, @SMTNotificationSourceType.Source source: Int,
                                   notifModel: SMTNotificationData) {
        val hashMap: HashMap<String, Any> = HashMap()
        hashMap[SMTEventParamKeys.SMT_TRID] = trid
        hashMap[SMTEventParamKeys.SMT_PNMETA] = if (notifModel.mPNMeta.isNullOrEmpty()) JSONObject().toString() else notifModel.mPNMeta
        hashMap[SMTEventParamKeys.SMT_IS_AMPLIFIED] = when (source == SMTNotificationSourceType.NOTIFICATION_SOURCE_PAMP) {
            true -> 1
            false -> 0
        }
        hashMap[SMTEventParamKeys.SMT_GWSOURCE] = when (source) {
            SMTNotificationSourceType.NOTIFICATION_SOURCE_PN -> SMTGWSource.FCM.value
            SMTNotificationSourceType.NOTIFICATION_SOURCE_PAMP -> SMTGWSource.PUSH_AMP.value
            SMTNotificationSourceType.NOTIFICATION_SOURCE_XIAOMI -> SMTGWSource.XIAOMI.value
            else -> SMTGWSource.FCM.value
        }

        hashMap[SMTEventParamKeys.SMT_CT] = when (notifModel.mIsScheduledPN == 1) {
            true -> 1
            false -> 0
        }


        recordEvent(SMTEventId.EVENT_PN_DELIVERED, SMTEventId.getEventName(SMTEventId.EVENT_PN_DELIVERED), hashMap, SMTEventType.EVENT_TYPE_SYSTEM)
    }

    /**
     * Records notification click to Event and Notificaiton table as well
     * @param trid - notification transaction id
     * @param apnClickLink - deeplink path
     * @param source - notificaiton source type
     *
     */
    fun recordNotificationClick(trid: String, pnMeta: String, apnClickLink: String,
                                @SMTNotificationSourceType.Source source: Int, smtAttributes: HashMap<String, String>, isSheduledPn: Int = 0) {

        mDbService?.updateInboxMessageStatus(trid, SMTInboxMessageStatus.Status.READ)

        SMTCommonUtility.updateAttributionParams(context, apnClickLink, smtAttributes)

        if(mDbService?.getNotificationClickedStatusById(trid) == false) {
            mDbService?.updateNotification(trid, SMTNotificationTable.KEY_IS_CLICKED, true)

            val hashMap: HashMap<String, Any> = HashMap()
            hashMap[SMTEventParamKeys.SMT_TRID] = trid
            hashMap[SMTEventParamKeys.SMT_PNMETA] = pnMeta
            hashMap[SMTEventParamKeys.SMT_IS_AMPLIFIED] = when (source == SMTNotificationSourceType.NOTIFICATION_SOURCE_PAMP) {
                true -> 1
                false -> 0
            }
            hashMap[SMTEventParamKeys.SMT_GWSOURCE] = when (source) {
                SMTNotificationSourceType.NOTIFICATION_SOURCE_PN -> SMTGWSource.FCM.value
                SMTNotificationSourceType.NOTIFICATION_SOURCE_PAMP -> SMTGWSource.PUSH_AMP.value
                SMTNotificationSourceType.NOTIFICATION_SOURCE_XIAOMI -> SMTGWSource.XIAOMI.value
                else -> SMTGWSource.FCM.value
            }
            hashMap[SMTEventParamKeys.SMT_APN_CLICK_LINK] = apnClickLink
            hashMap[SMTEventParamKeys.SMT_CT] = when (isSheduledPn == 1) {
                true -> 1
                false -> 0
            }

            recordEvent(SMTEventId.EVENT_PN_CLICKED, SMTEventId.getEventName(SMTEventId.EVENT_PN_CLICKED), hashMap, SMTEventType.EVENT_TYPE_SYSTEM)
        }
    }

    /**
     * Records notification dismiss to Event and Notificaiton table as well
     * @param trid - notification transaction id
     * @param source - notificaiton source type
     *
     */
    fun recordNotificationDismiss(trid: String, pnMeta: String, @SMTNotificationSourceType.Source source: Int, isSheduledPn: Int = 0) {

        mDbService?.updateNotification(trid, SMTNotificationTable.KEY_IS_DISMISSED, true)

        mDbService?.updateInboxMessageStatus(trid, SMTInboxMessageStatus.Status.DELETED)

        val hashMap: HashMap<String, Any> = HashMap()
        hashMap[SMTEventParamKeys.SMT_TRID] = trid
        hashMap[SMTEventParamKeys.SMT_IS_AMPLIFIED] = when (source == SMTNotificationSourceType.NOTIFICATION_SOURCE_PAMP) {
            true -> 1
            false -> 0
        }
        hashMap[SMTEventParamKeys.SMT_GWSOURCE] = when (source) {
            SMTNotificationSourceType.NOTIFICATION_SOURCE_PN -> SMTGWSource.FCM.value
            SMTNotificationSourceType.NOTIFICATION_SOURCE_PAMP -> SMTGWSource.PUSH_AMP.value
            SMTNotificationSourceType.NOTIFICATION_SOURCE_XIAOMI -> SMTGWSource.XIAOMI.value
            else -> SMTGWSource.FCM.value
        }

        hashMap[SMTEventParamKeys.SMT_PNMETA] = pnMeta
        hashMap[SMTEventParamKeys.SMT_CT] = when (isSheduledPn == 1) {
            true -> 1
            false -> 0
        }

        /**
         *
         *  Check status of EventSync worker is running.
         *  If worker is running, It will automatically pickup the delivery of notification event.
         *  If worker is not running, start Event Sync worker so it will send
         *  the delivery event to server,
         *
         */

        // If event tracking is allowed then sync PN dissmiss event
        if (SMTCommonUtility.checkIfTrackingAllowed(context)) {
            SMTWorkerScheduler.getInstance().checkStatusAndScheduleEventWorker(context)
        }

        recordEvent(SMTEventId.EVENT_PN_DISMISSED, SMTEventId.getEventName(SMTEventId.EVENT_PN_DISMISSED), hashMap, SMTEventType.EVENT_TYPE_SYSTEM)
    }

    /**
     * Records the events both custom and system
     *
     * @param eventId event id
     * @param eventName event name
     * @param payload payload in HashMap format
     * @param eventType event type - system or custom
     */
    fun recordEvent(eventId: Int, eventName: String?, payload: HashMap<String, Any>?, eventType: String, isEventFromHansel: Boolean = false) {
        //val eventName = eventName?.toLowerCase(Locale.getDefault())


        /**
         *  Bypass this validation if event id is  SMTEventId.EVENT_APP_INSTALLED / SMTEventId.EVENT_PN_TOKEN_GENERATED, / SMTEventId.EVENT_LOCATION_FETCH_ENABLED
         *  Reason to this to record event if sdk is not initialised but we have to prevent other events
         *  from getting logged.
         */
        val isTrackingAllowed = SMTCommonUtility.checkIfTrackingAllowed(context)
        SMTLogger.internal("SMTEventRecorder", "Status of tracking: $isTrackingAllowed")
        if (!SMTCommonUtility.eventsRepository(eventId) && !isTrackingAllowed) {
            return
        }

        mSmtInfo?.let {

            // Passing data to Hansel SDK and adding data back to our payload only if its a Hansel event.

            if (!isEventFromHansel) {
                val hansel = Smartech.getInstance(WeakReference(context)).getHanselInstance()
                val hanselData = hansel?.logEvent(eventName!!, SMTHanselConst.SMT_KEY, payload)
                if (hanselData != null) {
                    payload?.putAll(hanselData)
                }
            }

            val eventMap = SMTEventPayloadCreator.createEventPayload(context, eventId, eventName, it, payload)

            // if payload is null then set empty string else create the json string
            val eventPayload = if (eventMap.size == 0) {
                return
            } else {

                var obj = JSONObject(eventMap)

                var lat = obj.optString(SMTEventParamKeys.SMT_LATITUDE)
                var long = obj.optString(SMTEventParamKeys.SMT_LONGITUDE)

                if (lat.isEmpty() || lat.toDouble() == 0.0) {
                    obj.put(SMTEventParamKeys.SMT_LATITUDE, JSONObject.NULL)
                }

                if (lat.isEmpty() || long.toDouble() == 0.0) {
                    obj.put(SMTEventParamKeys.SMT_LONGITUDE, JSONObject.NULL)
                }

                obj.toString()
            }

            mDbService?.insertEventToDb(eventId, eventName, eventPayload, eventType)
            mDbService?.deleteEvents(SMTPreferenceHelper.getAppPreferenceInstance(context, null)
                    .getInt(SMTPreferenceConstants.EVENT_LIMIT, SMTConfigConstants.DEFAULT_EVENT_LIMIT_SIZE))
            SMTLogger.i("SMTEventRecorder", "Event name: $eventName")
            SMTLogger.i("SMTEventRecorder", "Event type: $eventType")
            SMTLogger.i("SMTEventRecorder", "Event Payload: $eventPayload")
            SMTLogger.i("SMTEventRecorder", "Event id: $eventId")
            if (!SMTInAppUtility.isInAppEvent(eventId)) {
                // check inapp rule
                if (eventId > 0) {
                    if (validateSystemEventsForInApp(context, eventId)) {
                        SMTInAppHandler.getInstance().checkRule(eventMap)
                    }
                } else {
                    SMTInAppHandler.getInstance().checkRule(eventMap)
                }

            }

        } ?: SMTLogger.internal("SMTEventRecorder", "SMTInfo is null.")

    }


    internal fun recordInboxEvent(trid: String, deepLinkPath: String?, @SMTInboxMessageStatus.Status status: Int) {

        mDbService?.updateInboxMessageStatus(trid, status)

        val eventId = when (status) {
            SMTInboxMessageStatus.Status.DELETED -> SMTEventId.EVENT_INBOX_DISMISSED
            SMTInboxMessageStatus.Status.VIEWED -> SMTEventId.EVENT_INBOX_VIEWED
            SMTInboxMessageStatus.Status.READ -> SMTEventId.EVENT_INBOX_CLICKED
            SMTInboxMessageStatus.Status.DELIVERED -> SMTEventId.EVENT_INBOX_DELIVERED
            else -> SMTEventId.EVENT_INBOX_DELIVERED
        }

        if (eventId == SMTEventId.EVENT_INBOX_CLICKED || eventId == SMTEventId.EVENT_INBOX_DISMISSED) {
            updateNotificationTray(trid, eventId)
        }


        deepLinkPath?.let { path ->
            SMTCommonUtility.updateAttributionParams(context, path)
        }


        val hashMap: HashMap<String, Any> = HashMap()
        hashMap[SMTEventParamKeys.SMT_TRID] = trid
        if (eventId == SMTEventId.EVENT_INBOX_CLICKED) {
            hashMap[SMTEventParamKeys.SMT_INBOX_CLICK_LINK] = deepLinkPath ?: ""
        }

        recordEvent(eventId, SMTEventId.getEventName(eventId), hashMap, SMTEventType.EVENT_TYPE_SYSTEM)
    }

    private fun updateNotificationTray(trid: String, eventId: Int) {
        val notif = mDbService?.getNotificationById(trid)
        notif?.let {
            val mNotifyManager = context.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
            mNotifyManager.cancel(it.notificationId)

            if (it.mNotificationType == SMTNotificationType.AUDIO.type) {
                val intent = Intent(SMTConfigConstants.BROADCAST_EVENT_AUDIO_DISMISS)

                val bundle = Bundle()
                bundle.putParcelable(SMTNotificationConstants.NOTIFICATION_PARCEL, it)
                intent.putExtras(bundle)
                // You can also include some extra data.
                context.let { appContext ->
                    LocalBroadcastManager.getInstance(appContext).sendBroadcast(intent)
                }
            }
        }
    }


    /**
     * Records scheduled notification pn_rendered to Event and Notificaiton table as well
     * @param trid - notification transaction id
     * @param payload - notification payload
     * @param source - notificaiton source type
     * @param notifModel - notificaiton model
     *
     */
    fun recordNotificationPNRendered(trid: String, payload: String, @SMTNotificationSourceType.Source source: Int,
                                     notifModel: SMTNotificationData) {


        val hashMap: HashMap<String, Any> = HashMap()
        hashMap[SMTEventParamKeys.SMT_TRID] = trid
        hashMap[SMTEventParamKeys.SMT_PNMETA] = if (notifModel.mPNMeta.isNullOrEmpty()) JSONObject().toString() else notifModel.mPNMeta
        hashMap[SMTEventParamKeys.SMT_IS_AMPLIFIED] = when (notifModel.mSourceType == SMTNotificationSourceType.NOTIFICATION_SOURCE_PAMP) {
            true -> 1
            false -> 0
        }

        hashMap[SMTEventParamKeys.SMT_GWSOURCE] = when (source == SMTNotificationSourceType.NOTIFICATION_SOURCE_PAMP) {
            true -> SMTGWSource.PUSH_AMP.value
            false -> SMTGWSource.FCM.value
        }
        hashMap[SMTEventParamKeys.SMT_CT] = when (notifModel.mIsScheduledPN == 1) {
            true -> 1
            false -> 0
        }


        recordEvent(SMTEventId.EVENT_PN_RENDERED, SMTEventId.getEventName(SMTEventId.EVENT_PN_RENDERED), hashMap, SMTEventType.EVENT_TYPE_SYSTEM)
    }

    private fun validateSystemEventsForInApp(context: Context, eventId: Int): Boolean {

        val isEventPresent = when (eventId) {
            SMTEventId.EVENT_APP_INSTALLED,
            SMTEventId.EVENT_APP_REINSTALLED,
            SMTEventId.EVENT_APP_LAUNCHED,
            SMTEventId.EVENT_DEVICE_DETAILS_UPDATED,
            SMTEventId.EVENT_LOCATION_FETCH_ENABLED,
            SMTEventId.EVENT_LOCATION_FETCH_DISABLED,
            SMTEventId.EVENT_USER_ENABLED_PN -> true
            else -> false
        }

        val smartech = Smartech.getInstance(WeakReference(context))
        val status = smartech.getInAppRuleListStatus()
        SMTLogger.i("SMTEventRecorder", "InApp : Flag is $status , Event id $eventId present $isEventPresent")
        if (isEventPresent) {
            if (!status) {
                val systemList = smartech.getSystemInAppEventList();
                systemList.add(eventId)
                return false
            }
        }
        return true;
    }
}

