package com.netcore.android.notification

import android.app.NotificationManager
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.graphics.BitmapFactory
import android.graphics.Color
import android.os.Build
import android.os.Bundle
import android.os.Parcelable
import androidx.core.app.NotificationCompat
import com.netcore.android.db.SMTDataBaseService
import com.netcore.android.notification.carousel.SMTCarouselSetup
import com.netcore.android.notification.channel.SMTNotificationChannelHelper
import com.netcore.android.notification.models.SMTNotificationData
import com.netcore.android.utility.SMTCommonUtility
import java.lang.ref.WeakReference

/**
 * Copyright © 2019 Netcore. All rights reserved.
 *
 * Base abstract notification generator class having common methods
 * All other notification generator classes Extends this class
 *
 * @author Netcore
 * @version 1.0
 * @since 26-02-2019
 */
internal abstract class SMTBaseNotificationGenerator {

    /**
     * Notificaiton Manager
     */
    protected var notificationManager: NotificationManager? = null

    /**
     * It is used to check if there is any action buttons are there
     * If there then add to Notification Builder
     * @param context - App context
     * @param notifModel - Parsed Motification data
     * @param notificationBuilder - reference of notification builder object
     */

    /**
     * Channel helper class to get the channel ID and sound
     */
    private lateinit var mSmtNotificationChannelHelper: SMTNotificationChannelHelper

    protected fun checkAndAddActionButtons(context: Context, notifModel: SMTNotificationData, notificationBuilder: NotificationCompat.Builder) {
        val actionButtonList = notifModel.mActionButtonList
        var randomActionId: Int
        actionButtonList?.forEach { actionButton ->
            randomActionId = SMTCommonUtility.getRandomId()

            // Adding the info to bundle
            val actionBundle = Bundle()
            actionBundle.putString(SMTNotificationConstants.NOTIF_ACTION_NAME_KEY, actionButton.actionName)
            actionBundle.putString(SMTNotificationConstants.NOTIF_ACTION_DEEPLINK_KEY, actionButton.actionDeeplink)
            actionBundle.putInt(SMTNotificationConstants.NOTIF_ACTION_ID, randomActionId)
            actionBundle.putString(SMTNotificationConstants.NOTIF_ACTION, SMTNotificationConstants.NOTIF_ACTION)
            actionBundle.putParcelable(SMTNotificationConstants.NOTIFICATION_PARCEL, notifModel)


            //Creating Action Intent
            val actionIntent = Intent(context, SMTPNActionReceiver::class.java)
            actionIntent.action = SMTNotificationConstants.NOTIF_ACTIONS
            actionIntent.addFlags(Intent.FLAG_INCLUDE_STOPPED_PACKAGES)
            actionIntent.putExtras(actionBundle)

            // Add the action Pending Intent to notificaiton builder
            val actionPIntent = PendingIntent.getBroadcast(context, randomActionId, actionIntent, PendingIntent.FLAG_UPDATE_CURRENT)
            notificationBuilder.addAction(0, actionButton.actionName, actionPIntent)
        }
    }

    /**
     * Create Delete Pending Intent, used to listen whenever the notification
     * dismissed by user by swiping out or clicking clear all button
     */
    open fun createDeleteIntent(context: Context, notificationParcel: Parcelable): PendingIntent {

        val intent = Intent(context, SMTPNActionReceiver::class.java)
        intent.action = SMTNotificationConstants.NOTIF_ACTION_DISMISS

        val bundle = Bundle()
        if (notificationParcel is SMTNotificationData) {
            bundle.putString(SMTNotificationConstants.NOTIF_TYPE_KEY, notificationParcel.mNotificationType)
        } else if (notificationParcel is SMTCarouselSetup) {
            bundle.putString(SMTNotificationConstants.NOTIF_TYPE_KEY, notificationParcel.notifType)
        }
        bundle.putParcelable(SMTNotificationConstants.NOTIFICATION_PARCEL, notificationParcel)

        intent.putExtras(bundle)
        val notifId = if (notificationParcel is SMTNotificationData) {
            notificationParcel.notificationId
        } else {
            (notificationParcel as SMTCarouselSetup).carouselNotificationId
        }
        return PendingIntent.getBroadcast(
                context,
                notifId,
                intent,
                PendingIntent.FLAG_ONE_SHOT
        )
    }

    /**
     * Initialize the notification manager and channel Id.
     * The notification builder has the basic initialization:
     *     - AutoCancel=true
     *     - LargeIcon = SmallIcon
     * @param [context] application context for associate the notification with.
     * @param [notificationTitle] notification title.
     * @param [notificationText] notification text.
     * @return the PendingIntent to be used on this notification.
     */
    protected fun getNotificationBuilder(context: Context,
                                         notificationTitle: String,
                                         notificationText: String,
                                         notificationSubtitle: String,
                                         contentPendingIntent: PendingIntent,
                                         notificationParcel: Parcelable): NotificationCompat.Builder {
        // Define the notification channel for newest Android versions

        lateinit var builder: NotificationCompat.Builder
        var mSound = true
        lateinit var mSoundFile: String
        lateinit var smtChannelId: String
        mSmtNotificationChannelHelper = SMTNotificationChannelHelper.getInstance(WeakReference(context))

        val channelData = when (notificationParcel) {
            is SMTNotificationData -> {
                Triple(notificationParcel.mChannelId
                        ?: "", notificationParcel.mSound, notificationParcel.mSoundFile ?: "")
            }

            is SMTCarouselSetup -> {
                Triple(notificationParcel.mChannelId
                        ?: "", notificationParcel.mSound, notificationParcel.mSoundFile ?: "")
            }
            else -> Triple("", "", "")
        }

        smtChannelId = channelData.first
        mSound = channelData.second as Boolean
        mSoundFile = channelData.third


        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            smtChannelId = mSmtNotificationChannelHelper.getNotificationChannelID(smtChannelId)
            builder = NotificationCompat.Builder(context, smtChannelId)
        } else {
            @Suppress("DEPRECATION")
            builder = NotificationCompat.Builder(context)
        }

        val notifOptions = SMTCommonUtility.getNotificationOptions(context)

        // Build the content of the notification
        builder.setContentTitle(SMTCommonUtility.parseHtml(notificationTitle))
                .setContentText(SMTCommonUtility.parseHtml(notificationText))
                .setLargeIcon(BitmapFactory.decodeResource(context.resources, notifOptions.largeIconId))
                .setAutoCancel(true)
                .setContentIntent(contentPendingIntent)
                .setOnlyAlertOnce(true)
                .setDeleteIntent(createDeleteIntent(context, notificationParcel))
                .setBadgeIconType(NotificationCompat.BADGE_ICON_SMALL)

        if (notificationSubtitle.trim().isNotEmpty()) {
            builder.setSubText(SMTCommonUtility.parseHtml(notificationSubtitle))
        }

        // This will work only on version below 8.0. For above it will take sound from channel.
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.O && mSound) {
            builder.setSound(mSmtNotificationChannelHelper.getSoundUri(mSoundFile))
        }

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            builder.setSmallIcon(notifOptions.smallIconTransparentId)
            builder.color = Color.parseColor(notifOptions.transparentIconBgColor)
        } else {
            builder.setSmallIcon(notifOptions.smallIconId)
        }

        return builder
    }

    /**
     * Abstract method, all subclasses required to implement this method
     * to handle notificaiton dismiss
     */
    abstract fun handleNotificationDismiss(context: Context?, extras: Bundle)

    internal fun updateNotificationId(context: Context, notifModel: SMTNotificationData) {
        SMTDataBaseService.getInstance(WeakReference(context)).updateNotificationId(notifModel.mTrid, notifModel.notificationId)
    }
}