package com.netcore.android.network.models

import com.netcore.android.network.SMTEnumHttpMethodType
import com.netcore.android.network.SMTResponseListener
import com.netcore.android.utility.SMTCommonUtility
import org.json.JSONArray
import org.json.JSONObject
import java.util.*


/**
 * @Description This is the request class, it will have following info of a request.
 * It uses Builder design pattern to create the request.
 * @param builder Request builder class reference
 *
 * @author Netcore
 * @version 1.0
 * @since 26-02-2019
 */
open class SMTRequest(builder: Builder) {

    private var requestId: Long = 0

    private val httpMethod: SMTEnumHttpMethodType?
    private val headers: HashMap<String, String>?
    private val urlEndPoint: String?
    private val params: JSONObject?
    private val responseListener: SMTResponseListener?
    private val apiID: SMTApiTypeID
    private var retryCount: Int = 0
    private val baseUrl: String

    init {
        httpMethod = builder.httpMethod
        headers = builder.headers
        urlEndPoint = builder.urlEndPoint
        params = builder.jsonRequestObject
        requestId = builder.createRequestId()
        responseListener = builder.responseListener
        apiID = builder.apiID
        baseUrl = builder.baseUrl ?: ""
    }

    internal fun getAPITypeID(): SMTApiTypeID {
        return apiID
    }

    internal fun getBaseUrl(): String {
        return baseUrl
    }

    internal fun setRetryCount(retryCount: Int = 0) {
        this.retryCount = retryCount
    }

    internal fun getRetryCount(): Int {
        return retryCount
    }

    internal fun getHttpMethod(): SMTEnumHttpMethodType? {
        return httpMethod
    }

    /*fun getHeaders(): HashMap<String, String>? {
        return headers
    }*/

    internal fun getApiEndPoint(): String? {
        return urlEndPoint
    }

    internal fun getHParams(): JSONObject? {
        return params
    }

    internal fun getRequestId(): Long {
        return requestId
    }

    internal fun getResponseListener(): SMTResponseListener? {
        return responseListener
    }

    /**
     * Request Builder class used to create the request object.
     */
    /**
     * This is the request class, it will have following info of a request.
     * It uses Builder design pattern to create the request.
     * httpMethod - Type of Http Request
     * headers - Header params of the request
     * urlEndPoint - The URL of the request
     * params - The request body
     * requestId - Unique Identifier of the request
     * responseListener - Listener of request caller
     */
    class Builder {
        internal var httpMethod: SMTEnumHttpMethodType? = null
        internal var headers: HashMap<String, String>? = null
        internal var urlEndPoint: String? = null
        private var params: JSONArray? = null
        internal var responseListener: SMTResponseListener? = null
        internal var jsonRequestObject: JSONObject? = null
        internal lateinit var apiID: SMTApiTypeID
        internal var baseUrl: String? = null

        internal fun setBaseUrl(url: String) = apply {
            this.baseUrl = url
        }

        internal fun setApiId(apiId: SMTApiTypeID) = apply {
            this.apiID = apiId
        }

        internal fun setHttpMethod(method: SMTEnumHttpMethodType) = apply { this.httpMethod = method }

        /*fun setHeaders(headers: HashMap<String, String>?) = apply {
            if (headers != null) headers["Content-Type"] = "application/json"
            this.headers = headers
        }*/

        internal fun setEndPoint(url: String) = apply {
            this.urlEndPoint = url
        }

        internal fun setParams(params: JSONArray) = apply {
            this.params = params
        }

        internal fun setResponseListener(listener: SMTResponseListener) = apply { this.responseListener = listener }

        internal fun build(): SMTRequest {
            when (httpMethod?.value) {
                SMTEnumHttpMethodType.GET.value -> createUrlEndPoint()
                else -> createJsonRequestObject()
            }
            return SMTRequest(this)
        }

        private fun createJsonRequestObject() {
            jsonRequestObject = JSONObject()

            params?.let {
                if (it.length() == 1) {
                    jsonRequestObject = it.getJSONObject(0)
                } else {
                    jsonRequestObject!!.put("data", it)

                }
            }

        }

        /**
         * To construct the URL using params, in case of GET HTTP method.
         */
        private fun createUrlEndPoint() {
            var isFirstParam = true
            if (params != null && params!!.length() > 0) {
                val jsonObject = params!!.getJSONObject(0)
                val map = SMTCommonUtility.jsonToMap(jsonObject.toString())
                map?.let {
                    for (key in it.keys) {
                        val value = it.getValue(key)
                        if (isFirstParam) {
                            isFirstParam = false
                            urlEndPoint = "$urlEndPoint?"
                            urlEndPoint = "$urlEndPoint$key=$value"
                        } else {
                            urlEndPoint = "$urlEndPoint&"
                            urlEndPoint = "$urlEndPoint$key=$value"
                        }
                    }
                }
            }
        }

        /**
         * System Time Stamp used as request Id
         * @return current time in millis
         */
        internal fun createRequestId(): Long {
            return System.currentTimeMillis()
        }


    }


    /**
     * API ID
     */
    internal enum class SMTApiTypeID(val value: Int) {
        SDK_INITIALIZE(1),
        SDK_INITIALIZE_ON_SESSION_REFRESH(2),
        BATCH_PROCESSING_API(3),
        PUSH_AMPLIFICATION(4),
        INBOX_API(5),
        LIST_SEGMENT(6)
    }
}