package com.netcore.android.inbox.views

import android.content.Context
import android.graphics.Bitmap
import android.os.Handler
import android.os.Looper
import android.util.AttributeSet
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.LinearLayout
import android.widget.ProgressBar
import androidx.appcompat.widget.AppCompatImageView
import androidx.appcompat.widget.AppCompatTextView
import com.netcore.android.R
import com.netcore.android.mediadownloader.SMTDownloaderUtility
import com.netcore.android.mediadownloader.SMTMediaDownloadManager
import com.netcore.android.notification.SMTNotificationOptions
import com.netcore.android.notification.gif.GifRetriever
import com.netcore.android.notification.models.SMTNotificationData
import com.netcore.android.utility.SMTCommonUtility

/**
 * Copyright © 2019 Netcore. All rights reserved.
 *
 * Inbox Image Message Custom View - Handles showing image message
 * Also it downloads the image, record events
 *
 * @author Netcore
 * @version 1.0
 * @since 29-04-2019
 */
internal class SMTInboxGifMessageView(
        context: Context, attrs: AttributeSet?) : SMTBaseView(context, attrs) {

    private lateinit var imageInboxView: View
    private lateinit var gifImageView: AppCompatImageView

    private var mGifImageRetriever: GifRetriever? = null
    private var mCurrentBitmapFrame: Bitmap
    private var notifOptions: SMTNotificationOptions

    private val mMainUiHandler: Handler

    init {
        initLayout()
        notifOptions = SMTCommonUtility.getNotificationOptions(context)
        mCurrentBitmapFrame = SMTCommonUtility.getBitmapFromResId(context, notifOptions.placeHolderIcon)
        mMainUiHandler = Handler(Looper.getMainLooper())
    }

    private fun initLayout() {

        imageInboxView = LayoutInflater.from(context).inflate(R.layout.inbox_gif_message_layout, this)

        val lp = LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.WRAP_CONTENT)

        this.layoutParams = lp

        this.orientation = LinearLayout.VERTICAL

        gifImageView = findViewById(R.id.iv_gif_image_view)

    }

    override fun setNotificationData(notification: SMTNotificationData) {
        setData(notification)

        setViewDetail()

        setLayoutAction()

        setBackgroundShape()

        fetchGif()
    }

    private fun setViewDetail() {
        imageInboxView.findViewById<AppCompatTextView>(R.id.tv_gif_title).text = mNotificationData.mTitle
        imageInboxView.findViewById<AppCompatTextView>(R.id.tv_gif_timestamp).text = SMTCommonUtility.getFormattedTimeDifference(mNotificationData.mPublishedTimeStamp)
        imageInboxView.findViewById<AppCompatTextView>(R.id.tv_gif_message).text = mNotificationData.mMessage
    }

    private fun showProgressBar() {

        imageInboxView.findViewById<ProgressBar>(R.id.loadingProgressBar).visibility = View.VISIBLE
    }

    private fun hideProgressBar() {

        imageInboxView.findViewById<ProgressBar>(R.id.loadingProgressBar).visibility = View.GONE
    }

    /**
     * If media already downloaded then show it
     * else download
     */
    private fun fetchGif() {
        // T)DO check if media is already downloaded

        val notifOptions = SMTCommonUtility.getNotificationOptions(context)

        showGifImage(SMTCommonUtility.getBitmapFromResId(context, notifOptions.placeHolderIcon), false)

        if (mNotificationData.mMediaLocalPath.isNullOrEmpty()) {
            showProgressBar()
            downloadGif()
        } else {
            hideProgressBar()
            fetchGifFrameAndDisplay(true)
        }
    }

    /**
     * Download image
     */
    private fun downloadGif() {
        mNotificationData.mIsForInbox = true

        if (!mNotificationData.mIsDownloadInProgress) {
            mNotificationData.mIsDownloadInProgress = true

            // Download media
            SMTMediaDownloadManager().downloadMedia(context, mNotificationData, object : SMTMediaDownloadManager.MediaDownloadListener {

                override fun onDownloadSuccess(notification: SMTNotificationData) {
                    if (notification.mTrid == mNotificationData.mTrid) {
                        mNotificationData.mIsDownloadInProgress = false
                        hideProgressBar()
                        setData(notification)
                        processGifImage()
                    }
                }

                override fun onDownloadFailed(notification: SMTNotificationData) {
                    if (notification.mTrid == mNotificationData.mTrid) {
                        hideProgressBar()
                        mNotificationData.mIsDownloadInProgress = false
                    }
                }
            })
        }
    }

    /**
     * to set the bitmap in Image and also decides whether to show the GIF play icon
     * @param bitmap btimap to show
     * @param showGifIcon to decide GIF play icon
     */
    private fun showGifImage(bitmap: Bitmap, showGifIcon: Boolean) {
        mMainUiHandler.post {
            gifImageView.setImageBitmap(bitmap)
            toggleGifPlayIcon(showGifIcon)
        }
    }

    private fun toggleGifPlayIcon(showGifIcon: Boolean) {
        if (showGifIcon) {
            findViewById<AppCompatImageView>(R.id.iv_gif_play_icon).visibility = View.VISIBLE
        } else {
            findViewById<AppCompatImageView>(R.id.iv_gif_play_icon).visibility = View.INVISIBLE
        }
    }

    /**
     * Fetch the GIF frame and display
     */
    private fun processGifImage() {
        fetchGifFrameAndDisplay(true)
    }

    /**
     * Fetch GIF frames and render it on the ui
     * @param isOnlyRetrieveFirstFrame - very first time fetch first fram and display
     */
    private fun fetchGifFrameAndDisplay(isOnlyRetrieveFirstFrame: Boolean) {
        mGifImageRetriever = GifRetriever()
        mGifImageRetriever?.onFrameAvailable = GifRetriever.OnFrameAvailable { bitmap ->
            bitmap?.let {
                var showGifIcon = false
                // If GIF looped once then stop it
                // If user wants to see it again then have to click on GIF play button
                if ((mGifImageRetriever?.currentFrameIndex == (mGifImageRetriever?.frameCount!! - 1))
                        || isOnlyRetrieveFirstFrame) {
                    showGifIcon = true
                    mGifImageRetriever?.stopAnimation()
                    mGifImageRetriever = null
                }
                if (isOnlyRetrieveFirstFrame) {
                    mCurrentBitmapFrame = it
                }
                showGifImage(it, showGifIcon)
            }
        }
        mGifImageRetriever?.setBytes(SMTDownloaderUtility.getGifBytes(mNotificationData.mMediaLocalPath))
        mGifImageRetriever?.startAnimation()
    }


    /**
     * Set action listener to the layout
     */
    private fun setLayoutAction() {
        imageInboxView.setOnClickListener {
            handleClick(mNotificationData.mDeepLinkPath)
        }
        findViewById<AppCompatImageView>(R.id.iv_gif_play_icon).setOnClickListener {
            handleGifPlayIconClick()
        }
    }

    /**
     * Handled Gif Play icon click
     * if its playing stop it else if
     */
    private fun handleGifPlayIconClick() {

        handleClick(null)

        if (mGifImageRetriever?.isAnimating == true) {
            showGifImage(mCurrentBitmapFrame, true)
            mGifImageRetriever?.stopAnimation()
            mGifImageRetriever = null
            toggleGifPlayIcon(true)
        } else {
            toggleGifPlayIcon(false)
            fetchGifFrameAndDisplay(false)
        }

    }

}