package com.netcore.android.inapp

import android.app.Activity
import android.content.Context
import androidx.annotation.VisibleForTesting
import com.netcore.android.SMTActivityLifecycleCallback
import com.netcore.android.SMTConfigConstants
import com.netcore.android.SMTEventParamKeys
import com.netcore.android.SMTInAppConst
import com.netcore.android.db.SMTDataBaseService
import com.netcore.android.event.SMTEventId
import com.netcore.android.event.SMTEventPayloadCreator
import com.netcore.android.inapp.model.SMTEvents
import com.netcore.android.inapp.model.SMTInAppRule
import com.netcore.android.logger.SMTLogger
import com.netcore.android.preference.SMTPreferenceConstants
import com.netcore.android.preference.SMTPreferenceHelper
import com.netcore.android.utility.SMTCommonUtility
import org.json.JSONArray
import org.json.JSONObject
import java.lang.Float
import java.lang.ref.WeakReference
import java.text.SimpleDateFormat
import java.util.*
import java.util.concurrent.TimeUnit
import java.util.regex.Pattern

internal class SMTInAppUtility {


    companion object {
        private val TAG = SMTInAppUtility::class.java.simpleName

        private const val FILTER_PAYLOAD_TYPE_ARRAY_OF_STRINGS_OR_INT = 1
        private const val FILTER_PAYLOAD_TYPE_ARRAY_OF_JSON_OBJECT = 2
        private const val FILTER_PAYLOAD_TYPE_NORMAL_OBJECT = 3
        private const val FILTER_PAYLOAD_TYPE_NESTED_OBJECT = 4
        private const val FILTER_PAYLOAD_TYPE_UNKNOWN = 5


        //Frequency Type constants
        const val FREQUENCY_TYPE_DAY = "day"
        const val FREQUENCY_TYPE_SESSION = "session"
        const val FREQUENCY_TYPE_CAMPAIGN = "campaign"


        /**
         * Check if app is in foreground
         */
        fun isAppInForeground(): Boolean {

            return SMTActivityLifecycleCallback.getInstance().isAppInForeground()
        }

        /**
         * Getting the foreground activity from ActivityLifeCycleCallback
         */
        fun getForeGroundActivity(): Activity? {
            return SMTActivityLifecycleCallback.getInstance().getActivity()?.get()
        }


        /**
         * Method to identify the payload structure. Payload can be a normal object,
         * array of string/integer, array of json objects,
         * nested json object or it could be unknown type
         */
        @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
        private fun getFilterPayloadType(filterString: String): Int {
            when {
                //example - cell_phone[].brand array of json object
                Pattern.matches("^[^.]+\\[]\\.[^.]+\$", filterString) -> {
                    return FILTER_PAYLOAD_TYPE_ARRAY_OF_JSON_OBJECT
                }

                //example - country[]  json array of strings or int
                Pattern.matches("^[^.]+\\[]\$", filterString) -> {
                    return FILTER_PAYLOAD_TYPE_ARRAY_OF_STRINGS_OR_INT
                }
                //example - students.first_name  nested json object
                Pattern.matches("^[^.]+\\.[^.]+\$", filterString) -> {
                    return FILTER_PAYLOAD_TYPE_NESTED_OBJECT
                }
                //example - age normal json object
                Pattern.matches("^[^.]+\$", filterString) -> {
                    return FILTER_PAYLOAD_TYPE_NORMAL_OBJECT
                }
            }
            return FILTER_PAYLOAD_TYPE_UNKNOWN
        }


        /**
         * Method it get the main key from the filter string
         * example s^students.first_name -> students is main key and first_name is the secondary key in the payload to look for
         */
        @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
        private fun getFilterKey(filterString: String, filterPayloadType: Int, isPrimaryKey: Boolean = true): String? {
            var filterKey: String? = null
            when (filterPayloadType) {
                FILTER_PAYLOAD_TYPE_ARRAY_OF_STRINGS_OR_INT -> {
                    if (isPrimaryKey) {
                        filterKey = filterString.substring(0, filterString.indexOf("["))
                    }
                }
                FILTER_PAYLOAD_TYPE_ARRAY_OF_JSON_OBJECT -> {
                    filterKey = if (isPrimaryKey) {
                        filterString.substring(filterString.indexOf("^") + 1, filterString.indexOf("["))
                    } else {
                        filterString.substring(filterString.indexOf(".") + 1)
                    }
                }
                FILTER_PAYLOAD_TYPE_NESTED_OBJECT -> {
                    filterKey = if (isPrimaryKey) {
                        filterString.substring(0, filterString.indexOf("."))
                    } else {
                        filterString.substring(filterString.indexOf(".") + 1)

                    }
                }
                FILTER_PAYLOAD_TYPE_NORMAL_OBJECT -> {
                    filterKey = filterString
                }
            }
            return filterKey
        }


        /**
         * Function to compare the integer operator with given conditions
         */
        @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
        private fun validateFilterIntRules(payLoadValue: String?, filter: SMTInAppRule.Filters): Boolean {

            var isShow = false
            if (payLoadValue != null) {
                val flValue = filter.value.trim().toLowerCase(Locale.getDefault())
                when (filter.operator.trim()) {
                    "==" -> isShow = Float.parseFloat(payLoadValue) == Float.parseFloat(flValue)
                    "!=" -> isShow = Float.parseFloat(payLoadValue) != Float.parseFloat(flValue)
                    "<" -> isShow = Float.parseFloat(payLoadValue) < Float.parseFloat(flValue)
                    ">" -> isShow = Float.parseFloat(payLoadValue) > Float.parseFloat(flValue)
                    "<=" -> isShow = Float.parseFloat(payLoadValue) <= Float.parseFloat(flValue)
                    ">=" -> isShow = Float.parseFloat(payLoadValue) >= Float.parseFloat(flValue)
                }
            }
            return isShow
        }

        /**
         * Function to compare the payloadvalue with the filter value using the string operator
         */
        @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
        private fun validateFilterStringRules(payLoadValue: String?, filter: SMTInAppRule.Filters): Boolean {

            var isShowInApp = false
            if (payLoadValue != null) {
                val plValue = payLoadValue.toLowerCase(Locale.getDefault())
                val flValue = filter.value.toLowerCase(Locale.getDefault())
                when (filter.operator.trim()) {
                    "is" -> isShowInApp = plValue == flValue
                    "contains" -> isShowInApp = plValue.contains(flValue)
                    "doesNotContains" -> isShowInApp = !plValue.contains(flValue)
                    "startsWith" -> isShowInApp = plValue.startsWith(flValue)
                    "endsWith" -> isShowInApp = plValue.endsWith(flValue)
                }
            }
            return isShowInApp
        }


        /**
         * Method to check the filters against the eventPayload
         * @param filters ArrayList
         * @param filterType all or any
         * @param eventPayLoad Payload of an event
         * @return true or false
         * Description:  Method validates the given filter against the event payload. If event matches then returns true otherwise false
         *
         */
        internal fun checkRuleFilter(filters: ArrayList<SMTInAppRule.Filters>?, filterType: String, eventPayLoad: HashMap<String, Any>): Boolean {
            //Convert payload keys into lowercase
            val objPayloadWithLowerKeys = SMTCommonUtility.jsonKeyCaseConverter(JSONObject(eventPayLoad), true)
            val modifiedEventPayLoad = SMTCommonUtility.jsonToHashMap(objPayloadWithLowerKeys)
            //println("inapp modifiedEventPayLoad: $modifiedEventPayLoad")

            var isShowInApp = true
            try {
                if (filters != null && filters.size > 0) {
                    var isShow = false
                    for (i in 0 until filters.size) {

                        //Get the filter key from the InAppRule
                        val filterString = filters[i].key
                        val filterPayloadType = getFilterPayloadType(filterString)
                        val isStringOperator = filters[i].datatype.equals("string", ignoreCase = true)
                        //println("inapp :filterPayloadType $filterPayloadType datatype ${filters[i].datatype} isStringOperator $isStringOperator")
                        //get payload from the custom event
                        val payLoad: String? = modifiedEventPayLoad[SMTEventParamKeys.SMT_PAYLOAD].toString()
                        //println("inapp :payLoad $payLoad")
                        if (!payLoad.isNullOrEmpty()) {
                            val payload = JSONObject(payLoad.trim())
                            when (filterPayloadType) {
                                FILTER_PAYLOAD_TYPE_ARRAY_OF_STRINGS_OR_INT -> {
                                    val filterPrimaryKey = getFilterKey(filterString, FILTER_PAYLOAD_TYPE_ARRAY_OF_STRINGS_OR_INT, true)
                                    if (!filterPrimaryKey.isNullOrEmpty()) {
                                        val payLoadArray = payload.optJSONArray(filterPrimaryKey)
                                        isShow = checkFilterAgainstPayloadArray(payLoadArray, filters[i], isStringOperator)
                                    }
                                }
                                FILTER_PAYLOAD_TYPE_ARRAY_OF_JSON_OBJECT -> {
                                    val filterPrimaryKey = getFilterKey(filterString, FILTER_PAYLOAD_TYPE_ARRAY_OF_JSON_OBJECT, true)
                                    if (!filterPrimaryKey.isNullOrEmpty()) {
                                        val payLoadJsonArray = payload.optJSONArray(filterPrimaryKey)
                                        val filterSecondaryKey = getFilterKey(filterString, FILTER_PAYLOAD_TYPE_ARRAY_OF_JSON_OBJECT, false)

                                        isShow = checkFilterAgainstPayloadJsonArray(payLoadJsonArray, filters[i], isStringOperator, filterSecondaryKey)
                                    }
                                }
                                FILTER_PAYLOAD_TYPE_NESTED_OBJECT -> {
                                    val filterPrimaryKey = getFilterKey(filterString, FILTER_PAYLOAD_TYPE_NESTED_OBJECT, true)
                                    if (!filterPrimaryKey.isNullOrEmpty()) {
                                        val payLoadJsonObject = payload.optJSONObject(filterPrimaryKey)
                                        if (payLoadJsonObject != null) {
                                            val filterSecondaryKey = getFilterKey(filterString, FILTER_PAYLOAD_TYPE_NESTED_OBJECT, false)
                                            filterSecondaryKey?.let {
                                                val payLoadValue = payLoadJsonObject.opt(it).toString()
                                                isShow = checkIfThePayLoadValueMatchesFilterValue(payLoadValue, filters[i], isStringOperator)

                                            }
                                        }
                                    }

                                }
                                FILTER_PAYLOAD_TYPE_NORMAL_OBJECT -> {
                                    val filterPrimaryKey = getFilterKey(filterString, FILTER_PAYLOAD_TYPE_NORMAL_OBJECT, true)
                                    if (!filterPrimaryKey.isNullOrEmpty()) {
                                        //println("inapp : $filterString --- $filterPrimaryKey")
                                        val payLoadValue = payload.optString(filterPrimaryKey)
                                        isShow = checkIfThePayLoadValueMatchesFilterValue(payLoadValue, filters[i], isStringOperator)

                                    }
                                }
                                else -> // it is unknown payload set false
                                    isShow = false
                            }
                        }

                        if (filterType.equals("all", ignoreCase = true)) {
                            //if all of the filter validation fails then override the default value to false
                            if (!isShow) {
                                isShowInApp = isShow
                                break
                            }
                        } else if (filterType.equals("any", ignoreCase = true)) {
                            //if any filter is validate true then break the loop and then return
                            if (isShow) {
                                isShowInApp = isShow
                                break
                            }
                        }
                    }
                }
            } catch (e: Exception) {
                SMTLogger.e(TAG, "Json parsing error: " + e.localizedMessage)
                isShowInApp = false
            }
            SMTLogger.i(TAG, "InApp 1: $isShowInApp")
            //println("inapp checkRuleFilter: $isShowInApp")
            return isShowInApp
        }


        /**
         * Method to loop through array of json object to check if the filter value matches
         */
        @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
        private fun checkFilterAgainstPayloadJsonArray(payLoadJsonArray: JSONArray?, filter: SMTInAppRule.Filters, stringOperator: Boolean, key: String?): Boolean {
            var isShowInApp = false
            if (payLoadJsonArray != null && key != null) {
                for (i in 0 until payLoadJsonArray.length()) {
                    val isShow = checkIfThePayLoadValueMatchesFilterValue((payLoadJsonArray[i] as JSONObject).get(key).toString(), filter, stringOperator)
                    if (isShow) {
                        isShowInApp = isShow

                        break
                    }
                }
            }
            return isShowInApp

        }

        /**
         * method to loop through the array of int/string to check if the filter value matches
         */
        @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
        private fun checkFilterAgainstPayloadArray(payLoadArray: JSONArray?, filter: SMTInAppRule.Filters, stringOperator: Boolean): Boolean {
            var isShowInApp = false
            if (payLoadArray != null) {
                for (i in 0 until payLoadArray.length()) {
                    val isShow = checkIfThePayLoadValueMatchesFilterValue(payLoadArray[i].toString(), filter, stringOperator)
                    if (isShow) {
                        isShowInApp = isShow
                        break
                    }
                }
            }
            return isShowInApp
        }


        /**
         * function to compare the payload value against the filter value
         */
        @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
        private fun checkIfThePayLoadValueMatchesFilterValue(payLoadValue: String, filter: SMTInAppRule.Filters, isStringOperator: Boolean): Boolean {
            return if (isStringOperator) {
                validateFilterStringRules(payLoadValue.trim(), filter)
            } else {
                validateFilterIntRules(payLoadValue.trim(), filter)
            }

        }

        /**
         * Check if the rule is applicable now
         */
        internal fun isApplicableNow(inAppRule: SMTInAppRule, date: Date): Boolean {
//            var dayOfWeek = SimpleDateFormat("u", Locale.getDefault()).format(date)
            var dayOfWeek = Calendar.getInstance().get(Calendar.DAY_OF_WEEK)
            dayOfWeek -= 1
            //println("inapp : isApplicableNow $dayOfWeek")

            var isApplicableNow = false
            if (inAppRule.whenTo.days.size > 0) {
                isApplicableNow = inAppRule.whenTo.days.contains(dayOfWeek.toString())
                if (isApplicableNow) {
                    isApplicableNow = isCurrentTimeWithinTheRange(inAppRule, date)
                }
            }
            return isApplicableNow
        }

        /**
         * Method to check if the current time is within the bounds of time range given
         */
        internal fun isCurrentTimeWithinTheRange(inAppRule: SMTInAppRule, date: Date): Boolean {
            if (inAppRule.whenTo.time.size > 0) {
                if (inAppRule.whenTo.time.size == 1) {
                    val curTime = SimpleDateFormat("HH:mm", Locale.getDefault()).format(date)
                    val currentTime = SimpleDateFormat("HH:mm", Locale.getDefault()).parse(curTime)
                    val toTime = SimpleDateFormat("HH:mm", Locale.getDefault()).parse(inAppRule.whenTo.time[0].to)
                    val fromTime = SimpleDateFormat("HH:mm", Locale.getDefault()).parse(inAppRule.whenTo.time[0].from)

                    //println("inapp time $curTime")
                    //println("inapp time $currentTime")
                    //println("inapp time $toTime")
                    //println("inapp time $fromTime")
                    //println("inapp time ${toTime.after(currentTime)}")
                    //println("inapp time ${fromTime.before(currentTime)}")
                    return (toTime.after(currentTime) && fromTime.before(currentTime))
                }
            }
            return true
        }

        internal fun getTodayInMilliSec(): Long {
//            return SimpleDateFormat("dd mm yyyy", Locale.getDefault()).parse(SimpleDateFormat("dd mm yyyy", Locale.getDefault()).format(Date())).time
            val calendar = Calendar.getInstance()
            calendar.set(Calendar.HOUR_OF_DAY, 0)
            calendar.set(Calendar.MINUTE, 0)
            calendar.set(Calendar.SECOND, 0)
            calendar.set(Calendar.MILLISECOND, 0)
            return calendar.time.time
        }

        /**
         * Method to convert the time given in yyyy-mm-dd'T'HH:mm:ss format to timestamp
         */
        internal fun getTimeInMilliSec(time: String): String {
            val simpleDateFormat = SimpleDateFormat(SMTConfigConstants.SERVER_TIME_FORMAT, Locale.getDefault())
            simpleDateFormat.timeZone = TimeZone.getTimeZone("UTC")
            return simpleDateFormat.parse(time).time.toString()
        }

        /**
         * Method to convert the time given in yyyy-mm-dd HH:mm:ss format to timestamp
         */
        internal fun getTimeInMilliSecForModifiedDate(time: String): String {
            return try {
                val simpleDateFormat = SimpleDateFormat(SMTConfigConstants.SERVER_TIME_FORMAT, Locale.getDefault())
                simpleDateFormat.timeZone = TimeZone.getTimeZone("UTC")
                simpleDateFormat.parse(time).time.toString()
            } catch (e: Exception) {
                ""
            }
        }

        internal fun isInAppEvent(eventId: Int): Boolean {
            var isInAppEvent = false
            when (eventId) {
                SMTEventId.EVENT_INAPP_DISMISSED,
                SMTEventId.EVENT_INAPP_VIEWED,
                SMTEventId.EVENT_INAPP_CLICKED -> {
                    isInAppEvent = true
                }
            }
            return isInAppEvent
        }


        /**
         * Custom HTML listener for providing control to user.
         * @param inAppCustomHTMLListener InAppCustomHTMLListener
         * check payload data for valid json and set payload data to listener
         * @param payload
         */
        internal fun processInAppCustomHtmlData(inAppCustomHTMLListener: InAppCustomHTMLListener?, payload: String?) {
            try {
                if (payload != null) {
                    val objPayload = JSONObject(payload)
                    inAppCustomHTMLListener?.let { listener ->
                        {
                            objPayload.let {
                                listener.customHTMLCallback(SMTCommonUtility.jsonToMap(it))
                            }
                        }
                    }
                }
            } catch (e: java.lang.Exception) {
                SMTLogger.e(TAG, "Netcore Error: " + e.message)
            }
        }

        /**
         * Visitor can be new /old / all and type can be identified/ anon / all
         */
        internal fun checkTheVisitor(inAppRule: SMTInAppRule): Boolean {
            if (inAppRule.whomTo.visitor == SMTInAppConst.VALUE_VISITOR && inAppRule.whomTo.visitorType == SMTInAppConst.VALUE_VISITOR) {
                return true
            }
            return false
        }

        internal fun checkListAndSegmentIds(inAppRule: SMTInAppRule, listIds: List<String>, segIds: List<String>): Boolean {
            //println("inapp : checkListAndSegmentIds ${inAppRule.whomTo.listIds}---${inAppRule.whomTo.segIds}")
            //println("inapp : checkListAndSegmentIds url $listIds---$segIds")

            var isShowInApp = true
            val listIdSize = inAppRule.whomTo.listIds.size
            val segIdSize = inAppRule.whomTo.segIds.size
            try {
                if (listIdSize > 0 || segIdSize > 0) {
                    val isListIdPresent = SMTCommonUtility.compareLists(listIds, inAppRule.whomTo.listIds)
                    if (!isListIdPresent) {
                        isShowInApp = SMTCommonUtility.compareLists(segIds, inAppRule.whomTo.segIds)
                    }
                }
            } catch (e: Exception) {
                SMTLogger.e(TAG, e.message.toString())
            }
            SMTLogger.i(TAG, "InApp 3: $isShowInApp")
            return isShowInApp
        }

        internal fun checkMultiEvents(context: Context, inAppRule: SMTInAppRule): Boolean {
            var isShowInApp = true
            try {
                val multiEventsRules = inAppRule.whomTo.events.rules
                val targetRule = inAppRule.whomTo.events.targetRule

                if (multiEventsRules.size > 0) {
                    var isShow = false
                    var smtEventList: MutableList<SMTEvents>? = mutableListOf()
                    for (i in 0 until multiEventsRules.size) {
                        val multiEventsRule = multiEventsRules[i]

                        // storing previous event time beacuse next event depend on it.
                        var previousEventTime = "0"
                        if (smtEventList != null && smtEventList.size > 0 &&
                                targetRule.equals(SMTInAppConst.VALUE_TARGET_RULE_ALL, ignoreCase = true)) {
                            previousEventTime = smtEventList[0].eventDate
                        }
                        multiEventsRule.eventTime = previousEventTime
                        val eventPayLoad: HashMap<String, Any> = SMTEventPayloadCreator.createMultiEventPayload(multiEventsRule)

                        // Check eventname present in event list table
                        smtEventList = SMTDataBaseService.getInstance(WeakReference(context.applicationContext)).getStoredEventData(eventPayLoad)

                        val performed = multiEventsRule.performed
                        if (smtEventList != null) {

                            val filters = multiEventsRule.filters

                            if (performed == "yes" && smtEventList.size > 0) {
                                /**
                                 * If event perform yes then it should present in db if its not then return false.
                                 * If found in db then go for filter
                                 * if filter is not present then directly return true otherwise check appropriate filter and return true/false
                                 */
                                val smtEvent = smtEventList[0]
                                //println("inapp : ${smtEvent.eventId} ---- ${smtEvent.eventName}")
                                //println("inapp : ${smtEvent.eventPayload}")
                                //println("inapp : ${smtEvent.eventDate}")
                                isShow = if (filters.size > 0) {
                                    checkRuleFilter(filters, multiEventsRule.filterType,
                                            SMTCommonUtility.jsonToHashMap(JSONObject(smtEvent.eventPayload)))
                                } else {
                                    true
                                }
                            } else if (performed == "no") {
                                /**
                                 * If event perform no and there is no filter then it should not present in db. (found in db return false)
                                 * if there is filter and matched with that filter then return false. (return true if not matched with filter)
                                 * if above cases are valid then go for wait time.
                                 */
                                isShow = if (smtEventList.size > 0 && filters.size > 0) {
                                    val smtEvent = smtEventList[0]
                                    //println("inapp : ${smtEvent.eventId} ---- ${smtEvent.eventName}")
                                    //println("inapp : ${smtEvent.eventPayload}")
                                    //println("inapp : ${smtEvent.eventDate}")
                                    val isFilterValid = checkRuleFilter(filters, multiEventsRule.filterType,
                                            SMTCommonUtility.jsonToHashMap(JSONObject(smtEvent.eventPayload)))
                                    //println("inapp : $isFilterValid")
                                    /**
                                     *  Here we are converting true to false because of perform no condition.
                                     *  because the condition is valid if event is not matches with filter.
                                     */

                                    !isFilterValid

                                } else (smtEventList.size == 0)

                                if (isShow) {
                                    //currenttime - previoustime > waituntill show directly or
                                    // wait for remaining time check in db event happn or not if no then show
                                    val waitUntil: Long = multiEventsRule.waitUntil
                                    if (waitUntil > 0) {
                                        val diffInEventTime = System.currentTimeMillis() - previousEventTime.toLong()
                                        val waitTimeInMilliSec = TimeUnit.SECONDS.toMillis(waitUntil)

                                        //println("inapp waitUntil : $waitUntil")
                                        //println("inapp prr : $previousEventTime")
                                        //println("inapp curr : ${System.currentTimeMillis()}")
                                        //println("inapp difff : $diffInEventTime")
                                        //println("inapp curr : ${TimeUnit.SECONDS.toMillis(waitUntil)}")
                                        //println("inapp wait time : ${waitTimeInMilliSec - diffInEventTime}")

                                        if (waitTimeInMilliSec > diffInEventTime) {
                                            SMTPreferenceHelper.getAppPreferenceInstance(context, null)
                                                    .setLong(SMTPreferenceConstants.SMT_INAPP_WAIT_TIME, waitTimeInMilliSec - diffInEventTime)
                                            SMTPreferenceHelper.getAppPreferenceInstance(context, null)
                                                    .setInt(SMTPreferenceConstants.SMT_INAPP_ME_POS, i)
                                        } else {
                                            SMTPreferenceHelper.getAppPreferenceInstance(context, null)
                                                    .setLong(SMTPreferenceConstants.SMT_INAPP_WAIT_TIME, 0)
                                        }
                                    }
                                }
                            } else {
                                isShow = false
                            }
                        }
                        //println("$i inapp : $isShow")
                        if (targetRule.equals(SMTInAppConst.VALUE_TARGET_RULE_ALL, ignoreCase = true)) {
                            //if all of the filter validation fails then override the default value to false
                            if (!isShow) {
                                break
                            }
                        } else if (targetRule.equals(SMTInAppConst.VALUE_TARGET_RULE_ANY, ignoreCase = true)) {
                            //if any filter is validate true then break the loop and then return
                            if (isShow) {
                                break
                            }
                        }
                    }
                    isShowInApp = isShow
                }
            } catch (e: Exception) {
                SMTLogger.e(TAG, e.message.toString())
            }
            SMTLogger.i(TAG, "InApp 4: $isShowInApp")
            //println("inapp checkMultiEvents: $isShowInApp")
            return isShowInApp
        }
    }
}