package com.netcore.android.inbox.views

import android.content.Context
import android.util.AttributeSet
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.LinearLayout
import androidx.appcompat.widget.AppCompatTextView
import com.netcore.android.R
import com.netcore.android.notification.models.SMTActionButtonData
import com.netcore.android.notification.models.SMTNotificationData
import com.netcore.android.utility.SMTCommonUtility

/**
 * Copyright © 2019 Netcore. All rights reserved.
 *
 * Inbox Simple Message Custom View - Handles showing simple message
 * It record inbox events as well
 *
 * @author Netcore
 * @version 1.0
 * @since 29-04-2019
 */

internal class SMTInboxSimpleMessageView(context: Context, attrs: AttributeSet?) : SMTBaseView(context, attrs) {

    private lateinit var mView: View

    init {
        initLayout(context)
    }

    private fun initLayout(context: Context) {
        mView = LayoutInflater.from(context).inflate(R.layout.inbox_simple_message_layout, this)

        val lp = LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.WRAP_CONTENT)

        this.layoutParams = lp

        this.orientation = LinearLayout.VERTICAL
    }

    override fun setNotificationData(notification: SMTNotificationData) {
        super.setData(notification)

        setViewDetail()
        setActionButtons()

        setLayoutAction()

        setBackgroundShape()
    }

    private fun setViewDetail() {
        mView.findViewById<AppCompatTextView>(R.id.tv_simple_title).text = mNotificationData.mTitle
        mView.findViewById<AppCompatTextView>(R.id.tv_simple_timestamp).text = SMTCommonUtility.getFormattedTimeDifference(mNotificationData.mPublishedTimeStamp)
        mView.findViewById<AppCompatTextView>(R.id.tv_simple_message).text = mNotificationData.mMessage
    }


    /**
     * Set action listener to the layout
     */
    private fun setLayoutAction() {
        mView.setOnClickListener {
            handleClick(mNotificationData.mDeepLinkPath)
        }
    }

    /**
     * set action button and click listener if there is any action button
     */
    private fun setActionButtons() {
        if (mNotificationData.mActionButtonList?.isNotEmpty() == true) {
            mView.findViewById<LinearLayout>(R.id.actionLayout).visibility = View.VISIBLE
            hideAllActionButtons()
            mNotificationData.mActionButtonList?.let { actionList ->
                var count = 0
                actionList.forEach { buttonData ->
                    ++count
                    when (count) {
                        1 -> setDataToActionButton(R.id.tv_simple_left_action, buttonData)
                        2 -> setDataToActionButton(R.id.tv_simple_middle_action, buttonData)
                        3 -> setDataToActionButton(R.id.tv_simple_right_action, buttonData)
                    }
                }
            }
        } else {
            mView.findViewById<LinearLayout>(R.id.actionLayout).visibility = View.GONE
        }
    }

    private fun hideAllActionButtons() {
        mView.findViewById<AppCompatTextView>(R.id.tv_simple_left_action).visibility = View.INVISIBLE
        mView.findViewById<AppCompatTextView>(R.id.tv_simple_middle_action).visibility = View.INVISIBLE
        mView.findViewById<AppCompatTextView>(R.id.tv_simple_right_action).visibility = View.INVISIBLE
    }

    /**
     * Set the data to Action Buttons
     * @param viewid view id to which action to be set
     * @param buttonData data related to the action button
     */
    private fun setDataToActionButton(viewid: Int, buttonData: SMTActionButtonData) {
        val action = mView.findViewById<AppCompatTextView>(viewid)
        action?.let {
            action.text = buttonData.actionName
            action.visibility = View.VISIBLE
            action.setOnClickListener {
                handleClick(buttonData.actionDeeplink)
            }
        }
    }

}
