package com.netcore.android.inbox.views

import android.content.Context
import android.util.AttributeSet
import android.widget.LinearLayout
import android.widget.Toast
import com.netcore.android.R
import com.netcore.android.db.SMTAppInboxTable
import com.netcore.android.event.SMTEventRecorder
import com.netcore.android.inbox.utility.SMTInboxMessageStatus
import com.netcore.android.notification.models.SMTNotificationData
import com.netcore.android.utility.SMTCommonUtility

internal abstract class SMTBaseView(context: Context, attrs: AttributeSet?) : LinearLayout(context, attrs) {

    internal lateinit var mNotificationData: SMTNotificationData

    /**
     * Sets card background based on message Read status
     * If message is read then grey else white background
     */
    protected fun setBackgroundShape() {
        if (mNotificationData.mStatus == SMTAppInboxTable.STATUS_DELIVERED) {
            setBackgroundResource(R.drawable.grey_rounded_color)
        } else {
            setBackgroundResource(R.drawable.white_rounded_corner)
        }
    }

    /**
     * Broadcasts the item clicked and clear the existing carousel
     * @param itemLink deeplink path
     */
    private fun sendItemClickedBroadcast(itemLink: String) {
        SMTCommonUtility.handleNotificationClick(context, itemLink, mNotificationData.mCustomPayload)
    }

    private fun recordClickEvent(deepLinkPath: String?) {
        SMTEventRecorder.getInstance(context).recordInboxEvent(this.mNotificationData.mTrid, deepLinkPath, SMTInboxMessageStatus.Status.READ)
    }

    protected fun setData(notification: SMTNotificationData) {
        mNotificationData = notification
    }

    protected fun showError(error: String) {

        Toast.makeText(context, error, Toast.LENGTH_LONG).show()
    }

    /**
     * Handles layout click
     * changes color of layout background
     * handles deeplink if any
     * records the event and also updates the DB
     */
    protected fun handleClick(deepLinkPath: String?) {

        mNotificationData.mStatus = SMTAppInboxTable.STATUS_CLICKED

        setBackgroundShape()

        sendItemClickedBroadcast(deepLinkPath ?: "")

        recordClickEvent(deepLinkPath)
    }

    abstract fun setNotificationData(notification: SMTNotificationData)
}