package com.netcore.android.workmgr

import android.content.Context
import androidx.work.*
import com.netcore.android.SMTConfigConstants
import com.netcore.android.SMTWorkManagerConst
import com.netcore.android.logger.SMTLogger
import com.netcore.android.preference.SMTPreferenceConstants
import com.netcore.android.preference.SMTPreferenceHelper
import com.netcore.android.utility.SMTCommonUtility
import java.util.concurrent.TimeUnit

/**
 *  This class will manage all the worker running in SDK.
 *
 *  @author Netcore
 *
 *  @Description This class will provide all the methods required to start, cancel and check status of worker
 *  running inside the SDK.
 *  There are 3 workers.
 *  1. EventSync worker: This worker is responsible for syncing pending events to server.
 *  It will also sync Notification events realtime.
 *
 *  2. Background Worker: This worker will sync all Pending and Failed events in background and
 *  it wll run in background and before running it will check EventSync worker is not running to
 *  keep consistency with the events batch
 *
 *  3. InProgressEventWorker: This worker will change the status of all events whose status is InProgress and
 *  present in DB for more then 2 days. It's required to make sure that we retry sync of all events who failed because
 *  of applicaton killed without notifying the worker.
 *
 */
internal class SMTWorkerScheduler private constructor() {

    val TAG: String = SMTWorkerScheduler::class.java.simpleName

    companion object {
        @Volatile
        private var INSTANCE: SMTWorkerScheduler? = null

        fun getInstance(): SMTWorkerScheduler = INSTANCE
                ?: synchronized(SMTWorkerScheduler::class.java) {
                    INSTANCE ?: buildInstance().also { INSTANCE = it }
                }

        private fun buildInstance(): SMTWorkerScheduler {
            return SMTWorkerScheduler()
        }
    }

    /**
     * Check if scheduled worker is running for the given tag
     * @param workManager Instance of WorkManager
     * @param tag Worker Tag
     * @return isRunning Boolean to tell whether Job is running or not
     */
    fun isWorkerRunning(workManager: WorkManager, tag: String): Boolean {
        var isRunning = false
        val future = workManager.getWorkInfosByTag(tag)
        val list = future.get()
        list.forEach {
            if (it.state == WorkInfo.State.RUNNING) {
                isRunning = true
            }
        }
        return isRunning
    }

    /**
     * Check if scheduled worker is running for the given tag
     * @param context context
     * @return isRunning Boolean to tell whether Job is running or not
     */
    fun isEventWorkerRunning(context: Context): Boolean {
        val workManager = WorkManager.getInstance(context)
        var isRunning = false
        val future = workManager.getWorkInfosByTag(SMTWorkManagerConst.SMT_EVENTSYNC_WORKER_TAG)
        val list = future.get()
        list.forEach {
            if (it.state == WorkInfo.State.RUNNING) {
                isRunning = true
            }
        }
        return isRunning
    }

    /**
     *  This worker will be used to start EventSync worker
     */
    fun scheduleEventWorker(context: Context) {
        val request = OneTimeWorkRequestBuilder<EventSyncWorker>().addTag(SMTWorkManagerConst.SMT_EVENTSYNC_WORKER_TAG).build()
        WorkManager.getInstance(context).enqueueUniqueWork(SMTWorkManagerConst.SMT_EVENTSYNC_WORKER_TAG, ExistingWorkPolicy.KEEP, request)
    }

    /**
     *  This method will be return the status of EventSyncWorker
     *  If worker is not running then start the worker to sync
     *  pending events.
     */
    fun checkStatusAndScheduleEventWorker(context: Context) {
        val status = isWorkerRunning(WorkManager.getInstance(context), SMTWorkManagerConst.SMT_EVENTSYNC_WORKER_TAG)
        SMTLogger.internal(TAG, "Event worker status : $status")
        if (!status) {
            scheduleEventWorker(context)
        }
    }

    /**
     *  This method will schedule InProgress SyncWorker
     */
    fun scheduleInProgressEventWorker(context: Context) {
        val isTrackingAllowed = SMTCommonUtility.checkIfTrackingAllowed(context)
        if (isTrackingAllowed) {
            val request = OneTimeWorkRequestBuilder<InProgressEventWorker>().addTag(SMTWorkManagerConst.SMT_INPROGRESS_WORKER_TAG).build()
            WorkManager.getInstance(context).enqueueUniqueWork(SMTWorkManagerConst.SMT_INPROGRESS_WORKER_TAG, ExistingWorkPolicy.KEEP, request)
        }
    }

    /**
     *  This method will schedule BackgroundSyncWorker
     */
    fun scheduleBackgroundSyncWorker(context: Context) {
        val request = PeriodicWorkRequestBuilder<BackgroundSyncWorker>(15, TimeUnit.MINUTES).addTag(SMTWorkManagerConst.SMT_BACKGROUND_WORKER_TAG).build()
        WorkManager.getInstance(context).enqueueUniquePeriodicWork(SMTWorkManagerConst.SMT_BACKGROUND_WORKER_TAG, ExistingPeriodicWorkPolicy.REPLACE, request)
    }

    /**
     *  This method will be used to schedule PushAmp worker and will
     *  set constraint required by the worker
     *  @param context Context
     */
    fun schedulePushAmp(context: Context) {
        val interval = getPushAmpInterval(context)
        val request = PeriodicWorkRequestBuilder<SMTPushAmpWorker>(interval, TimeUnit.MINUTES)
                .addTag(SMTWorkManagerConst.SMT_PUSHAMP_WORKER_TAG)
                .build()
        WorkManager.getInstance(context).enqueueUniquePeriodicWork(SMTWorkManagerConst.SMT_PUSHAMP_WORKER_TAG, ExistingPeriodicWorkPolicy.REPLACE, request)
    }

    private fun getPushAmpInterval(context: Context): Long {
        return SMTPreferenceHelper.getAppPreferenceInstance(context, null)
                .getInt(SMTPreferenceConstants.PUSH_AMP_INTERVAL, SMTConfigConstants.DEFAULT_PUSH_AMP_INTERVAL).toLong()
    }

    /**
     *  This method will cancel PushAmp worker running in background
     *  @param ctx Context
     */
    fun cancelPushAmp(ctx: Context) {
        SMTLogger.e(TAG, "PushAmp worker cancelled")
        WorkManager.getInstance(ctx).cancelAllWorkByTag(SMTWorkManagerConst.SMT_PUSHAMP_WORKER_TAG)
    }

    /**
     *  This method will cancel BackgroundSync worker running in background
     *  @param ctx Context
     */
    fun cancelBackgroundSyncWorker(ctx: Context) {
        WorkManager.getInstance(ctx).cancelAllWorkByTag(SMTWorkManagerConst.SMT_BACKGROUND_WORKER_TAG)
    }

    /**
     *  This method will cancel BackgroundSync worker running in background
     *  @param ctx Context
     */
    @Suppress("unused")
    fun cancelInProressSyncWorker(ctx: Context) {
        WorkManager.getInstance(ctx).cancelAllWorkByTag(SMTWorkManagerConst.SMT_INPROGRESS_WORKER_TAG)
    }

    /**
     *  This method will check all conditions before running pushamp worker.
     *  @param context App context
     *  @return boolean
     */
    fun arePushampConditionsSatisfied(context: Context): Boolean {
        val isPushAmpEnabled = SMTPreferenceHelper.getAppPreferenceInstance(context, null)
                .getBoolean(SMTPreferenceConstants.IS_PUSH_AMP_ENABLED)
        val areNotificationEnabled = SMTCommonUtility.areNotificationsEnabled(context)
        val isUserPNOptedIn = SMTPreferenceHelper.getAppPreferenceInstance(context, null).getBoolean(SMTPreferenceConstants.OPT_IN_OUT_PUSH_NOTIFICATION, true)
        val isSDKPanelActive = SMTCommonUtility.checkPanelAndSDKActiveStatus(context)
        return (isPushAmpEnabled && areNotificationEnabled && isUserPNOptedIn && isSDKPanelActive)
    }
}