package com.netcore.android.notification.audio

import android.app.Service
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.media.MediaPlayer
import android.os.Build
import android.os.IBinder
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import com.netcore.android.SMTConfigConstants
import com.netcore.android.logger.SMTLogger
import com.netcore.android.notification.SMTNotificationConstants
import com.netcore.android.notification.models.SMTNotificationData
import com.netcore.android.utility.SMTCommonUtility
import java.io.File

internal class SMTMusicService : Service(), MediaPlayer.OnCompletionListener, MediaPlayer.OnPreparedListener {

    private var mediaPlayer: MediaPlayer? = null
    private var mCurrentAudioParcel: SMTNotificationData? = null

    companion object {
        const val ACTION_START = "ACTION_START"
        const val ACTION_INIT = "ACTION_INIT"
        const val ACTION_PAUSE = "ACTION_PAUSE"
        const val ACTION_COMPLETED = "ACTION_COMPLETED"
        const val ACTION_STOP = "ACTION_STOP"
        const val ACTION_NO_MUSIC = "ACTION_NO_MUSIC"
        const val ACTION_PLAY = "ACTION_PLAY"
        const val ACTION_PLAY_PAUSE = "ACTION_PLAY_PAUSE"
        const val ACTION_UPDATE = "ACTION_UPDATE"
        const val ACTION_CLOSE = "ACTION_CLOSE"
    }

    override fun onCreate() {
        super.onCreate()

//        displayNotificaiton(mCurrentAudioParcel)

        registerLocalBroadcast()

    }

    private fun registerLocalBroadcast() {
        val intentFilter = IntentFilter(SMTConfigConstants.BROADCAST_EVENT_AUDIO_DISMISS)
        LocalBroadcastManager.getInstance(this).registerReceiver(mMessageReceiver, intentFilter)
    }

    // Our handler for received Intents. This will be called whenever an Intent
    // with an action named "custom-event-name" is broadcasted.
    private val mMessageReceiver = object : BroadcastReceiver() {
        override fun onReceive(context: Context, intent: Intent) {
            // Get extra data included in the Intent

            SMTLogger.d("receiver", "Got message:")
            intent.extras?.let {
                SMTRichAudioPNGenerator.getInstance().dismissNotification(it, context, false)
            }
        }
    }

    private fun displayNotificaiton(audioParcel: SMTNotificationData?) {
        startForeground(audioParcel?.notificationId!!,
                SMTRichAudioPNGenerator.getInstance().showNotification(this, audioParcel, false))
    }

    override fun onBind(intent: Intent?): IBinder? {
        return null
    }

    override fun onCompletion(mp: MediaPlayer?) {
        pauseMusic()
        // In case user wants to restart it again
        mediaPlayer?.seekTo(0)
        mCurrentAudioParcel?.let {
            it.isPlaying = false
            it.action = ACTION_COMPLETED
            displayNotificaiton(it)
        }
    }

    override fun onPrepared(mp: MediaPlayer?) {
        playMusic()
    }

    fun playMusic() {
        startMusic()
//        notifyCallerToShowNotification()
    }

    fun startMusic() {
        mediaPlayer?.start()
    }

    fun pauseMusic() {
        mediaPlayer?.pause()
    }

    override fun onStartCommand(intent: Intent?, flags: Int, startId: Int): Int {
        intent?.let {
            var newAudioParcel: SMTNotificationData? = null
            it.extras?.let { extras ->
                newAudioParcel = if (extras.containsKey(SMTNotificationConstants.NOTIFICATION_PARCEL)) {
                    extras.getParcelable(SMTNotificationConstants.NOTIFICATION_PARCEL)
                } else {
                    null
                }
            }
            newAudioParcel?.let {
                if (it.notificationId != mCurrentAudioParcel?.notificationId || mCurrentAudioParcel == null) {

                    var changeMusic = true
                    if (it.action == ACTION_CLOSE || it.action == ACTION_STOP) {
                        changeMusic = false
                    }
                    if (changeMusic) {
                        mediaPlayer?.stop()
//                    mediaPlayer?.release()
                        mediaPlayer?.reset()
//                    mediaPlayer = null

                        initMediaPlayer(it.mMediaLocalPath!!)
                        it.isPlaying = true
                        displayNotificaiton(it)
                        mCurrentAudioParcel?.let {
                            mCurrentAudioParcel?.isPlaying = false
//                        displayNotificaiton(mCurrentAudioParcel)
                            SMTRichAudioPNGenerator.getInstance().clearForegroundNotificationAndShowSimpleNotification(this, mCurrentAudioParcel!!)
                        }
                        mCurrentAudioParcel = it
                        return super.onStartCommand(intent, flags, startId)
                    }
                } else {
                    mCurrentAudioParcel = it
                }
            }
            mCurrentAudioParcel?.let {
                when (it.action) {
                    ACTION_INIT -> {
                        var path = it.mMediaLocalPath
                        if (!path.isNullOrEmpty()) {
                            Thread(object : Runnable {
                                override fun run() {
                                    initMediaPlayer(path)
                                }
                            }).start()
                        }
                    }
                    ACTION_PAUSE -> pauseMusic()
                    ACTION_PLAY -> startMusic()
                    ACTION_PLAY_PAUSE -> handlePlayPauseMusic()
                    ACTION_CLOSE, ACTION_STOP -> clearData()
                }
            }

        }
        return super.onStartCommand(intent, flags, startId)
    }

    private fun handlePlayPauseMusic() {
        if (mediaPlayer?.isPlaying == true) {
            pauseMusic()
        } else {
            startMusic()
        }
        displayNotificaiton(mCurrentAudioParcel)
    }

    private fun initMediaPlayer(path: String) {
        if (mediaPlayer == null) {
            mediaPlayer = MediaPlayer()
        }
        val file = File(path)
        if (file.isFile && file.exists()) {
            mediaPlayer?.setDataSource(path)
            mediaPlayer?.setOnPreparedListener(this)
            mediaPlayer?.setOnCompletionListener(this)
            mediaPlayer?.prepare()
        } else {
            mCurrentAudioParcel?.let {
                it.isPlaying = false
                it.action = ACTION_NO_MUSIC
                displayNotificaiton(it)
            }
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                stopForeground(true)
            }
            stopSelf()
        }
    }

    /* override fun onDestroy() {
 //        clearData()
         super.onDestroy()
     }*/

    private fun clearData() {
        mediaPlayer?.let {
            it.stop()
            it.release()
        }
        mCurrentAudioParcel?.let { it ->
            it.mMediaLocalPath?.let { path ->
                SMTCommonUtility.deleteFile(path)
            }
        }
        mediaPlayer = null
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            stopForeground(true)
        }
        stopSelf()
    }

    override fun onDestroy() {
        // Unregister since the activity is about to be closed.
        LocalBroadcastManager.getInstance(this).unregisterReceiver(mMessageReceiver)
        super.onDestroy()
    }

}