package com.netcore.android.notification

import android.content.Context
import android.content.Intent
import com.netcore.android.SMTConfigConstants
import com.netcore.android.db.SMTDataBaseService
import com.netcore.android.logger.SMTLogger
import com.netcore.android.mediadownloader.SMTMediaDownloadManager
import com.netcore.android.notification.models.SMTNotificationData
import com.netcore.android.notification.services.SMTAlarmService
import org.json.JSONArray
import org.json.JSONObject
import java.lang.ref.WeakReference
import java.text.SimpleDateFormat
import java.util.*


/**
 * Copyright © 2019 Netcore. All rights reserved.
 *
 * Handles the incoming notification
 *
 * @author Netcore
 * @version 1.0
 * @since 26-02-2019
 */

class SMTScheduleNotification {
    //    val mScheduleNotification: Any
    val TAG = SMTScheduleNotification::class.java.name


    @Synchronized
    internal fun parseScheduledNotification(context: Context, scheduledNotification: ArrayList<String>?, sourceType: Int, isFromBootReciever: Boolean) {
        try {
            if (!scheduledNotification.isNullOrEmpty()) {
                val dbService = SMTDataBaseService.getInstance(WeakReference(context))
                for (payload in scheduledNotification) {
                    val jsonObject = JSONObject(payload)
                    jsonObject.put(SMTNotificationConstants.NOTIF_SOURCE_TYPE, sourceType)
                    jsonObject.put(SMTNotificationConstants.NOTIF_IS_SCHEDULED_PN, 1)

                    // clearing model class on new local notification
                    val mNotificationModel = getNotificationModel(jsonObject.toString(), sourceType)
                    if (mNotificationModel?.mTrid != null) {
                        if (!dbService.findNotificationWithId(mNotificationModel.mTrid, sourceType)) {
                            setNotificationInDatabase(context, jsonObject.toString(), mNotificationModel, isFromBootReciever)
                        }
                    }
                    Thread.sleep(800L)
                }
            }
        } catch (e: Exception) {
            SMTLogger.e(TAG, e.message.toString())
        }

    }

    private fun setNotificationInDatabase(context: Context, payload: String, mNotificationModel: SMTNotificationData?, isFromBootReciever: Boolean) {
        try {

            mNotificationModel?.let {
                if (isValidScheduledDate(it)) {
                    SMTMediaDownloadManager().downloadMedia(context, it, object : SMTMediaDownloadManager.MediaDownloadListener {
                        override fun onDownloadSuccess(notification: SMTNotificationData) {
                            setMediaPathIfItHasImages(context, payload, notification, isFromBootReciever)
                        }

                        override fun onDownloadFailed(notification: SMTNotificationData) {
                            setMediaPathIfItHasImages(context, payload, notification, isFromBootReciever)
                        }
                    })
                }
            }

        } catch (e: Exception) {
            SMTLogger.e(TAG, e.message.toString())
        }
    }

    fun setMediaPathIfItHasImages(context: Context, payload: String, mNotificationModel: SMTNotificationData, isFromBootReciever: Boolean) {
        val mPayload = JSONObject(payload)
        when (mNotificationModel.mNotificationType) {
            SMTNotificationType.CAROUSEL_PORTRAIT.type, SMTNotificationType.CAROUSEL_LANDSCAPE.type -> {
                val mCarouselArray = JSONArray()
                mNotificationModel.mCarouselList?.forEach {
                    val mCarouselObject = JSONObject()
                    mCarouselObject.put(SMTNotificationConstants.NOTIF_ID, it.id)
                    mCarouselObject.put(SMTNotificationConstants.NOTIF_CAROUSEL_IMG_DEEPLINK_KEY, it.imgDeeplink)
                    mCarouselObject.put(SMTNotificationConstants.NOTIF_CAROUSEL_IMG_MSG_KEY, it.imgMsg)
                    mCarouselObject.put(SMTNotificationConstants.NOTIF_CAROUSEL_IMG_TITLE_KEY, it.imgTitle)
                    mCarouselObject.put(SMTNotificationConstants.NOTIF_CAROUSEL_IMG_URL_KEY, it.imgUrl)
                    mCarouselObject.put(SMTNotificationConstants.NOTIF_DOWNLOAD_STATUS, it.mDownloadStatus)
                    mCarouselObject.put(SMTNotificationConstants.NOTIF_MEDIA_LOCAL_PATH_KEY, it.mMediaLocalPath)
                    mCarouselArray.put(mCarouselObject)
                }

                val mKey = SMTNotificationConstants.NOTIF_CAROUSEL_KEY
                val mCustomPayload = SMTNotificationConstants.NOTIF_SMT_PAYLOAD_KEY
                if (mPayload.has(mCustomPayload)) {
                    val mPPayload = mPayload.getJSONObject(mCustomPayload)
                    if (mPPayload.has(mKey)) {
                        mPPayload.remove(mKey)
                        mPPayload.put(mKey, mCarouselArray)
                    }
                }
            }
            SMTNotificationType.BIG_IMAGE.type, SMTNotificationType.AUDIO.type, SMTNotificationType.GIF.type -> {
                val mCustomPayload = SMTNotificationConstants.NOTIF_SMT_PAYLOAD_KEY
                if (mPayload.has(mCustomPayload)) {
                    val mPPayload = mPayload.getJSONObject(mCustomPayload)
                    mPPayload.put(SMTNotificationConstants.NOTIF_MEDIA_LOCAL_PATH_KEY, mNotificationModel.mMediaLocalPath)
                    mPPayload.put(SMTNotificationConstants.NOTIF_DOWNLOAD_STATUS, mNotificationModel.mDownloadStatus)
                }
            }
        }
        // Inserting schedule notification inside database
        insertNotification(context, mPayload.toString(), mNotificationModel, isFromBootReciever)
    }

    private fun insertNotification(context: Context, payload: String, mNotificationModel: SMTNotificationData, isFromBootReciever: Boolean) {
        if (mNotificationModel.mScheduledDate != null && mNotificationModel.mTtl != null) {
            val dbService = SMTDataBaseService.getInstance(WeakReference(context))
            dbService.insertSchedulePNToTable(mNotificationModel.mTrid, payload, mNotificationModel.mSourceType,
                    mNotificationModel.mScheduledDate!!, SMTNotificationConstants.NOTIF_IS_SCHEDULED, mNotificationModel.mTtl!!)
            scheduleNotification(context, payload, mNotificationModel, isFromBootReciever)
        }
    }

    /**
     * scheduled local notfication
     * @param context - pass context
     * @param notifData - notification data
     * @param isFromBootReciever -journey from boot receiver true or false
     */
    private fun scheduleNotification(context: Context, payload: String, notifData: SMTNotificationData, isFromBootReciever: Boolean) {
        val intent = Intent()
        intent.putExtra(PUT_EXTRA_NOTIF_DATA, notifData)
        intent.putExtra(PUT_EXTRA_PAYLOAD, payload)
        intent.putExtra(PUT_EXTRA_IS_FROM_BOOT, isFromBootReciever)
        SMTAlarmService.enqueueWork(context, intent)
    }

    private fun getUtcDateTime(scheduledTimePN: String): Date {
        var simpleDateFormat = SimpleDateFormat(SMTConfigConstants.SERVER_TIME_FORMAT, Locale.getDefault())
        simpleDateFormat.timeZone = TimeZone.getTimeZone("UTC")
        return simpleDateFormat.parse(scheduledTimePN)
    }

    /**
     * Provides notification data model after parsing the notification payload
     */
    fun getNotificationModel(notifData: String, sourceType: Int): SMTNotificationData? {
        return SMTNotificationParser().parse(notifData, sourceType)
    }


    fun onBootComplete(context: Context) {
        try {
            val dbService = SMTDataBaseService.getInstance(WeakReference(context))
            val mScheduledNotificationList = dbService.getScheduledNotification()
            mScheduledNotificationList?.forEach {
                if (isValidScheduledDate(it.notificationData)) {
                    scheduleNotification(context, it.payload, it.notificationData, true)
                } else {
                    SMTLogger.d(TAG, "Scheduled Notification has been Expired")
                }
            }
        } catch (e: Exception) {
            SMTLogger.e(TAG, e.message.toString())
        }
    }

    private fun isValidScheduledDate(mNotificationModel: SMTNotificationData): Boolean {
        val mScheduleDate = mNotificationModel.mScheduledDate
        val mTtl = mNotificationModel.mTtl
        if (mScheduleDate != null && mTtl != null) {
            return getUtcDateTime(mScheduleDate).before(getUtcDateTime(mTtl))
        } else {
            return false
        }
    }

    fun updateScheduledNotification(context: Context, mTrid: String, status: String) {
        try {
            val dbService = SMTDataBaseService.getInstance(WeakReference(context))
            dbService.updateScheduledStatus(mTrid, status)
        } catch (e: Exception) {
            SMTLogger.e(TAG, e.message.toString())
        }
    }

    companion object {
        val PUT_EXTRA_NOTIF_DATA = "notif_data"
        val PUT_EXTRA_IS_FROM_BOOT = "is_from_boot_reciever"
        val PUT_EXTRA_PAYLOAD = "payload"
    }
}