package com.netcore.android.utility

import android.content.Context
import android.content.pm.PackageInfo
import android.content.pm.PackageManager
import androidx.core.content.pm.PackageInfoCompat
import com.netcore.android.BuildConfig
import com.netcore.android.logger.SMTLogger
import java.lang.ref.WeakReference

/**
 * Copyright © 2019 Netcore. All rights reserved.
 *
 * Singleton class to provides Application related Info
 *
 * @author Netcore
 * @version 1.0
 * @since 26-02-2019
 */
internal class SMTAppInfo(private val context: Context) {

    val TAG = SMTAppInfo::class.java.simpleName

    /**
     * App version
     */
    val appVersion = getAppVersionName(context)

    /**
     * Application version code
     */
    val appBuild = getAppVersionCode(context)

    /**
     * App package name
     */
    val appBundleId = context.packageName

    val manifestConfig = SMTManifestInfo.getInstance(WeakReference(context))

    val identity: String? = null

    val smtSdkVersion = getSdkVersion()

    /**
     * Provides SDK version
     * @return Int - SDK version code
     */
    private fun getSdkVersion(): String {
        return BuildConfig.VERSION_NAME
    }

    /**
     * Provided App version code
     * @param context App context
     * @return String - App version code
     */
    private fun getAppVersionCode(context: Context): String {
        try {
            val pInfo = context.packageManager.getPackageInfo(context.packageName, 0)
            val longVersionCode = PackageInfoCompat.getLongVersionCode(pInfo)
            return longVersionCode.toString()
        } catch (e: PackageManager.NameNotFoundException) {
            SMTLogger.d(TAG, "Unable to get app build, error :- $e")
        }
        return ""
    }

    /**
     * Provides app version Name
     * @param context - App context
     * @return String - App version name
     */
    private fun getAppVersionName(context: Context): String {
        val packageInfo: PackageInfo
        try {
            packageInfo = context.packageManager.getPackageInfo(context.packageName, 0)
            return packageInfo.versionName
        } catch (e: PackageManager.NameNotFoundException) {
            SMTLogger.e(TAG, "Unable to get app version name, error :- $e")
        }
        return ""
    }
}