package com.netcore.android.notification

import android.content.Context
import com.netcore.android.Smartech
import com.netcore.android.db.SMTDataBaseService
import com.netcore.android.event.SMTEventRecorder
import com.netcore.android.event.SMTNotificationSourceType
import com.netcore.android.logger.SMTLogger
import com.netcore.android.notification.audio.SMTRichAudioPNGenerator
import com.netcore.android.notification.carousel.SMTCarouselLandscapePNGenerator
import com.netcore.android.notification.carousel.SMTCarouselPortraitPNGenerator
import com.netcore.android.notification.gif.SMTRichGifPNGenerator
import com.netcore.android.notification.models.SMTNotificationData
import com.netcore.android.preference.SMTPreferenceConstants
import com.netcore.android.preference.SMTPreferenceHelper
import com.netcore.android.utility.SMTCommonUtility
import com.netcore.android.workmgr.SMTWorkerScheduler
import java.lang.ref.WeakReference

/**
 * Copyright © 2019 Netcore. All rights reserved.
 *
 * Handles the incoming notification
 *
 * @author Netcore
 * @version 1.0
 * @since 26-02-2019
 */
internal class SMTPNHandler(private val notificationGeneratorProvider: SMTNotificationGeneratorProvider) {

    private val TAG = SMTPNHandler::class.java.simpleName

    private var mNotificationModel: SMTNotificationData? = null

    /**
     * Handled the incoming notification
     * @param context - App context
     * @param notifData - Notification JSON in string format
     */
    fun handleNotification(context: Context, notifData: String, @SMTNotificationSourceType.Source sourceType: Int, disableDelivery: Boolean = false): Boolean {


        SMTCommonUtility.handleAppUpdate(context)

        if (!SMTCommonUtility.checkPanelAndSDKActiveStatus(context)) {
            SMTLogger.i(TAG, "SDK / Panel is not active")
            return false
        }

        if (!validateNotificationSettings(context)) {
            SMTLogger.i(TAG, "Either PN notification settings disabled or PN is opt out.")
            return false
        }

        /**
         *  Written this code explicitly to handle PN permission event to be recorded
         *  even if user in terminate state enables the receives notification.
         *  It is placed here to record permission in terminate state.
         */
        SMTCommonUtility.checkAndRecordNotificationPermissionStatus(context)

        SMTLogger.v(TAG, "Notification source check: $sourceType")
        // Parses the notification data
        mNotificationModel = getNotificationModel(notifData, sourceType)
        when (sourceType) {
            SMTNotificationSourceType.NOTIFICATION_SOURCE_PN -> {
                val mSchDt = mNotificationModel?.mScheduledDate
                val mTtl = mNotificationModel?.mTtl
                if (mSchDt != null && mTtl != null && mNotificationModel?.mIsScheduledPN!! == 0) {
                    if (mSchDt.isNotEmpty() && mTtl.isNotEmpty()) {
                        val mScheduledNotification = SMTScheduleNotification()
                        val mList = ArrayList<String>()
                        mList.add(notifData)
                        mScheduledNotification.parseScheduledNotification(context, mList, SMTNotificationSourceType.NOTIFICATION_SOURCE_PN, false)
                        return true
                    }
                }
            }
        }


        if (mNotificationModel?.isFromSmartech == false) {
            SMTLogger.i(TAG, "Notification source is not Smartech.")
            return false
        }

        // If Transaction Id is not there then don't show the Notification
        if (mNotificationModel?.mTrid == null) {
            SMTLogger.w(TAG, "Notification transaction id is null")
            return false
        }

        // Insert Notification into Notification table
        mNotificationModel?.mTrid?.let {

            val dbService = SMTDataBaseService.getInstance(WeakReference(context))
            val isTridExist = dbService.findNotificationWithId(mNotificationModel?.mTrid!!, sourceType)

            // If Notification already present in the DB, then don't process it again, just ignore it
            if (mNotificationModel?.mIsScheduledPN == 0 && isTridExist) {
                SMTLogger.i(TAG, "Notification already displayed to user for trId : $it")
                return false
            }


            // If user has opted for Notification control then we don't need to send delivery again'
            if (!disableDelivery) {
                // Insert notification into NotificationTable
                if (mNotificationModel?.mIsScheduledPN == 0) {
                    if (isTridExist) {
                        SMTLogger.i(TAG, "Notification already displayed to user for trId : $it")
                        return true
                    }
                    dbService.insertPNToNotificationTable(it, notifData, sourceType)
                    SMTEventRecorder.getInstance(context).recordNotificationDelivery(it, notifData,
                            sourceType, mNotificationModel!!)
                }
            }
        }
        // Check if Notification listener is enabled in manifest file then
        // send the notification to client via interface
        val isNotificationListenerEnabled = SMTNotificationUtility.getInstance().checkIfNotificationListenerEnabled(context)
        SMTLogger.v(TAG, "value of isNotificationEnabled $isNotificationListenerEnabled")
        if (isNotificationListenerEnabled == 1) {
            Smartech.getInstance(WeakReference(context)).getSMTNotificationListener()?.getSmartechNotifications(notifData, sourceType)
        } else {
            mNotificationModel?.let {
                val notificationGenerator = notificationGeneratorProvider.getNotificationGenerator(it.mNotificationType!!, context)
                val type = it.mNotificationType?.toLowerCase() ?: SMTNotificationType.SIMPLE.type
                when (type) {
                    SMTNotificationType.CAROUSEL_LANDSCAPE.type -> (notificationGenerator as SMTCarouselLandscapePNGenerator).handle(it, sourceType)
                    SMTNotificationType.CAROUSEL_PORTRAIT.type -> (notificationGenerator as SMTCarouselPortraitPNGenerator).handle(it, sourceType)
                    SMTNotificationType.GIF.type -> (notificationGenerator as SMTRichGifPNGenerator).handle(context, it, sourceType)
                    SMTNotificationType.AUDIO.type -> (notificationGenerator as SMTRichAudioPNGenerator).handle(context, it, sourceType)
                    SMTNotificationType.BIG_IMAGE.type -> (notificationGenerator as SMTRichImagePNGenerator).handle(context, it, sourceType)
                    else -> (notificationGenerator as SMTSimplePNGenerator).handle(context, it)
                }

                if (mNotificationModel?.mIsScheduledPN == 1) {
                    SMTEventRecorder.getInstance(context).recordNotificationPNRendered(it.mTrid, notifData,
                            sourceType, it)
                }
            }
        }

        /**
         *
         *  Check status of EventSync worker is running.
         *  If worker is running, It will automatically pickup the delivery of notification event.
         *  If worker is not running, start Event Sync worker so it will send
         *  the delivery event to server,
         *
         */
        SMTWorkerScheduler.getInstance().checkStatusAndScheduleEventWorker(context)


        return true
    }

    /**
     * Provides notification data model after parsing the notification payload
     */
    fun getNotificationModel(notifData: String, sourceType: Int): SMTNotificationData? {
        if (mNotificationModel != null) {
            return mNotificationModel
        }
        return SMTNotificationParser().parse(notifData, sourceType)
    }

    fun renderNotification(context: Context, notifData: String, @SMTNotificationSourceType.Source notifSource: Int): Boolean {
        val notificationModel = getNotificationModel(notifData, notifSource)
        if (notificationModel != null) {

            if (!notificationModel.isFromSmartech) {
                SMTLogger.i(TAG, "Notification source is not Smartech.")
                return false
            }

            // If Transaction Id is not there then don't show the Notification
            if (notificationModel.mTrid.isNullOrEmpty()) {
                SMTLogger.w(TAG, "Notification transaction id is null")
                return true
            }

            notificationModel.let {
                val notificationGenerator = notificationGeneratorProvider.getNotificationGenerator(it.mNotificationType!!, context)
                val type = it.mNotificationType?.toLowerCase() ?: SMTNotificationType.SIMPLE.type
                when (type) {
                    SMTNotificationType.CAROUSEL_LANDSCAPE.type -> (notificationGenerator as SMTCarouselLandscapePNGenerator).handle(it, notifSource)
                    SMTNotificationType.CAROUSEL_PORTRAIT.type -> (notificationGenerator as SMTCarouselPortraitPNGenerator).handle(it, notifSource)
                    SMTNotificationType.GIF.type -> (notificationGenerator as SMTRichGifPNGenerator).handle(context, it, notifSource)
                    SMTNotificationType.AUDIO.type -> (notificationGenerator as SMTRichAudioPNGenerator).handle(context, it, notifSource)
                    SMTNotificationType.BIG_IMAGE.type -> (notificationGenerator as SMTRichImagePNGenerator).handle(context, it, notifSource)
                    else -> (notificationGenerator as SMTSimplePNGenerator).handle(context, it)
                }
            }
        }
        return true
    }

    private fun validateNotificationSettings(context: Context): Boolean {
        // check if notification permission disabled from application settings
        val preference = SMTPreferenceHelper.getAppPreferenceInstance(context, null)
        val isNotificationOptin = preference.getBoolean(SMTPreferenceConstants.OPT_IN_OUT_PUSH_NOTIFICATION, true)
        val areNotificationsEnabled = SMTCommonUtility.areNotificationsEnabled(context)

        return (isNotificationOptin && areNotificationsEnabled)
    }
}