package com.netcore.android

import android.app.Application
import android.app.NotificationChannel
import android.app.NotificationChannelGroup
import android.content.Context
import android.content.res.Resources
import android.location.Location
import android.net.Uri
import androidx.annotation.Keep
import androidx.appcompat.app.AppCompatActivity
import com.google.firebase.BuildConfig
import com.netcore.android.db.SMTDataBaseService
import com.netcore.android.db.SMTInAppRulesTable
import com.netcore.android.event.*
import com.netcore.android.inapp.InAppCustomHTMLListener
import com.netcore.android.inapp.SMTInAppApiService
import com.netcore.android.inapp.SMTInAppHandler
import com.netcore.android.inbox.network.SMTInboxApiService
import com.netcore.android.inbox.utility.SMTInboxMessageStatus
import com.netcore.android.inbox.utility.SMTInboxMessageType
import com.netcore.android.listeners.SMTInboxCallback
import com.netcore.android.logger.SMTDebugLevel
import com.netcore.android.logger.SMTLogger
import com.netcore.android.network.SMTResponseListener
import com.netcore.android.network.SMTThreadPoolManager
import com.netcore.android.network.models.SMTRequest
import com.netcore.android.network.models.SMTResponse
import com.netcore.android.network.models.SMTSdkInitializeResponse
import com.netcore.android.notification.*
import com.netcore.android.notification.channel.SMTNotificationChannel
import com.netcore.android.notification.channel.SMTNotificationChannelHelper
import com.netcore.android.notification.models.SMTNotificationData
import com.netcore.android.preference.SMTPreferenceConstants
import com.netcore.android.preference.SMTPreferenceHelper
import com.netcore.android.smartech.userprofile.SMTUserProfile
import com.netcore.android.utility.SMTCommonUtility
import com.netcore.android.utility.SMTGWSource
import com.netcore.android.utility.SMTInfo
import com.netcore.android.workmgr.SMTWorkerScheduler
import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject
import java.lang.ref.WeakReference
import java.net.URL
import java.net.URLDecoder
import java.util.*
import kotlin.collections.HashMap
import kotlin.collections.component1
import kotlin.collections.component2
import kotlin.collections.set

/**
 *
 * @author Netcore
 *
 *  This class is base class of Smartech SDK initiizations.
 *
 *  @Description This class will provide method which are required to use Smartech SDK.
 *  User need to create instnace of this class to use functionality
 *  provided by Smartech SDK.
 *
 *  @param context Application context
 *  @constructor private constructor
 *  @author Smartech
 */

class Smartech private constructor(val context: WeakReference<Context>) : SMTResponseListener {

    private val TAG = Smartech::class.java.simpleName

    // Settings info contains AppInfo, DeviceInfo, NetworkInfo
    private lateinit var mSmtInfo: SMTInfo

    // Helper class
    private lateinit var mSmartechHelper: SmartechHelper

    // To Keep track of events those needs to be tracked on SDK init
    private val mInitEventTrackList = arrayListOf<Int>()

    private var mInboxCallback: SMTInboxCallback? = null
    private var smtNotificationClickListener: SMTNotificationClickListener? = null
    private var inAppCustomHTMLListener: InAppCustomHTMLListener? = null
    private var smtNotificationListener: SMTNotificationListener? = null


    //Channel helper class to expose channel methods
    private lateinit var mSmtNotificationChannelHelper: SMTNotificationChannelHelper

    /**
     * Initialise the variables
     */
    private fun init() {
        // Init using context argument
        mSmtInfo = SMTInfo.getInstance(context)
        mSmartechHelper = SmartechHelper(context, mSmtInfo, mPreferences, this)
        mSmtNotificationChannelHelper = SMTNotificationChannelHelper.getInstance(context)

        SMTLogger.i(TAG, "Init block is called")

    }

    @Keep
    companion object {
        /**
         * Used to check whether Initialise Api has to call or not on session refresh
         * Use case - if app is launched after session expired, then there is a chance that
         * the initialise api will get called twice
         */
        private var isInitializeInProgress: Boolean = false
        private var isAppInitialized: Boolean = false
        private lateinit var mPreferences: SMTPreferenceHelper

        @Volatile
        private var INSTANCE: Smartech? = null

        /**
         * Getting instance of the class
         *
         * @param context App context
         * @return Smartech instance
         */
        @JvmStatic
        fun getInstance(context: WeakReference<Context>): Smartech =
                INSTANCE ?: synchronized(Smartech::class.java) {
                    INSTANCE ?: buildInstance(context).also { INSTANCE = it }
                }

        private fun buildInstance(context: WeakReference<Context>): Smartech {
            context.get()?.let {
                mPreferences = SMTPreferenceHelper.getAppPreferenceInstance(it, null)
            }
            return Smartech(context)
        }
    }

    /**
     * Exposed method to initialise the SDK
     */
    fun initializeSdk(applicationContext: Application?) {

        if (applicationContext != null) {
            SMTLogger.i(TAG, "SDK initialization started.")

            // Initialize Device, App, Network related info
            // Initialise preference, and helper class
            init()

            // Register for Application lifecycle
            SMTActivityLifecycleCallback.getInstance().register(applicationContext)
        } else {
            SMTLogger.w(TAG, "Application instance is null.")
        }
    }

    private fun startInitialization() {

        val initializeRunner = Runnable {
            isInitializeInProgress = true
            /**
             *  Make sure that ActivityLifecycle has been registered and On resumed of current activity is called
             *  We need instance of current activity to display InApp
             */
            Thread.sleep(2000)

            val appStatus = SMTActivityLifecycleCallback.getInstance().isAppInForeground()

            if (appStatus) {

                // set the debug level
                initDebugLevel()

                // start new session and reset the required values
                startNewSession()

                //Set App launch status in preference and record system events
                //recordAppLaunchEvents()

                // call initialize SDK api
                mSmartechHelper.makeInitializeApiCall(SMTRequest.SMTApiTypeID.SDK_INITIALIZE)

                // Delete expired notifications
                SMTDataBaseService.getInstance(context)
                        .checkAndDeleteExpiredNotifcation(System.currentTimeMillis() - SMTConfigConstants.NOTIFICATION_EXPIRY_TIME)

            } else {
                SMTLogger.i(TAG, "Initialization is skipped because application is in background. ")
                isInitializeInProgress = false
            }
        }

        /**
         *  Check panel is active or not before SDK Init
         *  YES - Init APK
         *  N0 - Use 7 days logic to active  or deactive SDK init
         */
        /*if (SMTCommonUtility.checkPanelStatus(applicationContext)) {
            // Intentionally blank
        }*/

        // Start running the thread to perform the init task
        SMTThreadPoolManager.getIntance().execute(initializeRunner)
    }

    private fun callListAndSegmentAPI() {

        val listSegmentRunner = Runnable {
            SMTInAppApiService.getInstance(context).makeListSegmentApiCall()
        }

        // Start running the thread to perform the task
        SMTThreadPoolManager.getIntance().execute(listSegmentRunner)
    }

    /**
     * set the debug level
     * if its not set then set default to Debug level none else
     * what ever user set or stored in preference
     */
    private fun initDebugLevel() {

        /* this function checks whether the guids list from preference matched with current guid */
        fun checkValidUserGuids(): Boolean {
            try {
                val guidLists = mPreferences.getString(SMTPreferenceConstants.GUIDS)
                val guidsArray = JSONArray(guidLists)
                val deviceUniqueId = getDeviceUniqueId()
                for (i in 0 until guidsArray.length()) {
                    if (guidsArray[i] == deviceUniqueId) return true
                }
            } catch (e: JSONException) {
                SMTLogger.e(TAG, e.message.toString())
            }
            return false
        }


        /*check the debug level from preference for remote logging and
         check guids matched with current device guid and set the debug level*/
        val storedDebugLevel = mPreferences.getInt(SMTPreferenceConstants.SMT_DEBUG_LEVEL, -1)
        var updatedDebugLevel = if (mPreferences.getBoolean(SMTPreferenceConstants.IS_LOG_ENABLED) && checkValidUserGuids()) {
            mPreferences.getInt(SMTPreferenceConstants.LOG_LEVEL)
        } else {
            if (storedDebugLevel < 0) {
                SMTDebugLevel.Level.NONE
            } else {
                storedDebugLevel
            }
        }
        mPreferences.setInt(SMTPreferenceConstants.SMT_DEBUG_LEVEL, updatedDebugLevel)
        SMTLogger.setDebugLevel(updatedDebugLevel)
        SMTLogger.i(TAG, "SDK debug level: $updatedDebugLevel")
    }


    /**
     *  This method will start background workers
     */
    private fun startBackgroundWorkers() {
        backgroundEventsSync(context)
        progressEventsSync(context)
    }

    /**
     *  This method will used to start the background sync worker
     *  to perform the task in background which will
     *  run after every 15 minutes interval to sync all pending anf Failed events.
     *
     *  @param context application
     *
     */
    private fun backgroundEventsSync(context: WeakReference<Context>) {
        context.get()?.let {
            SMTWorkerScheduler.getInstance().scheduleBackgroundSyncWorker(it)
        }
    }

    /**
     *  This method will be used to check the state of events
     *  whose state is InProgress, haven't synced to server and
     *  last timestamp is greater then 2 days so
     *  change the status of events to failed.
     *
     * @param context application
     */
    private fun progressEventsSync(context: WeakReference<Context>) {
        context.get()?.let {
            SMTWorkerScheduler.getInstance().scheduleInProgressEventWorker(it)
        }
    }

    private fun recordAppLaunchEvents() {
        if (mPreferences.getBoolean(SMTPreferenceConstants.SMT_IS_FIRST_LAUNCH, true)) {
            mInitEventTrackList.add(SMTEventId.EVENT_APP_REINSTALLED)
            mPreferences.setBoolean(SMTPreferenceConstants.SMT_IS_FIRST_LAUNCH, false)
        } else {
            mInitEventTrackList.add(SMTEventId.EVENT_APP_LAUNCHED)
            mInitEventTrackList.add(SMTEventId.EVENT_DEVICE_DETAILS_UPDATED)
        }

        mInitEventTrackList.add(SMTEventId.EVENT_LOCATION_FETCH_ENABLED)
        mInitEventTrackList.add(SMTEventId.EVENT_USER_ENABLED_PN)

        mSmartechHelper.recordInitAppEvents(mInitEventTrackList)
        mInitEventTrackList.clear()
    }


    private fun startNewSession() {
        // On Every app launch change session
        mPreferences.setLong(SMTPreferenceConstants.CURRENT_SESSION_ID, System.currentTimeMillis())

        //Session changed reset the usage
        context.get()?.let {
            SMTInAppHandler.getInstance().resetUsageForSessionTypeRule(it)
        }

        // Reset Attribution params on session refresh or on new session
        if (!mPreferences.getBoolean(SMTPreferenceConstants.IS_LAUNCHED_FROM_NOTIFICATION, false)) {
            mPreferences.setString(SMTPreferenceConstants.SMT_ATTRIBUTION_PARAMS, "")
        }

        mPreferences.setBoolean(SMTPreferenceConstants.IS_LAUNCHED_FROM_NOTIFICATION, false)

        // On session refresh also fetch inbox messages only if if auto fetch is enabled
        if (mPreferences.getInt(SMTPreferenceConstants.SMT_MF_IS_AUTO_FETCH_INBOX_MESSAGE) == 1 && !isInitializeInProgress) {
            SMTInboxApiService.getInstance(context).makeAppInboxApiCall(mInboxCallback, SMTInboxMessageType.Type.INBOX_MESSAGE)
        }
    }

    /**
     * On API call success process other tasks
     * @param response received from network call
     */
    override fun onResposneSuccess(response: SMTResponse) {
        when (response.smtApiTypeID) {
            SMTRequest.SMTApiTypeID.SDK_INITIALIZE_ON_SESSION_REFRESH,
            SMTRequest.SMTApiTypeID.SDK_INITIALIZE -> {
                proceedOnInitialiseApiSuccess(response)
            }
            else -> {
            }
        }
    }

    /**
     * On API call failed handle it
     * @param response - failure response
     */
    override fun onResponseFailure(response: SMTResponse) {
        when (response.smtApiTypeID) {
            SMTRequest.SMTApiTypeID.SDK_INITIALIZE_ON_SESSION_REFRESH,
            SMTRequest.SMTApiTypeID.SDK_INITIALIZE -> {
                proceedOnInitialiseApiFailure(response)
            }
            else -> {
            }
        }
    }

    /**
     * Handle the API failure scenario
     */
    private fun proceedOnInitialiseApiFailure(response: SMTResponse) {
        SMTLogger.i(TAG, "Smartech SDK not initialized successfully.")
        isInitializeInProgress = false
        isAppInitialized = false
        val requestType: Int = response.smtApiTypeID.value
        // check if any smartech settings stored, then fall back to it
        val isSettingsStored = mPreferences.getBoolean(SMTPreferenceConstants.IS_SMRATECH_SETTINGS_STORED)
        // first launch the default value is also not stored
        // Better to store it and use it
        if (!isSettingsStored) {
            // Store the default values
            mSmartechHelper.updateSmartechConfigs(SMTSdkInitializeResponse.SmartTechSettings())
        }
        // fall back to existing / stored settings in case of failure

        when (mPreferences.getBoolean(SMTPreferenceConstants.IS_SDK_ACTIVE) && mPreferences.getBoolean(SMTPreferenceConstants.IS_PANEL_ACTIVE)) {
            true -> {
                if (requestType != SMTRequest.SMTApiTypeID.SDK_INITIALIZE_ON_SESSION_REFRESH.value) {
                    // call push amp api on app launch only not in session refresh
                    mSmartechHelper.makePushAmpApiCall()

                    // Send regular app launch event on INit api failure but make sure
                    // SDK is active before sending event
                    recordAppLaunchEvents()

                    startBackgroundWorkers()
                }
                context.get()?.let {
                    SMTEventBatchProcessor.getInstance(it).makeEventBatchProcessingApiCall()
                }
            }
            false -> {
                mSmartechHelper.stopAndRefreshPushAmpBgTask()
                SMTDataBaseService.getInstance(context).resetInAppTableData(SMTInAppRulesTable.IN_APP_TABLE)
            }
        }
    }

    /**
     * Handles Init SDK Api success call
     * @param response InitSDK API call
     */
    private fun proceedOnInitialiseApiSuccess(response: SMTResponse) {
        SMTLogger.i(TAG, "Smartech SDK initialized successfully.")
        isInitializeInProgress = false
        isAppInitialized = true
        val initResponse = response as SMTSdkInitializeResponse
        val requestType: Int = response.smtApiTypeID.value
        val settings = initResponse.smartechSettings
        settings?.let {

            mSmartechHelper.updateSmartechConfigs(it)
            SMTLogger.i(TAG, "Smartech SDK status Panel: ${it.panelActive}, SDK: ${it.sdkActive}")
            SMTLogger.i(TAG, "SDK initialized from Smartech panel with settings : ${settings.toLimitString()}")

            if (it.panelActive && it.sdkActive) {

                startBackgroundWorkers()

                if (requestType != SMTRequest.SMTApiTypeID.SDK_INITIALIZE_ON_SESSION_REFRESH.value) {
                    // call push amp api on app launch only not in session refresh
                    mSmartechHelper.makePushAmpApiCall()
                }

                context.get()?.let { ctx ->
                    //println("InApp : isListAndSegmentPresent ${initResponse.isListAndSegmentPresent}")
                    if (initResponse.isListAndSegmentPresent) {
                        // Fetch list and segment
                        callListAndSegmentAPI()
                    }
                    //Process InApp Rules
                    SMTInAppHandler.getInstance().processInAppRules(initResponse, ctx)
                    // Handling PN
                    SMTNotificationUtility.getInstance().handlingPNTokenGenerationEvent(ctx)
                }


            } else {
                mSmartechHelper.stopAndRefreshPushAmpBgTask()
                SMTDataBaseService.getInstance(context).resetInAppTableData(SMTInAppRulesTable.IN_APP_TABLE)
            }
        }

        if (settings == null) {
            proceedOnInitialiseApiFailure(response)
        }

        // Saving current timestamp to shared preference
        context.get()?.let {
            SMTPreferenceHelper.getAppPreferenceInstance(it, null).setLong(SMTPreferenceConstants.SMT_SDK_INIT_TIMESTAMP, System.currentTimeMillis())
        }


        if (requestType != SMTRequest.SMTApiTypeID.SDK_INITIALIZE_ON_SESSION_REFRESH.value) {
            recordAppLaunchEvents()
        }

    }


    /**
     *  This function will bw called whenever application come in foreground state
     */
    fun onAppForeground() {

        context.get()?.let {
            SMTEventBatchProcessor.getInstance(it).checkForBatchProcessing(true)
        }

        fun isAppInstanceNotAvailable(): Boolean {
            return (!isInitializeInProgress && !isAppInitialized)
        }

        context.get()?.let { context ->
            try {
                SMTEventBatchProcessor.getInstance(context).makeEventBatchProcessingApiCall()
            } catch (e: Exception) {
                SMTLogger.e(TAG, e.message.toString())
            }
        }

        // If Apps instance is not available and internet connection is available
        // then start initializing SDK.
        if (isAppInstanceNotAvailable()) {
            SMTLogger.i(TAG, "Smartech SDK - v${BuildConfig.VERSION_NAME}, app id: ${mSmtInfo.mAppInfo?.manifestConfig?.appId}, guid: ${mSmtInfo.mDeviceInfo?.guid}")

            val appID = mSmartechHelper.getAppID(context)
            if (!appID.isNullOrEmpty()) {
                startInitialization()
            } else {
                context.get()?.let {
                    SMTEventBatchProcessor.getInstance(it).checkForBatchProcessing(false)
                }
                SMTLogger.w(TAG, "App Id is either null or empty.")
            }
        }

        /* Before session refresh check app is initialized or not */
        if (isAppInitialized) {
            val isSessionRefreshed = mSmartechHelper.checkIfCurrentSessionExpired()
            if (isSessionRefreshed) {
                SMTLogger.i(TAG, "Session expired")
                onSessionRefresh()
            }
        }
    }

    /**
     *  This method will be called when app goes in background
     *  so stop the Handler and set last active timestamp for the session.
     */
    fun onAppBackground() {
        context.get()?.let {
            SMTEventBatchProcessor.getInstance(it).checkForBatchProcessing(false)
        }
        setLastAppActiveTime()
    }

    /**
     * Get called once app comes to FG or goes into BG
     * According user in activity session is calculated
     */
    internal fun onAppForegroundStateChanged(isAppInFg: Boolean) {

        if (!isInitializeInProgress) {

            if (isAppInitialized) {
                if (isAppInFg) {

                    // running handler again to make sure that event processing handler is running
                    val appID = mSmartechHelper.getAppID(context)
                    if (!appID.isNullOrEmpty()) {
                        context.get()?.let {
                            SMTEventBatchProcessor.getInstance(it).checkForBatchProcessing(true)
                        }
                    }

                    mSmartechHelper.recordLocationAndNotificationPermissionStatus()

                    // here we need to set to false
                    mPreferences.setBoolean(SMTPreferenceConstants.IS_LAUNCHED_FROM_NOTIFICATION, false)
                }
            }
        }
    }

    private fun setLastAppActiveTime() {
        mPreferences.setLong(SMTPreferenceConstants.LAST_APP_ACTIVE_TIME_STAMP, System.currentTimeMillis())
    }

    /**
     * According user in activity session refresh is called
     * and does the work as per session changed
     */
    private fun onSessionRefresh() {
        val task = Runnable {
            // Start new session
            startNewSession()

            // Reset SMTInfo to re-fetch all the device details
            SMTInfo.resetInstance()

            // 3. make initialise API call
            mSmartechHelper.makeInitializeApiCall(SMTRequest.SMTApiTypeID.SDK_INITIALIZE_ON_SESSION_REFRESH)
        }
        SMTThreadPoolManager.getIntance().execute(task)
    }

    /**
     * This method is used to track custom event.
     *  @param eventName - name of the event for tracking.
     *  @param eventPayLoad - custom payload.
     */

    @JvmOverloads
    fun trackEvent(eventName: String?, eventPayLoad: HashMap<String, Any>? = null) {

        /**
         *  convert HashMap to JSONObject and convert all keys to lowercase after
         *  that convert back to HashMap  before sending event
         */

        if (!eventName.isNullOrEmpty()) {
            context.get()?.let {
                SMTEventRecorder.getInstance(it).recordEvent(SMTEventId.EVENT_CUSTOM,
                        eventName, eventPayLoad, SMTEventType.EVENT_TYPE_CUSTOM)
            }

        } else {
            SMTLogger.v(TAG, "Event name or HashMap is either null or empty.")
        }
    }

    /**
     * To track App installed
     * If App is reinstalled then don't record App Install
     */
    fun trackAppInstall(): Boolean {
        mInitEventTrackList.add(SMTEventId.EVENT_APP_INSTALLED)
        return true
    }

    /**
     * To track app install and update event by Smartech SDk
     */
    fun trackAppInstallUpdateBySmartech() {
        mInitEventTrackList.add(SMTEventId.EVENT_APP_INSTALL_UPDATE_NETCORE)
    }

    /**
     * To Track App updated
     */
    fun trackAppUpdate() {
        mInitEventTrackList.add(SMTEventId.EVENT_APP_UPDATED)
    }

    /**
     * This method is used to update profile related info.
     *  @param eventPayLoad - custom payload.
     */
    fun updateUserProfile(eventPayLoad: HashMap<String, Any>?) {
        eventPayLoad?.let {
            // convert profile push parameters to lowercase before sending back
            if (SMTUserProfile.getInstance(context).shouldSyncUserProfile(it)) { /* DE-DUP PROFILE BLOCK 4 of 4 */
                context.get()?.let { ctx ->
                    SMTEventRecorder.getInstance(ctx).recordEvent(SMTEventId.EVENT_USER_PROFILE_PUSH,
                            SMTEventId.getEventName(SMTEventId.EVENT_USER_PROFILE_PUSH), it, SMTEventType.EVENT_TYPE_SYSTEM)
                }

            }
        }
    }

    /**
     *  This method will be used to fetch existing generated token
     *  from device and save it to SharedPreferences.
     */
    @Suppress("unused")
    fun fetchAlreadyGeneratedTokenFromFCM() {
        mSmartechHelper.fetchExistingToken()
    }

    /**
     *  This method will be save device token in SDK.
     */
    @Suppress("unused")
    fun setDevicePushToken(token: String?) {
        if (!token.isNullOrEmpty()) {
            mSmartechHelper.setDevicePushToken(token, SMTGWSource.FCM)
        } else {
            SMTLogger.v(TAG, "Token is null or empty.")
        }

    }

    /**
     *  This method will return status of Tracking
     *  @return Boolean
     */
    @Suppress("unused")
    fun hasOptedTracking(): Boolean {
        return mPreferences.getBoolean(SMTPreferenceConstants.OPT_IN_OUT_TRACKING)
    }

    /**
     * This method will return status of Notification optin/optout
     * @return boolean value of Notification
     */
    @Suppress("unused")
    fun hasOptedPushNotification(): Boolean {
        return mPreferences.getBoolean(SMTPreferenceConstants.OPT_IN_OUT_PUSH_NOTIFICATION)
    }

    /**
     * This method will return status of InApp message
     * @return boolean
     */
    @Suppress("unused")
    fun hasOptedInAppMessage(): Boolean {
        return mPreferences.getBoolean(SMTPreferenceConstants.OPT_IN_OUT_IN_APP_MESSAGES)
    }

    /**
     * @description This method is used to set the user location.
     * @param location - the user location, with latitude and longitude set
     */
    fun setUserLocation(location: Location?) {
        if (location != null) {
            if (::mSmtInfo.isInitialized) {
                mSmtInfo.mDeviceInfo?.latitude = location.latitude.toString()
                mSmtInfo.mDeviceInfo?.longitude = location.longitude.toString()
            }
            mPreferences.setString(SMTPreferenceConstants.SMT_LAST_KNOWN_LATITUDE, location.latitude.toString())
            mPreferences.setString(SMTPreferenceConstants.SMT_LAST_KNOWN_LONGITUDE, location.longitude.toString())
        } else {
            SMTLogger.v(TAG, "Location is null.")
        }
    }

    /**
     * This method is used to get device unique identifier.
     * @return String - the device unique id.
     */
    fun getDeviceUniqueId(): String {
        return mSmtInfo.mDeviceInfo?.guid ?: ""
    }

    /**
     * This method is used to get Unread push notifications.
     * @return Int - the count of unread push notification.
     */
    /*fun getDBadgeCount(): Int {
        return 10
    }*/

    /**
    @brief This method is used to get device token.


    @return String - the device token.
     */
    fun getDevicePushToken(): String {
        return mPreferences.getString(SMTPreferenceConstants.PUSH_TOKEN_CURRENT)
    }

    /**
     * To set Opt In and Out option to user
     * if true means user opted in for Tracking
     * if false means user opted out for Tracking
     */
    fun optTracking(isOpted: Boolean) {
        val currentValue = mPreferences.getBoolean(SMTPreferenceConstants.OPT_IN_OUT_TRACKING)

        if (currentValue != isOpted) {
            mPreferences.setBoolean(SMTPreferenceConstants.OPT_IN_OUT_TRACKING, isOpted)
            context.get()?.let { ctx ->
                if (isOpted) {
                    SMTEventRecorder.getInstance(ctx).recordEvent(SMTEventId.EVENT_USER_TRACKING_OPT_IN,
                            SMTEventId.getEventName(SMTEventId.EVENT_USER_TRACKING_OPT_IN), null, SMTEventType.EVENT_TYPE_SYSTEM)
                } else {
                    SMTEventRecorder.getInstance(ctx).recordEvent(SMTEventId.EVENT_USER_TRACKING_OPT_OUT,
                            SMTEventId.getEventName(SMTEventId.EVENT_USER_TRACKING_OPT_OUT), null, SMTEventType.EVENT_TYPE_SYSTEM)

                    // Sync this event before events events are prevented from sync
                    context.get()?.let {
                        SMTWorkerScheduler.getInstance().checkStatusAndScheduleEventWorker(it)
                    }

                }
            }

        }
    }

    /**
     * To set Opt In adn Out option to user
     * if true means user opted in for PN
     * if false means user opted out for PN
     */
    fun optPushNotification(isOpted: Boolean) {
        val currentValue = mPreferences.getBoolean(SMTPreferenceConstants.OPT_IN_OUT_PUSH_NOTIFICATION)

        if (currentValue != isOpted) {
            context.get()?.let { ctx ->
                mPreferences.setBoolean(SMTPreferenceConstants.OPT_IN_OUT_PUSH_NOTIFICATION, isOpted)
                if (isOpted) {
                    SMTEventRecorder.getInstance(ctx).recordEvent(SMTEventId.EVENT_USER_PN_OPT_IN,
                            SMTEventId.getEventName(SMTEventId.EVENT_USER_PN_OPT_IN), null, SMTEventType.EVENT_TYPE_SYSTEM)
                } else {
                    SMTEventRecorder.getInstance(ctx).recordEvent(SMTEventId.EVENT_USER_PN_OPT_OUT,
                            SMTEventId.getEventName(SMTEventId.EVENT_USER_PN_OPT_OUT), null, SMTEventType.EVENT_TYPE_SYSTEM)
                }
            }

        }
    }

    /**
     * To set Opt In adn Out option to user
     * if true means user opted in for In App
     * if false means user opted out for In App
     */
    fun optInAppMessage(isOpted: Boolean) {
        val currentValue = mPreferences.getBoolean(SMTPreferenceConstants.OPT_IN_OUT_IN_APP_MESSAGES)

        context.get()?.let { ctx ->
            if (currentValue != isOpted) {
                mPreferences.setBoolean(SMTPreferenceConstants.OPT_IN_OUT_IN_APP_MESSAGES, isOpted)
                if (isOpted) {
                    SMTEventRecorder.getInstance(ctx).recordEvent(SMTEventId.EVENT_USER_INAPP_OPT_IN,
                            SMTEventId.getEventName(SMTEventId.EVENT_USER_INAPP_OPT_IN), null, SMTEventType.EVENT_TYPE_SYSTEM)
                } else {
                    SMTEventRecorder.getInstance(ctx).recordEvent(SMTEventId.EVENT_USER_INAPP_OPT_OUT,
                            SMTEventId.getEventName(SMTEventId.EVENT_USER_INAPP_OPT_OUT), null, SMTEventType.EVENT_TYPE_SYSTEM)
                }
            }
        }


    }

    /**
     * This method would clear the user identity.
     */


    /**
     * This method would get the user identity.
     */
    fun getUserIdentity(): String {
        return mPreferences.getString(SMTPreferenceConstants.SMT_USER_IDENTITY)
    }

    /**
     * This method would set the user identity.
     * @param userIdentity - the user identity.
     */
    @Suppress("unused")
    fun setUserIdentity(userIdentity: String?) {
        if (!userIdentity.isNullOrEmpty()) {
            mSmartechHelper.deletePNAttribution(userIdentity)
            mPreferences.setString(SMTPreferenceConstants.SMT_USER_IDENTITY, userIdentity)
        } else {
            SMTLogger.w(TAG, Resources.getSystem().getString(R.string.identity_unavailable))
        }
        //we are fetching list and segment data on basis of identity
        callListAndSegmentAPI()
    }

    /**
     * This method would logout the user and clear identity on Smartech.
     * @param  clearUserIdentity - the user identity.
     */
    fun logoutAndClearUserIdentity(clearUserIdentity: Boolean) {

        val hashMap: HashMap<String, Any> = HashMap()
        hashMap[SMTEventParamKeys.SMT_CLEAR_IDENTITY] = clearUserIdentity

        context.get()?.let { ctx ->
            SMTEventRecorder.getInstance(ctx).recordEvent(SMTEventId.EVENT_USER_LOGGED_OUT,
                    SMTEventId.getEventName(SMTEventId.EVENT_USER_LOGGED_OUT), hashMap, SMTEventType.EVENT_TYPE_SYSTEM)
        }


        if (clearUserIdentity) {
            clearUserIdentity()
        }
    }

    fun clearUserIdentity() {
        // Replacing old identity with current identity
        mPreferences.setString(SMTPreferenceConstants.SMT_USER_OLD_IDENTITY, mPreferences.getString(SMTPreferenceConstants.SMT_USER_IDENTITY, ""))
        mPreferences.setString(SMTPreferenceConstants.SMT_USER_IDENTITY, "")

        //we are fetching list and segment data on basis of identity
        callListAndSegmentAPI()
    }


    /**
     * This method would login the user on Smartech.
     * @param userIdentity - the user identity.
     */
    fun login(userIdentity: String?) {
        if (!userIdentity.isNullOrEmpty()) {
            mSmartechHelper.deletePNAttribution(userIdentity)
            mPreferences.setString(SMTPreferenceConstants.SMT_USER_IDENTITY, userIdentity)
            context.get()?.let { ctx ->
                SMTEventRecorder.getInstance(ctx).recordEvent(SMTEventId.EVENT_USER_LOGGED_IN,
                        SMTEventId.getEventName(SMTEventId.EVENT_USER_LOGGED_IN), null, SMTEventType.EVENT_TYPE_SYSTEM)
            }
            //we are fetching list and segment data on basis of identity
            callListAndSegmentAPI()
        } else {
            SMTLogger.w(TAG, Resources.getSystem().getString(R.string.identity_unavailable))
        }
    }

    /**
     * Set the debug logging level
     * @discussion
     * Set using SMTLogLevel enum values or the corresponding int values.
     *  NONE - turns off all SDK logging.
     *  VERBOSE - enables verbose logging.
     *  DEBUG - enables debug logging.
     *  INFO - enables minimal information related to SDK .
     *  WARN - enables warning information related to SDK .
     *  ERROR - enables errorn information related to SDK .
     *  FATAL - enables crash information related to SDK .
     * @param level  the level to set
     */
    fun setDebugLevel(@SMTDebugLevel.Level level: Int) {
        val isLogEnabled = mPreferences.getBoolean(SMTPreferenceConstants.IS_LOG_ENABLED)
        if (isLogEnabled) {
            val logLevel = mPreferences.getInt(SMTPreferenceConstants.LOG_LEVEL, SMTDebugLevel.Level.NONE)
            mPreferences.setInt(SMTPreferenceConstants.SMT_DEBUG_LEVEL, logLevel)
            SMTLogger.setDebugLevel(logLevel)
        } else {
            mPreferences.setInt(SMTPreferenceConstants.SMT_DEBUG_LEVEL, level)
            SMTLogger.setDebugLevel(level)
        }
    }


    /**
     * To set notification option
     * @param notificationOptions options / settings of the notification
     */
    fun setNotificationOptions(notificationOptions: SMTNotificationOptions?) {
        context.get()?.let {
            SMTCommonUtility.storeNotificationSettings(it, notificationOptions)
        }
    }

    fun processEventsManually() {
        context.get()?.let {
            SMTEventBatchProcessor.getInstance(it).makeEventBatchProcessingApiCall()
        }
    }

    /**
     * Method to fetch inbox messages
     * @param inboxCallback - Callback to return the inbox message data
     * @param messageType - type of message required
     * @see SMTInboxMessageType for message types will be returned
     */
    @Suppress("unused")
    fun getInboxMessages(inboxCallback: SMTInboxCallback, @SMTInboxMessageType.Type messageType: Int) {
        if (mPreferences.getInt(SMTPreferenceConstants.SMT_MF_IS_AUTO_FETCH_INBOX_MESSAGE) == 0) {
            SMTInboxApiService.getInstance(context).makeAppInboxApiCall(inboxCallback, messageType)
        } else {
            inboxCallback.onInboxSuccess(SMTDataBaseService.getInstance(context).getInboxMessages(messageType))
        }
    }

    /**
     * Method to return inbox message count based on type
     * @param messageType - type of message
     * @see SMTInboxMessageType for different message types
     */
    @Suppress("unused")
    fun getInboxMessageCount(@SMTInboxMessageType.Type messageType: Int): Int {
        return SMTDataBaseService.getInstance(context).getInboxMessageCount(messageType)
    }

    /**
     * Method to return inbox message count based on type
     * @param trId - id of message
     * @return notification data model
     */
    @Suppress("unused")
    fun getInboxMessageById(trId: String): SMTNotificationData? {
        return SMTDataBaseService.getInstance(context).getInboxMessageById(trId)
    }

    /**
     * Method to return inbox message count based on type
     * @param trId - message id
     * @param deeplinkPath deelink path associated with the click
     * @param status - message status to update
     * @see SMTInboxMessageStatus for different message status
     */
    @Suppress("unused")
    fun updateInboxMessageStatus(trId: String, deeplinkPath: String, @SMTInboxMessageStatus.Status status: Int) {
        context.get()?.let {
            SMTEventRecorder.getInstance(it).recordInboxEvent(trId, deeplinkPath, status)
        }

    }

    /**
     * Method to display App Inbox
     * @param containerId - Container id in which the fragment has to be shown
     * @param activity - AppcompatActivity reference
     */
    @Suppress("unused")
    fun displayAppInbox(containerId: Int, activity: AppCompatActivity) {
        mSmartechHelper.handleInboxDisplay(containerId, activity)
    }

    /**
     * Method to pass notification data back to SDK from App
     * @param notificationData - Notification message received from FCM
     */
    @Suppress("unused")
    fun handlePushNotification(notificationData: String?): Boolean {
        return context.get()?.let {
            if (notificationData != null) {
                SMTPNHandler(SMTNotificationGeneratorProvider()).handleNotification(it, notificationData, SMTNotificationSourceType.NOTIFICATION_SOURCE_PN)
            } else {
                SMTLogger.i(TAG, "Notification payload is null.")
                false
            }
        } ?: false
    }

    /**
     * This method will be used to render notification by client.
     * It should be used only when client is using SMTNotificationListener to
     * have custom control over notification
     * @param notificationData Notification payload provided by client
     * @param notificationSource Source of notification
     */
    fun renderNotification(notificationData: String, notificationSource: Int) {
        context.get()?.let {
            SMTPNHandler(SMTNotificationGeneratorProvider()).renderNotification(it, notificationData, notificationSource)
        }
    }

    /**
     * Method to provide flexibility to user to concat attrparams with deeplink url
     * @param url : deeplink url
     * @param context : application context
     * @return url with attrparams
     */
    fun getSmartechAttributionURL(context: Context, url: String): String {

        var jsonHashMap = HashMap<String, String>()
        val attributeParams = SMTPreferenceHelper.getAppPreferenceInstance(context, null)
                .getString(SMTPreferenceConstants.SMT_ATTRIBUTION_PARAMS)
        var updatedURL = url

        // Parsing deeplink url to fetch query parameters
        // and comparing with attribution parameters.
        fun parseAttributesAndDeeplink(json: JSONObject, uri: Uri) {
            try {
                val keys = json.keys()
                while (keys.hasNext()) {
                    val key = keys.next()
                    jsonHashMap[key] = json.optString(key)
                }
                uri.queryParameterNames.forEach { queryParam ->
                    queryParam?.let { key ->
                        if (!jsonHashMap.containsKey(key)) {
                            jsonHashMap[key] = uri.getQueryParameter(key)
                        }
                    }
                }
            } catch (e: Exception) {
                SMTLogger.e(TAG, e.message.toString())
            }
        }

        return try {
            var updatedURLBuilder = Uri.parse(url).buildUpon().clearQuery()
            val deepLinkUri = Uri.parse(url)
            if (attributeParams.isNotEmpty() && deepLinkUri.scheme != null && deepLinkUri.host != null) {
                val json = JSONObject(attributeParams)
                parseAttributesAndDeeplink(json, deepLinkUri)
                for ((key, value) in jsonHashMap) {
                    updatedURLBuilder.appendQueryParameter(key, URLDecoder.decode(value, "UTF-8"))
                }
                updatedURLBuilder.build()
                updatedURL = URL(updatedURLBuilder.toString()).toString()
            }
            updatedURL
        } catch (e: Exception) {
            SMTLogger.e(TAG, e.message.toString())
            updatedURL
        }
    }

    /**
     * create the notification channel group from Smartech
     * <p/>
     * Use this method when implementing your own FCM/GCM handling mechanism. Refer to the
     * SDK documentation for usage scenarios and examples.
     *
     * @param groupId A String for setting the id of the notification channel group
     * @param groupName  A String for setting the name of the notification channel group
     */
    fun createNotificationChannelGroup(groupId: String, groupName: CharSequence) {
        context.get()?.let {
            mSmtNotificationChannelHelper.createNotificationChannelGroup(groupId, groupName)
        }
    }

    /**
     * After creating the notification builder this method create the notification channel
     * @param smtNotificationChannel A notification builder
     */
    fun createNotificationChannel(smtNotificationChannel: SMTNotificationChannel) {
        context.get()?.let {
            mSmtNotificationChannelHelper.createNotificationChannel(smtNotificationChannel)
        }
    }

    /**
     * Delete the notification channel from Smartech
     *
     *
     * Use this method when implementing your own FCM handling mechanism. Refer to the
     * SDK documentation for usage scenarios and examples.
     *
     * @param groupId A String for setting the id of the notification channel group
     */
    fun deleteNotificationChannelGroup(groupId: String) {
        context.get()?.let {
            mSmtNotificationChannelHelper.deleteNotificationChannelGroup(groupId)
        }
    }

    /**
     * Delete the notification channel from Smartech
     *
     *
     * Use this method when implementing your own FCM handling mechanism. Refer to the
     * SDK documentation for usage scenarios and examples.
     *
     * @param context A reference to an Android context
     * @param channelId A String for setting the id of the notification channel
     */
    fun deleteNotificationChannel(channelId: String) {
        context.get()?.let {
            mSmtNotificationChannelHelper.deleteNotificationChannel(channelId)
        }
    }

    /**
     * Returns all notification channel groups belonging to the calling app.
     */
    fun getAllNotificationChannelGroups(): List<NotificationChannelGroup> {
        return context.let {
            mSmtNotificationChannelHelper.ncGetAllNotificationChannelGroups()
        } ?: ArrayList()
    }

    /**
     * Returns all notification channels belonging to the calling package.
     */
    fun getAllNotificationChannels(): List<NotificationChannel> {
        return context.let {
            mSmtNotificationChannelHelper.ncGetAllNotificationChannels()
        } ?: ArrayList()
    }

    fun getInAppCustomHTMLListener(): InAppCustomHTMLListener? {
        return inAppCustomHTMLListener
    }

    /**
     * Custom HTML listener for providing control to user.
     *
     * @param inAppListener InAppCustomHTMLListener
     */
    fun setInAppCustomHTMLListener(listener: InAppCustomHTMLListener?) {
        inAppCustomHTMLListener = listener
    }

    /**
     *  This function will be used to set listener for Notification Click event
     *
     *  @param notificationClickListener instance of  SMTNotificationClickListener
     */
    fun setSMTNotificationClickListener(notificationClickListener: SMTNotificationClickListener?) {
        smtNotificationClickListener = notificationClickListener
    }

    /**
     *  This function will be used to return instance of SMTNotificationClickListener
     *
     *  @return SMTNotificationClickListener
     */
    fun getSMTNotificationClickListener(): SMTNotificationClickListener? {
        return smtNotificationClickListener
    }

    /**
     *  This method will set instance of SMTNotificationListener.
     *  @param listener Instance of SMTNotificationClickListener
     */
    fun setSMTNotificationListener(listener: SMTNotificationListener?) {
        this.smtNotificationListener = listener
    }

    /**
     *  This method will instance of notification listener
     *  @return smtNotificationListener
     */
    fun getSMTNotificationListener(): SMTNotificationListener? {
        return this.smtNotificationListener
    }

    /**
     *  This method will provide functionality to developer to
     *  manually send notification open event to SDK.
     *
     *  @param context app context
     *  @param notificationObject Notification payload in json format
     *  @param sourceType Source of notification
     */
    fun deliverNotificationEvent(notificationObject: JSONObject, @SMTNotificationSourceType.Source sourceType: Int) {
        context.get()?.let {
            try {
                SMTNotificationUtility.getInstance().processOpenAndDeliverNotificationEvents(it, notificationObject, sourceType, SMTNotificationEventType.DELIVERY)
            } catch (e: Exception) {
                SMTLogger.e(TAG, e.message.toString())
            }
        }
    }

    /**
     *  This method will provide functionality to developer to
     *  manually send notification open event to SDK.
     *
     *  @param context app context
     *  @param notificationObject Notification payload in json format
     *  @param sourceType Source of notification
     */
    fun openNotificationEvent(notificationObject: JSONObject, sourceType: Int) {
        context.get()?.let {
            try {
                SMTNotificationUtility.getInstance().processOpenAndDeliverNotificationEvents(it, notificationObject, sourceType, SMTNotificationEventType.OPEN)
            } catch (e: Exception) {
                SMTLogger.e(TAG, e.message.toString())
            }
        }

    }

    /**
     *  This function will check source of notification
     *  @param payload Notification payload
     *  @return Boolean Source of notification
     */
    fun isNotificationFromSmartech(payload: JSONObject?): Boolean {
        return try {
            SMTNotificationUtility.getInstance().checkNotificationSource(payload)
        } catch (e: Exception) {
            SMTLogger.e(TAG, e.localizedMessage)
            false
        }
    }

    /**
     *  This method will return SDK version.
     */
    fun getSDKVersion(): String {
        return try {
            mSmtInfo.mAppInfo?.smtSdkVersion ?: ""
        } catch (e: Exception) {
            SMTLogger.e(TAG, "Error while reading SDK version.")
            ""
        }
    }

    /**
     *  This method will return App id from the Manifest file.
     */
    fun getAppID(): String? {
        return try {
            mSmartechHelper.getAppID(context)
        } catch (e: Exception) {
            SMTLogger.e(TAG, "Error while reading App id.")
            ""
        }
    }

    /**
     *  This function will be used to set Advertisement id.
     */
    fun setDeviceAdvertiserId(id: String?) {
        mSmartechHelper.setDeviceAdvertiserId(id)
    }

    @Suppress("unused")
    fun setXiaomiPushToken(xiaomiToken: String?) {
        if (!xiaomiToken.isNullOrEmpty()) {
            mSmartechHelper.setDevicePushToken(xiaomiToken, SMTGWSource.XIAOMI)
        } else {
            SMTLogger.v(TAG, "Xiaomi token is null or empty.")
        }
    }

    /**
     * This method will return the Xiaomi token saved in the preference.
     **/
    @Suppress("unused")
    fun getXiaomiPushToken(): String {
        return mPreferences.getString(SMTPreferenceConstants.XIAOMI_TOKEN_CURRENT)
    }

    /**
     * This method will handle the notifications received from Xiaomi gateway.
     **/
    @Suppress("unused")
    fun handleXiaomiNotification(notificationData: String?): Boolean {
        return context.get()?.let {
            if (notificationData != null) {
                SMTPNHandler(SMTNotificationGeneratorProvider()).handleNotification(it, notificationData, SMTNotificationSourceType.NOTIFICATION_SOURCE_XIAOMI)
            } else {
                SMTLogger.i(TAG, "Notification payload is null.")
                false
            }
        } ?: false
    }
}