package com.netcore.android.utility

import android.content.Context
import android.content.pm.PackageManager
import android.os.Bundle
import com.netcore.android.SMTManifestKeys
import com.netcore.android.logger.SMTLogger
import com.netcore.android.preference.SMTPreferenceConstants
import com.netcore.android.preference.SMTPreferenceHelper
import java.lang.ref.WeakReference

/**
 * Copyright © 2019 Netcore. All rights reserved.
 *
 * Class that reads the configuration values from App manifest file
 *
 * @author Netcore
 * @version 1.0
 * @since 11-03-2019
 */
internal class SMTManifestInfo private constructor(val context: WeakReference<Context>) {

    private val TAG = SMTManifestInfo::class.java.simpleName

    var useADID = 0 // App Config for ADID usage in the app
    var appId: String? = null // App config for Smartech Application ID
    var isAutoFetchedLocation = 1 // To keep track whether SDK is allowed to fetch location or not
    var isAutoFecthInboxNotifications = 0
    var isNotificationListenerEnabled = 0 // To keep track whether notification listener is enabled or disabled
    var isSDKv2ConfigOnUpdateEnabled = 1 // To check whether we should read identity from Old SDK to new SDK

    companion object {
        @Volatile
        private var INSTANCE: SMTManifestInfo? = null

        /**
         * Getting instance of the class
         */
        fun getInstance(context: WeakReference<Context>): SMTManifestInfo =
                INSTANCE ?: synchronized(SMTManifestInfo::class.java) {
                    INSTANCE ?: buildInstance(context).also { INSTANCE = it }
                }

        private fun buildInstance(context: WeakReference<Context>): SMTManifestInfo {
            return SMTManifestInfo(context)
        }
    }

    init {
        readManifestConfig()
    }

    /**
     * Reads Manifest config values
     */
    private fun readManifestConfig() {
        var metaData: Bundle?
        try {
            context.get()?.let {
                val pm = it.packageManager
                val ai = pm.getApplicationInfo(it.packageName, PackageManager.GET_META_DATA)
                metaData = ai?.metaData

                useADID = getManifestIntValueForKey(metaData as Bundle, SMTManifestKeys.SMT_USE_AD_ID)
                appId = getManifestStringValueForKey(metaData as Bundle, SMTManifestKeys.SMT_APP_ID)
                isAutoFetchedLocation = getManifestIntValueForKey(metaData as Bundle, SMTManifestKeys.SMT_IS_AUTO_FETCHED_LOCATION)
                isAutoFecthInboxNotifications = getManifestIntValueForKey(metaData as Bundle, SMTManifestKeys.SMT_IS_AUTO_FETCH_INBOX_NOTIFICATIONS)
                isNotificationListenerEnabled = getManifestIntValueForKey(metaData as Bundle, SMTManifestKeys.SMT_IS_NOTIFICATION_LISTENER_ENABLED)
                isSDKv2ConfigOnUpdateEnabled = getManifestIntValueForKey(metaData as Bundle, SMTManifestKeys.SMT_SDK_V2_CONFIG_ON_UPDATE)
                val preference = SMTPreferenceHelper.getAppPreferenceInstance(it, null)
                preference.setInt(SMTPreferenceConstants.SMT_MF_IS_USE_AD_ID, useADID)
                preference.setString(SMTPreferenceConstants.SMT_MF_APP_ID, appId ?: "")
                preference.setInt(SMTPreferenceConstants.SMT_MF_IS_AUTO_FETCH_LOCATION, isAutoFetchedLocation)
                preference.setInt(SMTPreferenceConstants.SMT_MF_IS_AUTO_FETCH_INBOX_MESSAGE, isAutoFecthInboxNotifications)
                preference.setInt(SMTPreferenceConstants.SMT_MF_IS_NOTIFICATION_LISTENER_ENABLED, isNotificationListenerEnabled)
                preference.setInt(SMTPreferenceConstants.SMT_MF_SDK_V2_CONFIG_ON_UPDATE, isSDKv2ConfigOnUpdateEnabled)

                SMTLogger.i(TAG, "Smartech Manifest report AppId: $appId, ADID: $useADID, AutoFetchLocationEnabled: $isAutoFetchedLocation, NLEnabled: $isNotificationListenerEnabled")
            }
        } catch (e: Exception) {
            SMTLogger.e(TAG, "Error while reading manifest meta data: ")
        }
    }

    /**
     * Reads String app config values
     */
    private fun getManifestStringValueForKey(manifest: Bundle, name: String): String {
        return try {
            val value = manifest.get(name)
            value?.toString() ?: ""
        } catch (e: Exception) {
            SMTLogger.i(TAG, "No value for $name in manifest.")
            ""
        }
    }

    /**
     * Reads Int app config values
     */
    private fun getManifestIntValueForKey(manifest: Bundle, name: String): Int {
        return try {
            val value = manifest.get(name)
            value as Int
        } catch (e: Exception) {
            SMTLogger.i(TAG, "No value for $name in manifest.")
            if (SMTManifestKeys.SMT_SDK_V2_CONFIG_ON_UPDATE == name) {
                1
            } else {
                0
            }
        }
    }

    /**
     * Reads Boolean app config values
     */
    private fun getManifestBooleanValueForKey(manifest: Bundle, name: String): Boolean {
        return try {
            val value = manifest.get(name)
            value as Boolean
        } catch (e: Exception) {
            SMTLogger.e(TAG, "No value for $name in manifest.")
            false
        }
    }


}