@file:Suppress("unused")

package com.netcore.android.utility

import android.content.ContentResolver
import android.content.Context
import android.content.Intent
import android.content.pm.PackageManager
import android.content.res.Configuration
import android.content.res.Resources
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.net.ConnectivityManager
import android.net.Uri
import android.os.Build
import android.os.Bundle
import android.os.Environment
import androidx.annotation.VisibleForTesting
import androidx.core.app.NotificationManagerCompat
import androidx.core.content.ContextCompat
import com.netcore.android.*
import com.netcore.android.event.SMTEventId
import com.netcore.android.event.SMTEventId.Companion.EVENT_LOCATION_FETCH_ENABLED
import com.netcore.android.event.SMTEventId.Companion.EVENT_PN_TOKEN_GENERATED
import com.netcore.android.event.SMTEventId.Companion.EVENT_USER_TRACKING_OPT_OUT
import com.netcore.android.event.SMTEventRecorder
import com.netcore.android.event.SMTEventType
import com.netcore.android.logger.SMTLogger
import com.netcore.android.network.models.SMTSdkInitializeResponse
import com.netcore.android.notification.SMTNotificationConstants
import com.netcore.android.notification.SMTNotificationOptions
import com.netcore.android.preference.SMTGUIDPreferenceHelper
import com.netcore.android.preference.SMTPreferenceConstants
import com.netcore.android.preference.SMTPreferenceHelper
import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject
import java.io.*
import java.lang.ref.WeakReference
import java.text.SimpleDateFormat
import java.util.*
import java.util.concurrent.TimeUnit
import kotlin.collections.HashMap


/**
 * Copyright © 2019 Netcore. All rights reserved.
 *
 * Singleton common utility class
 *
 * @author Netcore
 * @version 1.0
 * @since 26-02-2019
 */

internal object SMTCommonUtility {

    private val TAG = SMTCommonUtility::class.java.simpleName
    val Int.toPx: Int
        get() = (this * Resources.getSystem().displayMetrics.density).toInt()

    val Int.toDp: Int
        get() = (this / Resources.getSystem().displayMetrics.density).toInt()

    /**
     * Provides application icon ID
     * @param context - App context
     * @return Int - Icon ID
     */
    fun getAppIconId(context: Context): Int {
        val pm = context.packageManager
        val applicationInfo = pm.getApplicationInfo(context.packageName, PackageManager.GET_META_DATA)
        return applicationInfo.icon
    }

    /**
     * Provides bitmap of application icon
     * @param context - App context
     * @return Bitmap - Icon Bitmap
     */
    fun getAppIconBitmap(context: Context): Bitmap {
        val pm = context.packageManager
        val applicationInfo = pm.getApplicationInfo(context.packageName, PackageManager.GET_META_DATA)
        val resources = pm.getResourcesForApplication(applicationInfo)
        val appIconResId = applicationInfo.icon
        var appIconBitmap = BitmapFactory.decodeResource(resources, appIconResId)

        val drawable = pm.getApplicationIcon(context.packageName)

        var width = 1
        var height = 1
        drawable?.let {
            width = it.intrinsicWidth
            height = it.intrinsicHeight
        }

        if (appIconBitmap == null) {
            appIconBitmap = Bitmap.createBitmap(width, height, Bitmap.Config.ARGB_8888)
        }
        return appIconBitmap
    }

    internal fun getBitmapFromResId(context: Context, resourceId: Int): Bitmap {
        var appIconBitmap = BitmapFactory.decodeResource(context.resources, resourceId)

        val pm = context.packageManager
        val drawable = pm.getApplicationIcon(context.packageName)

        var width = 1
        var height = 1
        drawable?.let {
            width = it.intrinsicWidth
            height = it.intrinsicHeight
        }

        if (appIconBitmap == null) {
            appIconBitmap = Bitmap.createBitmap(width, height, Bitmap.Config.ARGB_8888)
        }
        return appIconBitmap
    }

    /**
     * Used to check network is available before making api call
     * @param context App context
     * @return status of connection
     */
    fun isNetworkAvailable(context: Context): Boolean {
        val connMgr = context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager
        val networkInfo = connMgr.activeNetworkInfo
        return networkInfo != null && networkInfo.isConnected
    }

    /**
     * Provides application name
     * @param context - App context
     * @return String - Application Name
     */
    fun getApplicationName(context: Context): String {
        val applicationInfo = context.applicationInfo
        val stringId = applicationInfo.labelRes
        return if (stringId == 0) applicationInfo.nonLocalizedLabel.toString() else context.getString(stringId)
    }

    /**
     * Helps to delete a file from file system
     * @param path - File path to be deleted
     */
    fun deleteFile(path: String) {
        val fdelete = File(path)
        if (fdelete.exists()) {
            if (fdelete.delete()) {
                SMTLogger.v(TAG, "File delete success :- $path")
            } else {
                SMTLogger.v(TAG, "File delete failed :- $path")
            }
        }
    }

    /**
     * Helps to load image files from local storage
     * @param path - Path of file to be loaded from file system
     * @return Bitmap - Bitmap image loaded from local storage
     */
    fun loadImageFromLocalStorage(path: String): Bitmap? {
        var bitmap: Bitmap? = null

        try {
            val f = File(path, "")
            bitmap = BitmapFactory.decodeStream(FileInputStream(f))
        } catch (e: Exception) {
            SMTLogger.e(TAG, "Error while loading file path:$path: error is : $e")
        }
        return bitmap

    }

    /**
     * Helps in deeplink broadcasting i.e. received by InAPp or Notificaiton payload or through Inbox
     * @param context - App context
     * @param itemLink - Deeplink path
     * @param customPayload - custom payload received in the notification
     */
    fun handleNotificationClick(context: Context, itemLink: String?, customPayload: HashMap<String, Any>?) {
        if (SMTActivityLifecycleCallback.getInstance().isAppInForeground()) {
            broadcastNotificationClick(context, itemLink, customPayload)
        } else {
            SMTPreferenceHelper.getAppPreferenceInstance(context, null).setBoolean(SMTPreferenceConstants.IS_LAUNCHED_FROM_NOTIFICATION, true)
            launchAppOnNotificationClick(context, itemLink, customPayload)
        }
    }

    /**
     * Launches the App and passed the required params in the intent
     * @param context - App context
     * @param itemLink - Deeplink path
     * @param customPayload - custom payload received in the notification
     */
    private fun launchAppOnNotificationClick(context: Context, itemLink: String?, customPayload: HashMap<String, Any>?) {

        val launchIntent = context.packageManager?.getLaunchIntentForPackage(context.packageName)
        launchIntent?.flags = Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_CLEAR_TASK

        val bundle = Bundle()
        itemLink?.let {
            bundle.putString(SMTBundleKeys.SMT_BUNDLE_KEY_CLICK_DEEPLINK, it)
        }

        customPayload?.let {
            val payload = JSONObject(it).toString()
            bundle.putString(SMTBundleKeys.SMT_BUNDLE_KEY_CLICK_CUSTOM_PAYLOAD, payload)
        }

        launchIntent?.putExtras(bundle)

        context.startActivity(launchIntent)
    }

    /**
     * Broadcasts the required params in the intent
     * @param context - App context
     * @param itemLink - Deeplink path
     * @param customPayload - custom payload received in the notification
     */
    private fun broadcastNotificationClick(context: Context, itemLink: String?, customPayload: HashMap<String, Any>?) {
        // not handling the deeplink just passing the info to registered receiver
        val intent = Intent(SMTConfigConstants.SMT_BROADCAST_EVENT_PN_INBOX_CLICK)

        val bundle = Bundle()
        itemLink?.let {
            bundle.putString(SMTBundleKeys.SMT_BUNDLE_KEY_CLICK_DEEPLINK, it)
        }

        customPayload?.let {
            val payload = JSONObject(it).toString()
            bundle.putString(SMTBundleKeys.SMT_BUNDLE_KEY_CLICK_CUSTOM_PAYLOAD, payload)
        }

        intent.putExtras(bundle)

        context.sendBroadcast(intent)

        // check if the SMTNotificationClickListener is active then send
        // intent via the SMTNotificationClickListener
        val listener = Smartech.getInstance(WeakReference(context)).getSMTNotificationClickListener()
        listener?.let {
            it.onNotificationClick(intent)
        }
    }

    /**
     * Generates rando id based on Current System Time Stamp
     * @return Int - Returns the generated Random ID
     */
    fun getRandomId(): Int {
        val random = Random()
        return random.nextInt(100000)
    }

    /**
     * Converts JSON to HashMap
     * @param json Any object i.e. to be converted to HashMap
     * @return HashMap<String, Any>
     */
    fun jsonToMap(json: Any): HashMap<String, Any>? {

        if (json is JSONObject)
            return _jsonToMap_(json)
        else if (json is String) {
            val jsonObject = JSONObject(json)
            return _jsonToMap_(jsonObject)
        }
        return null
    }


    private fun _jsonToMap_(json: JSONObject): HashMap<String, Any> {
        var retMap = HashMap<String, Any>()

        if (json != JSONObject.NULL) {
            retMap = toMap(json)
        }
        return retMap
    }


    private fun toMap(object1: JSONObject): HashMap<String, Any> {
        val map = HashMap<String, Any>()

        val keysItr = object1.keys()
        while (keysItr.hasNext()) {
            val key = keysItr.next()
            var value = object1.get(key)

            if (value is JSONArray) {
                value = toList(value)
            } else if (value is JSONObject) {
                value = toMap(value)
            }
            map[key] = value
        }
        return map
    }


    private fun toList(array: JSONArray): List<Any> {
        val list = ArrayList<Any>()
        for (i in 0..(array.length() - 1)) {
            var value = array.get(i)
            if (value is JSONArray) {
                value = toList(value)
            } else if (value is JSONObject) {
                value = toMap(value)
            }
            list.add(value)
        }
        return list
    }

    /**
     * Converts JSON to HashMap
     * @param json String to be converted to Map object
     * @return HashMap<String, String>
     */
    /*fun jsonToMap(json: String): HashMap<String, Any> {

        val map = HashMap<String, Any>()
        val jObject = JSONObject(json)
        val keys = jObject.keys()

        while (keys.hasNext()) {
            val key = keys.next()
            val value = jObject.getString(key)
            map[key] = value

        }
        return map
    }*/


    /**
     *  This method will be used to check Notifications permission status and if it's
     *  changed it wll record permission.
     */
    internal fun checkAndRecordNotificationPermissionStatus(context: Context) {

        val mPreferences = SMTPreferenceHelper.getAppPreferenceInstance(context, null)
        val currentPermission = areNotificationsEnabled(context)
        // If permission stored then check if current status is different or not
        // if it is different then record it

        val storedPermissionStatus = mPreferences.getBoolean(SMTPreferenceConstants.SMT_NOTIFICATION_PERMISSION)
        if (currentPermission != storedPermissionStatus) {
            recordNotificationPermission(context, currentPermission)
        }

        mPreferences.setBoolean(SMTPreferenceConstants.SMT_NOTIFICATION_PERMISSION, currentPermission)

    }

    private fun recordNotificationPermission(context: Context, currentPermission: Boolean) {
        val eventId: Int
        val eventName: String
        if (currentPermission) {
            eventId = SMTEventId.EVENT_USER_ENABLED_PN
            eventName = SMTEventId.getEventName(SMTEventId.EVENT_USER_ENABLED_PN)
        } else {
            eventId = SMTEventId.EVENT_USER_DISABLED_PN
            eventName = SMTEventId.getEventName(SMTEventId.EVENT_USER_DISABLED_PN)
        }
        SMTEventRecorder.getInstance(context).recordEvent(eventId,
                eventName, null, SMTEventType.EVENT_TYPE_SYSTEM)
    }

    /**
     * checks if any of the following param changed,
     * OS Version, App Version, SDK Version, Network Change, Language Change, ADID Change
     * if yes then send UDD param in the sdk Initialise request as true
     */
    internal fun checkIfDeviceDetailChanged(smtInfo: SMTInfo, context: Context): Boolean {

        val preference = SMTPreferenceHelper.getAppPreferenceInstance(context, null)
        updateDeviceInfo(smtInfo, context)

        if (preference.getString(SMTPreferenceConstants.SMT_OS_VERSION, "") != smtInfo.mDeviceInfo?.osVersion) {
            SMTLogger.i(TAG, "device detail updated for OS version " +
                    "Existing os version is: ${preference.getString(SMTPreferenceConstants.SMT_OS_VERSION)} " +
                    " new os version is : ${smtInfo.mDeviceInfo?.osVersion}")
            return true
        }
        if (preference.getString(SMTPreferenceConstants.SMT_CARRIER, "") != smtInfo.mNetworkInfo?.carrier) {
            SMTLogger.i(TAG, "device detail updated for Carrier " +
                    "Existing Carrier is: ${preference.getString(SMTPreferenceConstants.SMT_CARRIER)} " +
                    " new Carrier is : ${smtInfo.mNetworkInfo?.carrier}")
            return true
        }
        if (preference.getString(SMTPreferenceConstants.SMT_DEVICE_LOCALE, "") != smtInfo.mDeviceInfo?.deviceLocale) {
            SMTLogger.i(TAG, "device detail updated for Locale " +
                    "Existing Locale is: ${preference.getString(SMTPreferenceConstants.SMT_DEVICE_LOCALE)} " +
                    " new Locale is : ${smtInfo.mDeviceInfo?.deviceLocale}")
            return true
        }
        if (preference.getString(SMTPreferenceConstants.SMT_ADID, "") != smtInfo.mDeviceInfo?.advertiserId ?: "") {
            SMTLogger.i(TAG, "device detail updated for ADID " +
                    "Existing ADID is: ${preference.getString(SMTPreferenceConstants.SMT_ADID)} " +
                    " new ADID is : ${smtInfo.mDeviceInfo?.advertiserId}")
            return true
        }
        if (preference.getString(SMTPreferenceConstants.SMT_TIMEZONE, "") != smtInfo.mDeviceInfo?.timeZone ?: "") {
            SMTLogger.i(TAG, "device detail updated for SMT_TIMEZONE " +
                    "Existing SMT_TIMEZONE is: ${preference.getString(SMTPreferenceConstants.SMT_TIMEZONE)} " +
                    " new SMT_TIMEZONE is : ${smtInfo.mDeviceInfo?.timeZone}")
            return true
        }
        return false
    }

    /**
     *  This function will save  device details and it will be used to detect
     *  device update event.
     */
    private fun updateDeviceInfo(smtInfo: SMTInfo, context: Context) {
        val preference = SMTPreferenceHelper.getAppPreferenceInstance(context, null)

        if (preference.getString(SMTPreferenceConstants.SMT_OS_VERSION, "").isEmpty()) {
            preference.setString(SMTPreferenceConstants.SMT_OS_VERSION, smtInfo.mDeviceInfo?.osVersion
                    ?: "")
        }
        if (preference.getString(SMTPreferenceConstants.SMT_CARRIER, "").isEmpty()) {
            preference.setString(SMTPreferenceConstants.SMT_CARRIER, smtInfo.mNetworkInfo?.carrier
                    ?: "")
        }
        if (preference.getString(SMTPreferenceConstants.SMT_DEVICE_LOCALE, "").isEmpty()) {
            preference.setString(SMTPreferenceConstants.SMT_DEVICE_LOCALE, smtInfo.mDeviceInfo?.deviceLocale
                    ?: "")
        }
        if (preference.getString(SMTPreferenceConstants.SMT_TIMEZONE, "").isEmpty()) {
            preference.setString(SMTPreferenceConstants.SMT_TIMEZONE, smtInfo.mDeviceInfo?.timeZone
                    ?: "")
        }
    }

    /*
    * Check if version is marshmallow and above.
    * Used in deciding to ask runtime permission
    * */
    internal fun shouldCheckPermission(): Boolean {
        return Build.VERSION.SDK_INT >= Build.VERSION_CODES.M
    }

    /**
     * To check if the permission is previously granted or not
     */
    internal fun isPermissionGranted(context: Context, permission: String): Boolean {
        val permissionResult = ContextCompat.checkSelfPermission(context, permission)
        if (permissionResult == PackageManager.PERMISSION_GRANTED) {
            return true
        }
        return false
    }

    /**
     * To check if device is tablet or phone
     */
    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    fun isTablet(ctx: Context): Boolean {
        return ctx.resources.configuration.screenLayout and Configuration.SCREENLAYOUT_SIZE_MASK >= Configuration.SCREENLAYOUT_SIZE_LARGE
    }

    /**
     * Provides BOD value as per Mobile / Tablet
     */
    internal fun getBOD(context: Context): Int {
        return if (isTablet(context)) {
            SMTConfigConstants.BOD_FOR_TABLET
        } else {
            SMTConfigConstants.BOD_FOR_PHONE
        }
    }

    /**
     * Method used to convert Smartech settings POJO to JSON
     * NOTE As of now let it the method present
     * if not required then delete
    //     * @param smartechSettings - Smartech settings POJO class
     */
    /*internal fun convertSmartTechSettingsToJson(smartechSettings: SMTSdkInitializeResponse.SmartTechSettings): JSONObject {
        val hashMap = HashMap<String, Any>()
        hashMap[SmartechSettingsKeys.BATCH_INTERVAL] = smartechSettings.batchInterval
        hashMap[SmartechSettingsKeys.BATCH_SIZE] = smartechSettings.batchSize
        hashMap[SmartechSettingsKeys.FETCH_LOCATION] = smartechSettings.fetchLocation
        hashMap[SmartechSettingsKeys.PUSH_AMP_ENABLED] = smartechSettings.paEnabled
        hashMap[SmartechSettingsKeys.PUSH_AMP_INTERVAL] = smartechSettings.paInterval
        hashMap[SmartechSettingsKeys.PANEL_ACTIVE] = smartechSettings.panelActive
        hashMap[SmartechSettingsKeys.SDK_ACTIVE] = smartechSettings.sdkActive
        hashMap[SmartechSettingsKeys.SESSION_INTERVAL] = smartechSettings.sessionInterval
        return JSONObject(hashMap)
    }*/

    /**
     * Calcualte PushAmp next schedule time
     * @param context app context
     * @return time in milli second
     */
    internal fun getPushAmpNextScheduleTime(context: Context): Long {

        val count = SMTPreferenceHelper.getAppPreferenceInstance(context, null)
                .getInt(SMTPreferenceConstants.PUSH_AMP_TASK_COUNT, 0)

        val paInterVal = SMTPreferenceHelper.getAppPreferenceInstance(context, null)
                .getInt(SMTPreferenceConstants.PUSH_AMP_INTERVAL) * 60 * 1000// converting into millis


        // update count and store it in SP
        SMTPreferenceHelper.getAppPreferenceInstance(context, null)
                .setInt(SMTPreferenceConstants.PUSH_AMP_TASK_COUNT, (count + 1))

        // Exponentially increase the PAMP time
        var interval = (Math.pow(2.toDouble(), count.toDouble()) * paInterVal).toLong()

        // if the time is more than max push amp interval then reset it
        if (interval > SMTConfigConstants.MAX_CAP_PAMP_INTERVAL) {
            SMTPreferenceHelper.getAppPreferenceInstance(context, null)
                    .setInt(SMTPreferenceConstants.PUSH_AMP_TASK_COUNT, 0)
            SMTPreferenceHelper.getAppPreferenceInstance(context, null).setInt(SMTPreferenceConstants.JOB_SCHEDULER_JOB_ID, 1)
            interval = paInterVal.toLong()
        }

        return interval
    }

    /**
     * Checks whether notification is enabled / disabled from device settings
     * @param context app context
     */
    internal fun areNotificationsEnabled(context: Context): Boolean {
        return NotificationManagerCompat.from(context).areNotificationsEnabled()
    }

    /**
     * To retrieve the attribution params from the deeplink
     * and also store in preference for future use in events
     *
     * @param context app context
     * @param path deeplink path
     *
     * @return map with attribution parameters
     */
    fun updateAttributionParams(context: Context, path: String, attributionMap: HashMap<String, String> = HashMap()): HashMap<String, String> {

        //val path = "https://www.example.com/application.html?utm_campaign=repeat-order&utm_medium=payment&utm_source=sms&__sta=FJUVVHHQTTQVY%7CVI&__stm_medium=apn&__stm_source=smartech&__stm_id=12"
        val listSMTAttributekeys = listOf(SMTNotificationConstants.NOTIF_ATTRIBUTION_ID, SMTNotificationConstants.NOTIF_ATTRIBUTION_MEDIUM, SMTNotificationConstants.NOTIF_ATTRIBUTION_SOURCE, SMTNotificationConstants.NOTIF_ATTRIBUTION_STA)

        try {
            /**
             *  As per discussion with product team, we don't need to send utm parameters in
             *  other parameter node
             */
            /* val uri = Uri.parse(path)
             val attrSet = uri.queryParameterNames
             val objOther = JSONObject()
             val iterator = attrSet.iterator()

             while (iterator.hasNext()) {
                 val attrKey = iterator.next()
                 if (attrKey !in listSMTAttributekeys) {
                     objOther.put(attrKey, uri.getQueryParameter(attrKey))
                 }
             }
             attributionMap[SMTNotificationConstants.NOTIF_ATTRIBUTION_OTHER] = objOther.toString()
             */
            // comparing app identity with notification
            val dropAttribute = compareIdentity(context, attributionMap)
            SMTLogger.v(TAG, "Drop Attribute is $dropAttribute")
            val attrParams = JSONObject(attributionMap).toString()

            if (!dropAttribute) {
                // Add attributes to shared preference only if identity is same
                SMTPreferenceHelper.getAppPreferenceInstance(context, null)
                        .setString(SMTPreferenceConstants.SMT_ATTRIBUTION_PARAMS, attrParams)
            }

        } catch (e: Exception) {
            SMTLogger.e(TAG, "Error while fetching attribution param : $e")
        }
        return attributionMap
    }

    /**
     * retrieves notification option from shared preference and returns
     * @param context App context
     * @return SMTNotificationOptions notification options/settings
     */
    internal fun getNotificationOptions(context: Context): SMTNotificationOptions {
        val settingsString = SMTPreferenceHelper.getAppPreferenceInstance(context, null)
                .getString(SMTPreferenceConstants.SMT_NOTIFICATION_OPTION)
        var options = SMTNotificationOptions(context)
        // If options not stored then store default one first and return the same
        // else retrieve from SP and parse it and return back
        // Use case suppose the app developer does not set Notification Option
        // in that case store the default options
        if (settingsString.isEmpty()) {
            storeNotificationSettings(context, options)
        } else {
            try {
                val jsonObject = JSONObject(settingsString)
                options.smallIconId = jsonObject.getInt(SmartechNotificationOptionKeys.SMALL_ICON_ID)
                options.smallIconTransparentId = jsonObject.optInt(SmartechNotificationOptionKeys.SMALL_ICON_TRANSPARENT_ID)
                options.largeIconId = jsonObject.optInt(SmartechNotificationOptionKeys.LARGE_ICON_ID)
                options.placeHolderIcon = jsonObject.optInt(SmartechNotificationOptionKeys.PLACE_HOLDER_ICON)
                options.transparentIconBgColor = jsonObject.optString(SmartechNotificationOptionKeys.TRANS_ICON_BG_COLOR)
                //options.soundId = Uri.parse(jsonObject.optString(SmartechNotificationOptionKeys.SOUND_URI))
            } catch (e: Exception) {
                SMTLogger.e(TAG, e.message.toString())
            }
        }
        return options
    }

    /**
     * Stores notification settings set by User
     * If does not set then default settings will be stored
     */
    internal fun storeNotificationSettings(context: Context, notificationOptions: SMTNotificationOptions?) {
        var settings = notificationOptions
        if (settings == null) {
            settings = SMTNotificationOptions(context)
        }
        val map = hashMapOf<String, Any>()
        map[SmartechNotificationOptionKeys.LARGE_ICON_ID] = settings.largeIconId
        map[SmartechNotificationOptionKeys.SMALL_ICON_ID] = settings.smallIconId
        map[SmartechNotificationOptionKeys.SMALL_ICON_TRANSPARENT_ID] = settings.smallIconTransparentId
        map[SmartechNotificationOptionKeys.PLACE_HOLDER_ICON] = settings.placeHolderIcon
        map[SmartechNotificationOptionKeys.TRANS_ICON_BG_COLOR] = settings.transparentIconBgColor
        // map[SmartechNotificationOptionKeys.SOUND_URI] = settings.soundId.toString()

        val settingsString = JSONObject(map).toString()

        SMTPreferenceHelper.getAppPreferenceInstance(context, null)
                .setString(SMTPreferenceConstants.SMT_NOTIFICATION_OPTION, settingsString)
    }

    /**
     * Method to write ascii text characters to file on SD card. Note that you must add a
     * WRITE_EXTERNAL_STORAGE permission to the manifest file or this method will throw
     * a FileNotFound Exception because you won't have write permission.
     */
    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    fun writeToSDFile(value: String) {

        // Find the root of the external storage.
        // See http://developer.android.com/guide/topics/data/data-  storage.html#filesExternal

        val root = Environment.getExternalStorageDirectory()

        // See http://stackoverflow.com/questions/3551821/android-write-to-sd-card-folder

        val dir = File(root.absolutePath + SMTConfigConstants.EXTERNAL_FILE_DIR)

        if (!dir.exists()) {
            dir.mkdirs()
        }

        val file = File(dir, SMTConfigConstants.EXTERNAL_FILE_NAME)

        try {
            val f = FileOutputStream(file)
            val pw = PrintWriter(f)
            pw.println(value)
            pw.flush()
            pw.close()
            f.close()
        } catch (e: FileNotFoundException) {
            SMTLogger.e(TAG, e.message.toString())
            SMTLogger.i(TAG, "******* File not found. Did you  add a WRITE_EXTERNAL_STORAGE permission to the manifest? : $e")
        } catch (e: IOException) {
            SMTLogger.i(TAG, "******* File not found. Did you add a WRITE_EXTERNAL_STORAGE permission to the manifest? : $e")
            SMTLogger.e(TAG, e.message.toString())
        } catch (e: Exception) {
            SMTLogger.e(TAG, e.message.toString())
        }
    }

    /** Method to read in a text file placed in the res/raw directory of the application. The
     * method reads in all lines of the file sequentially.  */

    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    fun readFromSDFile(): String {

        val root = android.os.Environment.getExternalStorageDirectory()

        // See http://stackoverflow.com/questions/3551821/android-write-to-sd-card-folder

        val dir = File(root.absolutePath + SMTConfigConstants.EXTERNAL_FILE_DIR)

        if (!dir.exists()) return ""

        val file = File(dir, SMTConfigConstants.EXTERNAL_FILE_NAME)
        val sb = StringBuffer()

        if (file.exists()) {
            try {
                val fis = FileInputStream(file)
                val dis = DataInputStream(fis)
                val reader = dis.bufferedReader()
                val iterator = reader.lineSequence().iterator()
                while (iterator.hasNext()) {
                    val line = iterator.next()
                    sb.append(line)
                }
                reader.close()
            } catch (e: IOException) {
                SMTLogger.e(TAG, e.message.toString())
            } catch (e: java.lang.Exception) {
                SMTLogger.e(TAG, e.message.toString())
            }
        }
        return sb.toString()
    }

    /**
     * Method to check whether external media available and writable. This is adapted from
     * https://developer.android.com/training/data-storage/files.html#WriteExternalStorage
     */
    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    fun isExternalStorageWritable(): Boolean {
        return Environment.getExternalStorageState() == Environment.MEDIA_MOUNTED
    }

    /**
     * Method to check whether external media available and readable. This is adapted from
     * https://developer.android.com/training/data-storage/files.html#WriteExternalStorage
     */
    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    fun isExternalStorageReadable(): Boolean {
        return Environment.getExternalStorageState() in
                setOf(Environment.MEDIA_MOUNTED, Environment.MEDIA_MOUNTED_READ_ONLY)
    }

    private fun readGUIDFromSharedPref(context: Context): String {
        return SMTGUIDPreferenceHelper.getAppPreferenceInstance(context, null)
                .getString(SMTPreferenceConstants.SMT_GUID, "")
    }

    /**
     * To retrieve stored GUID, if not then create a new one and stores
     */
    internal fun getStoredGUID(context: Context): String {
        val storedGUID = readGUIDFromSharedPref(context)
        return if (storedGUID.isNotEmpty()) {
            SMTPreferenceHelper.getAppPreferenceInstance(context, null)
                    .setBoolean(SMTPreferenceConstants.IS_SMT_GUID_STORED_PREVIOUSLY, true)
            storedGUID
        } else {
            val newGuid = UUID.randomUUID().toString()
            SMTPreferenceHelper.getAppPreferenceInstance(context, null)
                    .setBoolean(SMTPreferenceConstants.IS_SMT_GUID_STORED_PREVIOUSLY, false)
            SMTGUIDPreferenceHelper.getAppPreferenceInstance(context, null)
                    .setString(SMTPreferenceConstants.SMT_GUID, newGuid)
            newGuid
        }

        // Old Code
        /*  // check is permission required to retrieve GUID
          if (SMTCommonUtility.shouldCheckPermission()) {
              // checks if permission granted
              // if not the read from shared preference
              //println("------------1" + isPermissionGranted(context, SMTConfigConstants.READ_STORAGE_PERMISSION_MF_KEY))
              return if (SMTCommonUtility.isPermissionGranted(context, SMTConfigConstants.READ_STORAGE_PERMISSION_MF_KEY)) {
                  // check if guid already stored in the external file
                  // if not then read from Shared preference
                  //println("------------2" + isExternalStorageReadable())
                  if (SMTCommonUtility.isExternalStorageReadable()) {
                      readGuidFromExternalStorage(context)
                  } else {
                      readGuidFromSharedPrefernce(context)
                  }

              } else {
                  readGuidFromSharedPrefernce(context)
              }
          } else {
              // checks if external storage available
              // if not the read from shared preference
              return if (SMTCommonUtility.isExternalStorageReadable()) {
                  readGuidFromExternalStorage(context)
              } else {
                  readGuidFromSharedPrefernce(context)
              }
          }*/
    }

    /**
     * Reads GUID from external file
     */
//    private fun readGuidFromExternalStorage(context: Context): String {
//        var storedGuid = readFromSDFile()
//        if (storedGuid.isEmpty()) {
//            storedGuid = SMTGUIDPreferenceHelper.getAppPreferenceInstance(context, null)
//                    .getString(SMTPreferenceConstants.SMT_GUID)
//        } else {
//            SMTPreferenceHelper.getAppPreferenceInstance(context, null)
//                    .setBoolean(SMTPreferenceConstants.IS_SMT_GUID_STORED_PREVIOUSLY, true)
//
//        }
//        return if (storedGuid.isEmpty()) {
//            val newId = UUID.randomUUID().toString()
//            if (SMTCommonUtility.isExternalStorageWritable()
//                    && isPermissionGranted(context, SMTConfigConstants.WRITE_STORAGE_PERMISSION_MF_KEY)) {
//                SMTCommonUtility.writeToSDFile(newId)
//            }
//            SMTGUIDPreferenceHelper.getAppPreferenceInstance(context, null)
//                    .setString(SMTPreferenceConstants.SMT_GUID, newId)
//            newId
//        } else {
//            storedGuid
//        }
//    }

    /**
     * Reads GUID from Shared preference
     */
    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    fun readGuidFromSharedPrefernce(context: Context): String {
        val storedGuid = SMTGUIDPreferenceHelper.getAppPreferenceInstance(context, null)
                .getString(SMTPreferenceConstants.SMT_GUID)

        return if (storedGuid.isEmpty()) {
            val newGuid = UUID.randomUUID().toString()
            SMTGUIDPreferenceHelper.getAppPreferenceInstance(context, null)
                    .setString(SMTPreferenceConstants.SMT_GUID, newGuid)
            newGuid
        } else {
            storedGuid
        }
    }

    internal fun checkPanelAndSDKActiveStatus(context: Context): Boolean {
        val preference = SMTPreferenceHelper.getAppPreferenceInstance(context, null)
        val isSDKActive = preference.getBoolean(SMTPreferenceConstants.IS_SDK_ACTIVE, false)
        val isPanelActive = preference.getBoolean(SMTPreferenceConstants.IS_PANEL_ACTIVE, false)
        return (isSDKActive && isPanelActive)
    }

    /**
     * Checks if tracing enabled by user or not
     * If tracking not set then by default its disabled
     * else what ever user preference will be returned
     */
    internal fun checkIfTrackingAllowed(context: Context): Boolean {
        val preference = SMTPreferenceHelper.getAppPreferenceInstance(context, null)
        val isUserOptedIn = preference.getBoolean(SMTPreferenceConstants.OPT_IN_OUT_TRACKING, true)
        return isUserOptedIn && checkPanelAndSDKActiveStatus(context)
    }

    /**
     *  Method to check events id belongs to one of these event.
     *  It's used to log SMTEventId.EVENT_APP_INSTALLED, SMTEventId.EVENT_PN_TOKEN_GENERATED, SMTEventId.EVENT_LOCATION_FETCH_ENABLED
     * events even if SDK is inactve
     */
    internal fun eventsRepository(id: Int): Boolean {
        return when (id) {
            SMTEventId.EVENT_APP_INSTALLED, EVENT_PN_TOKEN_GENERATED, EVENT_LOCATION_FETCH_ENABLED, EVENT_USER_TRACKING_OPT_OUT -> true
            else -> false
        }
    }

    /**
     * Method to convert the time given in yyyy-mm-dd'T'HH:mm:ss format to timestamp
     */
    internal fun convertStringDatetoTimeStamp(mPublishedTimeStamp: String?): Long {
        var time: Long = 0
        if (mPublishedTimeStamp != null) {

            try {
                var simpleDateFormat = SimpleDateFormat(SMTConfigConstants.SERVER_TIME_FORMAT, Locale.getDefault())
                simpleDateFormat.timeZone = TimeZone.getTimeZone("UTC")
                time = simpleDateFormat.parse(mPublishedTimeStamp).time
            } catch (e: Exception) {
                SMTLogger.e(TAG, e.message.toString())
            }
        }
        return time
    }

    fun getFormattedTimeDifference(mPublishedTimeStamp: String?): String? {

        val publishedTimeStamp = convertStringDatetoTimeStamp(mPublishedTimeStamp)

        return getDifferenceInWords(publishedTimeStamp, System.currentTimeMillis())
    }

    /**
     * Method to provide time stamp  difference in words
     */
    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    fun getDifferenceInWords(publishedTimeStamp: Long, currentTime: Long): String {
        val diff = currentTime - publishedTimeStamp
        var different = diff
        val secondsInMilli: Long = 1000
        val minutesInMilli = secondsInMilli * 60
        val hoursInMilli = minutesInMilli * 60
        val daysInMilli = hoursInMilli * 24
        val monthInMilli = daysInMilli * 30
        val yearInMilli = monthInMilli * 12

        val elapsedYears = different / yearInMilli
        if (elapsedYears > 1) {
            return "$elapsedYears yrs ago"
        } else if (elapsedYears == 1L) {
            return "$elapsedYears yr ago"
        }

        val elapsedMonths = different / monthInMilli
        if (elapsedMonths > 1) {
            return "$elapsedMonths months ago"
        } else if (elapsedMonths == 1L) {
            return "$elapsedMonths month ago"
        }

        val elapsedDays = different / daysInMilli
        if (elapsedDays > 1) {
            return "$elapsedDays days ago"
        } else if (elapsedDays == 1L) {
            return "$elapsedDays day ago"
        }
        different %= daysInMilli

        val elapsedHours = different / hoursInMilli
        if (elapsedHours > 1) {
            return "$elapsedHours hrs ago"
        } else if (elapsedHours == 1L) {
            return "$elapsedHours hr ago"
        }

        different %= hoursInMilli

        val elapsedMinutes = different / minutesInMilli
        if (elapsedMinutes > 1) {
            return "$elapsedMinutes mins ago"
        } else if (elapsedMinutes == 1L) {
            return "$elapsedMinutes min ago"
        }

        val elapsedSeconds = different / secondsInMilli
        return if (elapsedSeconds > 1) {
            "$elapsedSeconds secs ago"
        } else {
            "$elapsedSeconds sec ago"
        }

        /*return getFormatedDate(different)*/

    }

    /**
     *  This function will compare identity from notification and app after that it will
     *  save the identity in sharedprefrences.
     */
    private fun compareIdentity(context: Context, attributionMap: HashMap<String, String>): Boolean {

        var dropAttribute: Boolean = false

        val preference = SMTPreferenceHelper.getAppPreferenceInstance(context, null)

        val notifIdentity = attributionMap[SMTNotificationConstants.NOTIF_ATTRIBUTION_IDENTITY]?.toLowerCase()
                ?: ""

        val appIdentity = if (preference.getString(SMTPreferenceConstants.SMT_USER_IDENTITY).isNotEmpty()) {
            preference.getString(SMTPreferenceConstants.SMT_USER_IDENTITY).toLowerCase()
        } else {
            ""
        }

        SMTLogger.internal(TAG, "Identity: App identity: $appIdentity, Notification identity: $notifIdentity")
        val identity = when {
            appIdentity == notifIdentity -> {
                appIdentity
            }
            appIdentity.isNotEmpty() && notifIdentity.isEmpty() -> {
                appIdentity
            }
            appIdentity.isEmpty() && notifIdentity.isNotEmpty() -> {

                SMTPreferenceHelper.getAppPreferenceInstance(context, null).setString(SMTPreferenceConstants.SMT_USER_OLD_IDENTITY, appIdentity)
                notifIdentity
            }
            appIdentity != notifIdentity -> {
                dropAttribute = true
                appIdentity
            }
            else -> {
                ""
            }
        }
        // Remove Identity from Hashmap. It was used only for comparing identity
        attributionMap.remove(SMTNotificationConstants.NOTIF_ATTRIBUTION_IDENTITY)
        preference.setString(SMTPreferenceConstants.SMT_USER_IDENTITY, identity)
        preference.setString(SMTPreferenceConstants.SMT_NOTIFICATION_IDENTITY, notifIdentity)
        return dropAttribute
    }


    /*private fun getFormatedDate(different: Long): String {
        val format = SimpleDateFormat("dd/MM/yyyy", Locale.getDefault())
        return format.format(different)
    }*/

    /**
     *  This function will return status of Panel
     *  if Panel is active,send notification otherwise don't send notification.
     *  @return status of panel in boolean
     */
    internal fun checkPanelStatus(context: Context): Boolean {
        var status = true

        context.let {
            val sharedPref = SMTPreferenceHelper.getAppPreferenceInstance(it, null)
            status = sharedPref.getBoolean(SMTPreferenceConstants.IS_PANEL_ACTIVE)

            // If Base url is empty, no need to init SDK
            if (sharedPref.getString(SMTPreferenceConstants.SMT_BASE_URL).isEmpty()) {
                SMTLogger.v(TAG, "Base URL is empty")
                return false
            }

            status = when (sharedPref.getBoolean(SMTPreferenceConstants.IS_PANEL_ACTIVE)) {
                true -> true
                false -> {
                    val currentDate = System.currentTimeMillis()
                    val oldTimestamp = sharedPref.getLong(SMTPreferenceConstants.SMT_SDK_INIT_TIMESTAMP)
                    val diffInMillies = Math.abs(currentDate - oldTimestamp)
                    val diff = TimeUnit.DAYS.convert(diffInMillies, TimeUnit.MILLISECONDS)
                    SMTLogger.v(TAG, "Days difference ------> $diff")
                    diff >= 7
                }
            }
            SMTLogger.v(TAG, "SMT Panle status ------> $status")
        }
        return status
    }

    /**
     * This method takes the sound name without extension
     * and fetch the sound uri from app level raw folder.
     * @param context for fetch the package name
     * @param soundFile get the sound name without extension
     */
    fun getSoundUri(context: Context, soundFile: String?): Uri? {
        if (!soundFile.isNullOrEmpty()) {
            val soundId: Int = context.resources.getIdentifier(soundFile, "raw", context.packageName)
            if (soundId != 0) {
                return Uri.parse(ContentResolver.SCHEME_ANDROID_RESOURCE + "://" + context.packageName + "/" + soundId)
            }
        }
        return null
    }

    /**
     *  This function provides difference between two timestamp
     */

    fun checkDateDifferenceProgressEvent(currentTimeStamp: Long, oldTimeStamp: Long): Boolean {
        val diff: Long = currentTimeStamp - oldTimeStamp
        val seconds = diff / 1000
        val minutes = seconds / 60
        val hours = minutes / 60
        val days = hours / 24

        return when {
            days < 0 -> false
            days >= 2 -> true
            else -> false
        }
    }

    /**
     *  This function will be used to check FCM is available or not
     */
    fun isFCMAvailable(): Boolean {
        return try {
            Class.forName(SMTConstants.FCM_CLASS_NAME)
            true
        } catch (e: ClassNotFoundException) {
            false
        }
    }

    /**
     *  This function will convert Json object to hashmap
     *  @param jsonObject Data Payload
     *  @return hashmap
     */
    fun jsonToHashMap(jsonObject: JSONObject): HashMap<String, Any> {
        val hashMap = HashMap<String, Any>()
        try {
            val iterator = jsonObject.keys()
            while (iterator.hasNext()) {
                val key = iterator.next()
                val value = jsonObject[key]
                hashMap[key] = value
            }
        } catch (e: Exception) {
            SMTLogger.e(TAG, e.message.toString())
        }
        return hashMap
    }

    /**
     *  This method will parse JSONObject and will convert all keys to either lowercase / uppercase based
     *  on user choice
     *  @jsonObject Object whose all keys to be converted to uppercase / lowercase
     *  @isLowerCase boolean value
     *  @return JSONObject
     */

    fun jsonKeyCaseConverter(jsonObject: JSONObject, isLowerCase: Boolean): JSONObject {

        val resultJsonObject = JSONObject()
        val keys = jsonObject.keys()
        while (keys.hasNext()) {
            val key = keys.next()
            var value: Any? = null
            try {
                val objChild = jsonObject.get(key)
                value = when (objChild) {
                    is JSONArray -> parseJsonArray(objChild, isLowerCase)
                    is JSONObject -> this.jsonKeyCaseConverter(objChild, isLowerCase)
                    else -> jsonObject.get(key)
                }
            } catch (jsonException: JSONException) {
            }

            val updatedKey = when (isLowerCase) {
                true -> key.toLowerCase(Locale.getDefault())
                false -> key.toUpperCase(Locale.getDefault())
            }
            resultJsonObject.put(updatedKey, value)
        }
        return resultJsonObject
    }

    /**
     *  This will parse JSONArray and will convert keys to lowercase / uppercase
     *  @obj - JSONArray object
     *  @isLowerCase boolean value for case
     *  @return converted JSONArray
     */
    private fun parseJsonArray(obj: JSONArray, isLowercase: Boolean): JSONArray {

        val objResultArray = JSONArray()
        var value: Any?
        for (i in 0..(obj.length() - 1)) {

            val objChild = obj.get(i)

            value = when (objChild) {
                is JSONArray -> parseJsonArray(objChild, isLowercase)
                is JSONObject -> this.jsonKeyCaseConverter(objChild, isLowercase)
                else -> obj.get(i)
            }
            objResultArray.put(value)
        }
        return objResultArray
    }

    fun updateSmartechSettingsConfig(context: Context, settings: SMTSdkInitializeResponse.SmartTechSettings) {

        val mPreferences = SMTPreferenceHelper.getAppPreferenceInstance(context, null)
        mPreferences.setBoolean(SMTPreferenceConstants.IS_SMRATECH_SETTINGS_STORED, true)
        mPreferences.setInt(SMTPreferenceConstants.BATCH_INTERVAL, settings.batchInterval)
        mPreferences.setInt(SMTPreferenceConstants.BATCH_SIZE, settings.batchSize)
        mPreferences.setInt(SMTPreferenceConstants.TOKEN_INTERVAL, settings.tokenInterval)
        mPreferences.setBoolean(SMTPreferenceConstants.IS_PUSH_AMP_ENABLED, settings.paEnabled)
        mPreferences.setInt(SMTPreferenceConstants.PUSH_AMP_INTERVAL, settings.paInterval)
        mPreferences.setBoolean(SMTPreferenceConstants.IS_FETCH_LOCATION_ENABLED, settings.fetchLocation)
        mPreferences.setBoolean(SMTPreferenceConstants.IS_PANEL_ACTIVE, settings.panelActive)
        mPreferences.setBoolean(SMTPreferenceConstants.IS_SDK_ACTIVE, settings.sdkActive)
        mPreferences.setInt(SMTPreferenceConstants.SESSION_INTERVAL, settings.sessionInterval)
        mPreferences.setInt(SMTPreferenceConstants.EVENT_LIMIT, settings.eventLimit)
        mPreferences.setString(SMTPreferenceConstants.SMT_BASE_URL, settings.baseUrl ?: "")
        mPreferences.setString(SMTPreferenceConstants.SMT_BASE_URL_TRACKAPPACT, settings.smartechURL?.trackAppActUrl
                ?: "")
        mPreferences.setString(SMTPreferenceConstants.SMT_BASE_URL_INAPP, settings.smartechURL?.inAppUrl
                ?: "")
        mPreferences.setString(SMTPreferenceConstants.SMT_BASE_URL_INAPP_LIST_SEG, settings.smartechURL?.inAppListSegUrl
                ?: "")
        mPreferences.setString(SMTPreferenceConstants.SMT_BASE_URL_INBOX, settings.smartechURL?.inboxUrl
                ?: "")
        mPreferences.setString(SMTPreferenceConstants.SMT_BASE_URL_PUSHAMP, settings.smartechURL?.pushAmpUrl
                ?: "")
        mPreferences.setBoolean(SMTPreferenceConstants.IS_LOG_ENABLED, settings.debuglevel?.logEnabled
                ?: false)
        mPreferences.setInt(SMTPreferenceConstants.LOG_LEVEL, settings.debuglevel?.logLevel
                ?: 0)
        mPreferences.setString(SMTPreferenceConstants.GUIDS, settings.debuglevel?.guids?.toString()
                ?: "")
        SMTLogger.internal(TAG, "Smartech settings: $settings")
    }


    fun handleAppUpdate(context: Context) {
        val mPreferences = SMTPreferenceHelper.getAppPreferenceInstance(context, null)
        val trackAppActUrl = mPreferences.getString(SMTPreferenceConstants.SMT_BASE_URL_TRACKAPPACT, "")

        if (trackAppActUrl.isEmpty()) {
            SMTLogger.internal(TAG, "trackApt url is empty ")
            val settings = SMTSdkInitializeResponse.SmartTechSettings()
            settings.panelActive = true
            settings.sdkActive = true
            settings.baseUrl = "https://fpn.netcoresmartech.com/"

            val smartechUrl = SMTSdkInitializeResponse.SmartTechSettings.SmartTechBaseURL()
            smartechUrl.trackAppActUrl = "https://fpn.netcoresmartech.com/"

            val debugLevelC = SMTSdkInitializeResponse.SmartTechSettings.SmartTechDebugLevel()
            debugLevelC.logEnabled = true
            debugLevelC.logLevel = 9

            settings.smartechURL = smartechUrl
            settings.debuglevel = debugLevelC

            updateSmartechSettingsConfig(context, settings)

            SMTLogger.internal(TAG, "Before app update settings: $settings")
        } else {
            SMTLogger.internal(TAG, "trackApt url is not empty ")
        }

    }

    fun jsonArrayToStringList(array: JSONArray?): List<String> {
        val list: MutableList<String> = ArrayList()
        if (array != null && array.length() > 0) {
            for (i in 0 until array.length()) {
                list.add(array.optString(i))
            }
        }
        return list
    }

    fun compareLists(list1: List<String>?, list2: List<String>?): Boolean {
        try {
            if (list1 != null && list2 != null) {
                for (i in list1.indices) {
                    for (j in list2.indices) {
                        if (list1[i] == list2[j]) return true
                    }
                }
            }
        } catch (e: java.lang.Exception) {
            SMTLogger.e(TAG, e.message.toString())
        }
        return false
    }
}

/**
 *  Extension function to trim string if its not empty otherwise return empty string
 */
fun String.trim(): String {

    return when (this.isNotEmpty()) {
        true -> (this as CharSequence).trim().toString()
        false -> ""
    }
}