package com.netcore.android.db

import android.content.ContentValues
import android.database.Cursor
import com.netcore.android.event.SMTNotificationSourceType
import com.netcore.android.logger.SMTLogger
import com.netcore.android.notification.SMTNotificationParser
import com.netcore.android.notification.SMTScheduleNotification
import com.netcore.android.notification.SMTSchedulePNData
import com.netcore.android.notification.models.SMTNotificationData
import java.text.SimpleDateFormat

@Suppress("PrivatePropertyName")
/**
 * Copyright © 2019 Netcore. All rights reserved.
 *
 * Notification DB Table, All notification related operations has handled here
 *
 * @see SMTDataBaseWrapper for method definitions
 *
 * @author Netcore
 * @version 1.0
 * @since 26-02-2019
 */
internal class SMTNotificationTable(private val wrapper: SMTDataBaseWrapper) : SMTDBTable(wrapper) {
    private val TAG = SMTNotificationTable::class.java.simpleName

    // Notification Table Columns names
    private val KEY_TR_ID = "tr_id"   // Column I (Primary Key)
    private val KEY_PAYLOAD = "payload" //Column II json data
    private val KEY_SOURCE = "source" //Column III (FCM / PUSHAMP)
    private val KEY_IS_DISPLAYED_ON_TRAY = "isDisplayedOnTray" //Column VI (TRUE / FALSE)
    private val KEY_DELIVER_TIME = "time"  // Column VII
    private val KEY_NOTIFICATION_ID = "notif_id"  // Column VIII - used to sync with Inbox
    private val KEY_SCHEDULED_DATE = "scheduled_date"  // Column IX - modified_date
    private val KEY_SCHEDULED_STATUS = "schedule_status"  // Column X - schedule_status
    private val KEY_TTL = "ttl"  // Column X - schedule_status

    companion object {
        const val KEY_IS_DISMISSED = "isDismissed" //Column IV (TRUE / FALSE)
        const val KEY_IS_CLICKED = "isClicked"  // Column V (TRUE /FALSE)
    }

    private val mTableName = "pushNotification"


    /**
     * Creates the table
     */
    override fun createTable() {
        executeQuery(getNotificationTableCreateStatement())
    }

    /**
     * prepares the notification table creation query statement
     */
    private fun getNotificationTableCreateStatement(): String {
        return ("CREATE TABLE IF NOT EXISTS " + mTableName
                + " ( "
                + KEY_TR_ID + " TEXT PRIMARY KEY, " // The NOtification Transaction ID
                + KEY_PAYLOAD + " TEXT NOT NULL, " // Notification Payload
                + KEY_SOURCE + " TEXT NOT NULL, " // Notification SOURCE (FCM / PUSH AMP)
                + KEY_IS_DISMISSED + " INTEGER NOT NULL DEFAULT 0, " // either 0 / 1, (flase / true) - whether the Notification dismissed or not
                + KEY_IS_CLICKED + " INTEGER NOT NULL DEFAULT 0, " // either 0 / 1, (flase / true) - whether the Notification read or not
                + KEY_IS_DISPLAYED_ON_TRAY + " INTEGER NOT NULL DEFAULT 0, " // either 0 / 1, (flase / true) - whether the Notification displayed on notification tray or not
                + KEY_DELIVER_TIME + " LONG NOT NULL, " // Notification delivery time
                + KEY_NOTIFICATION_ID + " INTEGER NOT NULL DEFAULT 0, " // Notification ID
                + KEY_SCHEDULED_DATE + " TEXT, " // Scheduled Date
                + KEY_SCHEDULED_STATUS + " TEXT ," // Scheduled Notification Status
                + KEY_TTL + " TEXT " // Scheduled Notification Status
                + " ) ")
    }

    override fun upgradeTable(oldVersion: Int, newVersion: Int) {
        // if needs to do any action for updating the table
        if (newVersion > oldVersion) {
            try {
                //three columns are added
                wrapper.execSQL("ALTER TABLE $mTableName ADD COLUMN $KEY_SCHEDULED_DATE TEXT;")
                wrapper.execSQL("ALTER TABLE $mTableName ADD COLUMN $KEY_SCHEDULED_STATUS TEXT;")
                wrapper.execSQL("ALTER TABLE $mTableName ADD COLUMN $KEY_TTL TEXT;")
            } catch (e: Exception) {
                SMTLogger.e(TAG, e.message.toString())
            }
        }

    }

    /**
     * Insert a single notification to the notification table
     * @param trid notification transaction id
     * @param payload notification payload
     * @param source notification source like PN or PushAmp
     */
    internal fun insertNotification(trid: String, payload: String, @SMTNotificationSourceType.Source source: Int) {
        val values = ContentValues()
        values.put(KEY_TR_ID, trid)
        values.put(KEY_PAYLOAD, payload)
        values.put(KEY_SOURCE, source)
        values.put(KEY_DELIVER_TIME, System.currentTimeMillis())
        values.put(KEY_IS_DISPLAYED_ON_TRAY, 1)
        values.put(KEY_SCHEDULED_DATE, "")
        values.put(KEY_SCHEDULED_STATUS, "")
        values.put(KEY_TTL, "")
        val rowId = wrapper.insert(mTableName, null, values)
        SMTLogger.internal("SMTNotificationTable", "Value: $rowId")
    }

    /**
     * Insert a single notification to the notification table
     * @param trid notification transaction id
     * @param payload notification payload
     * @param source notification source
     * @param  modifiedDate notification modifie date
     * @param scheduledPNStatus notification schedule status
     */
    internal fun insertScheduledPN(trid: String, payload: String, @SMTNotificationSourceType.Source source: Int, scheduledDate: String, scheduledPNStatus: String, ttl: String) {
        val values = ContentValues()
        values.put(KEY_TR_ID, trid)
        values.put(KEY_PAYLOAD, payload)
        values.put(KEY_SOURCE, source)
        values.put(KEY_DELIVER_TIME, System.currentTimeMillis())
        values.put(KEY_IS_DISPLAYED_ON_TRAY, 1)
        values.put(KEY_SCHEDULED_DATE, scheduledDate)
        values.put(KEY_SCHEDULED_STATUS, scheduledPNStatus)
        values.put(KEY_TTL, ttl)
        wrapper.insert(mTableName, null, values)
    }

    internal fun updateScheduledPN(trid: String, payload: String, @SMTNotificationSourceType.Source source: Int, modifiedDate: String, scheduledPNStatus: String) {
        val values = ContentValues()
        values.put(KEY_TR_ID, trid)
        values.put(KEY_PAYLOAD, payload)
        values.put(KEY_SOURCE, source)
        values.put(KEY_DELIVER_TIME, System.currentTimeMillis())
        values.put(KEY_IS_DISPLAYED_ON_TRAY, 1)
        values.put(KEY_SCHEDULED_DATE, modifiedDate)
        values.put(KEY_SCHEDULED_STATUS, scheduledPNStatus)
        wrapper.update(mTableName, values, "$KEY_TR_ID = ?",
                arrayOf(trid))
    }

    /**
     * To find a notification with trId
     * @param trId notification trid
     * @return Retruns true if already notification present in Table else false
     */
    fun findNotificationById(trId: String): Boolean {
        val selectClause = "SELECT * FROM $mTableName where $KEY_TR_ID = ?"

        val cursor = executeRawQuery(selectClause, arrayOf(trId))
        cursor?.let {
            if (it.count > 0) {
                return true
            }
        }
        cursor?.close()
        return false
    }

    /**
     * To find a notification read status with trId
     * @param trId notification trid
     * @return Returns true if already notification is read
     */
    fun findNotificationReadStatusById(trId: String): Boolean {
        val selectClause = "SELECT * FROM $mTableName where $KEY_TR_ID = ?"

        val cursor = executeRawQuery(selectClause, arrayOf(trId))
        cursor?.let {
            if (it.count > 0) {
                if (it.moveToFirst()) {
                    return cursor.getString(cursor.getColumnIndex(KEY_IS_CLICKED)).toInt() == 1
                }
            }
        }
        cursor?.close()
        return false
    }

    fun getLastScheduledModifiedPNDate(trid: String): String {
        val selectClause = "SELECT * FROM $mTableName where $KEY_TR_ID = ?"
        val cursor = executeRawQuery(selectClause, arrayOf(trid))
        cursor?.let {
            if (it.moveToFirst()) {
                val data = it.getString(it.getColumnIndex(KEY_SCHEDULED_DATE))
                return data.toString()
            }
        }
        cursor?.close()
        return ""
    }

    fun ifDateisModified(trid: String, newDate: String): Boolean {
        val sdf = SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss")
        val oldDate = getLastScheduledModifiedPNDate(trid)
        if (!oldDate.equals("")) {
            return sdf.parse(oldDate).before(sdf.parse(newDate))
        }

        return false
    }

    fun getRowIdByTrid(trId: String): Int {
        val selectClause = "SELECT rowid FROM $mTableName where $KEY_TR_ID = ?"

        val cursor = executeRawQuery(selectClause, arrayOf(trId))
        cursor?.let {
            if (it.moveToFirst()) {
                val rowId = it.getInt(0)
                return rowId
            }
        }
        cursor?.close()
        return 0

    }

    fun getScheduledPNPayload(trid: String): String {
        val selectQuery = "SELECT * FROM $mTableName where $KEY_TR_ID = ?"
        val cursor = executeRawQuery(selectQuery, arrayOf(trid))

        cursor?.let {
            if (cursor.moveToFirst()) {
                val payload = it.getString(it.getColumnIndex(KEY_PAYLOAD))
                return payload
            }
        }
        cursor?.close()
        return ""
    }

    /**
     * To find a notification with trId
     * @param trId notification trid
     * @return Retruns true if already notification present in Table else false
     */
    fun getNotificationById(trId: String): SMTNotificationData? {
        val selectClause = "SELECT * FROM $mTableName where $KEY_TR_ID = ?"

        val cursor = executeRawQuery(selectClause, arrayOf(trId))
        cursor?.let {
            if (it.count > 0) {
                if (it.moveToFirst()) {
                    val source = if (cursor.getString(cursor.getColumnIndex(KEY_SOURCE)).toInt() == SMTNotificationSourceType.NOTIFICATION_SOURCE_PN) {
                        SMTNotificationSourceType.NOTIFICATION_SOURCE_PN
                    } else {
                        SMTNotificationSourceType.NOTIFICATION_SOURCE_PAMP
                    }
                    val payload = cursor.getString(cursor.getColumnIndex(KEY_PAYLOAD))
                    val notif = SMTNotificationParser().parse(payload, source)
                    notif?.mPayload = payload
                    notif?.notificationId = cursor.getInt(cursor.getColumnIndex(KEY_NOTIFICATION_ID))
                    return notif
                }
            }
        }
        cursor?.close()
        return null
    }

    /**
     *  This function will return the read status of Notification
     *  @return Boolean Notification read status
     */
    internal fun getNotificationReadStatus(trId: String): Boolean {
        val selectClause = "SELECT * FROM $mTableName where $KEY_TR_ID = ?"

        val cursor = executeRawQuery(selectClause, arrayOf(trId))
        cursor?.let {
            if (it.count > 0) {
                if (it.moveToFirst()) {
                    return cursor.getString(cursor.getColumnIndex(KEY_IS_CLICKED)).toInt() == 1
                }
            }
        }
        cursor?.close()
        return false
    }

    /**
     * Update notification status in notification table
     * @param trid - Notification ID, which to be updated
     * @param columnName - columname which to be updated
     * @param columnValue - value to be updated
     */
    internal fun updateNotification(trid: String, columnName: String, columnValue: Boolean) {

        val values = ContentValues()

        val intValue = if (columnValue) 1 else 0

        values.put(columnName, intValue)
        val selection = "$KEY_TR_ID = ?"

        wrapper.update(mTableName, values, selection, arrayOf(trid))
    }

    /**
     * Update notification status in notification table
     * @param trid - Notification ID, which to be updated
     * @param columnName - columname which to be updated
     * @param columnValue - value to be updated
     */
    internal fun updateNotificationId(trid: String, columnValue: Int) {

        val values = ContentValues()

        values.put(KEY_NOTIFICATION_ID, columnValue)

        val selection = "$KEY_TR_ID = ?"

        wrapper.update(mTableName, values, selection, arrayOf(trid))
    }

    /*internal fun updateMultipleNotifications(trid: Array<String>, values: ContentValues) {
        val selection = "$KEY_TR_ID = ?"
        wrapper.update(mTableName, values, selection, trid)
    }*/

    /**
     * Delete the notifications after certain time interval
     * currently its 30 days of notification expire time
     * @param timeStamp - The timestamp to be checked after which notifications to be deleted
     */
    internal fun deleteNotifications(timeStamp: Long) {

        val whereClause = "$KEY_DELIVER_TIME <= ?"
        wrapper.delete(mTableName, whereClause, arrayOf(timeStamp.toString()))
    }

    /**
     * to execute raw query
     * its used by unit test class SMTDatabaseTest
     */
    fun executeRawTestQuery(query: String): Cursor? {
        var finalQuery = query + mTableName
        return executeRawQuery(finalQuery)
    }

    /*
    * get all scheduled notification on reboot
    * */
    fun getScheduledNotification(): ArrayList<SMTSchedulePNData>? {
//        val selectClause = "SELECT * FROM $mTableName where $KEY_SCHEDULED_STATUS = 's'"
        val selectClause = "SELECT * FROM $mTableName where $KEY_SCHEDULED_STATUS  like 's' "
        val mModelList = ArrayList<SMTSchedulePNData>()
        val cursor = executeRawQuery(selectClause)
        val mScheduleNotification = SMTScheduleNotification()
        cursor?.let {
            if (it.count > 0) {
                while (it.moveToNext()) {
                    val payload = it.getString(it.getColumnIndex(KEY_PAYLOAD))
                    val source = it.getInt(it.getColumnIndex(KEY_SOURCE))
                    val notif = mScheduleNotification.getNotificationModel(payload, source)
                    notif?.notificationId = it.getInt(it.getColumnIndex(KEY_NOTIFICATION_ID))
                    if (notif != null) {
                        mModelList.add(SMTSchedulePNData(payload, notif))
                    }
                }
                it.close()
                return mModelList
            }
        }
        return null
    }

    fun updateScheduledPNStatus(trid: String, scheduledPNStatus: String) {
        val values = ContentValues()
        values.put(KEY_TR_ID, trid)
        values.put(KEY_SCHEDULED_STATUS, scheduledPNStatus)
        wrapper.update(mTableName, values, "$KEY_TR_ID = ?",
                arrayOf(trid))
    }

    /*
* get all scheduled notification on reboot
* */
    fun getScheduledNotificationForDemoApp(): ArrayList<SMTSchedulePNData>? {
//        val selectClause = "SELECT * FROM $mTableName where $KEY_SCHEDULED_STATUS = 's'"
        val selectClause = "SELECT * FROM $mTableName WHERE $KEY_SCHEDULED_STATUS IS NOT NULL"
        val mModelList = ArrayList<SMTSchedulePNData>()
        val cursor = executeRawQuery(selectClause)
        val mScheduleNotification = SMTScheduleNotification()
        cursor?.let {
            if (it.count > 0) {
                while (it.moveToNext()) {

                    val payload = it.getString(it.getColumnIndex(KEY_PAYLOAD))
                    val source = it.getInt(it.getColumnIndex(KEY_SOURCE))
                    val notif = mScheduleNotification.getNotificationModel(payload, source)
                    notif?.notificationId = it.getInt(it.getColumnIndex(KEY_NOTIFICATION_ID))
                    if (notif != null) {
                        mModelList.add(SMTSchedulePNData(payload, notif))
                    }
                }
                it.close()
                return mModelList
            }
        }
        return null
    }

}