package com.netcore.android.db

import android.content.Context
import android.database.sqlite.SQLiteDatabase
import android.database.sqlite.SQLiteOpenHelper
import java.lang.ref.WeakReference

/**
 * Copyright © 2019 Netcore. All rights reserved.
 *
 * SingleTon class Extends SQLiteOpenHelper class
 *
 *@param context - Weakrefernce of the context
 *
 * @author Netcore
 * @version 1.0
 * @since 26-02-2019
 */
internal class SMTDatabase private constructor(private val context: WeakReference<Context>) : SQLiteOpenHelper(context.get(), SMTDBConstants.DATABASE_NAME, null, SMTDBConstants.DATABASE_VERSION) {

    companion object {

        /**
         * Notification table reference
         */
        private lateinit var mNotificationTable: SMTNotificationTable

        /**
         * Event table reference
         */
        private lateinit var mEventTable: SMTEventTable

        /**
         * InApp rule table reference
         */
        private lateinit var mInAppRulesTable: SMTInAppRulesTable

        /**
         * InApp rule table reference
         */
        private lateinit var mAppInboxTable: SMTAppInboxTable

        @Volatile
        private var INSTANCE: SMTDatabase? = null

        @Volatile
        private var mDb: SQLiteDatabase? = null

        /**
         * Get method for creating singleton instance of the class
         * @param context - App context
         * @return SMTDatabase - Returns Database instance
         */
        fun getInstance(context: Context): SMTDatabase {
            return INSTANCE ?: synchronized(SMTDatabase::class.java) {
                INSTANCE ?: buildInstance(context).also { INSTANCE = it }
            }
        }

        /**
         * Buils the instance of SMTDatabase
         *
         * @param context - App context
         * @return SMTDatabase - Returns Database instance
         */
        private fun buildInstance(context: Context): SMTDatabase {
            val instance = SMTDatabase(WeakReference(context))
            initializeIfRequiredAndGetWriteDatabase(instance)
            initTables(context)
            SMTDataBaseWrapper.getInstance(context).setDatabaseInstance(mDb!!)
            return instance
        }

        /**
         * Intializing the tables, Notification, Event and InApp Rule table
         * @param context App Context
         */
        private fun initTables(context: Context) {
            var wrapper = SMTDataBaseWrapper.getInstance(context)
            mDb?.let {
                wrapper.setDatabaseInstance(it)
            }
            mNotificationTable = SMTNotificationTable(wrapper)
            mEventTable = SMTEventTable(wrapper)
            mInAppRulesTable = SMTInAppRulesTable(wrapper)
            mAppInboxTable = SMTAppInboxTable(wrapper)
        }

        /**
         * Initializes the writable DB instance
         * @param instance - SMTDatabase instance reference
         */
        private fun initializeIfRequiredAndGetWriteDatabase(instance: SMTDatabase) {
            if (mDb == null) {
                mDb = instance.writableDatabase
            }
            mDb?.let {
                if (!it.isOpen) {
                    mDb = instance.writableDatabase
                }
            }
        }
    }

    /**
     * Overridden method of SQLiteOpenHelper
     * creates database wrapper class, and the required tables
     */
    override fun onCreate(db: SQLiteDatabase?) {
        var wrapper: SMTDataBaseWrapper
        db?.let { dataBase ->
            context.get()?.let {
                wrapper = SMTDataBaseWrapper.getInstance(it)
                wrapper.setDatabaseInstance(dataBase)
                mNotificationTable = SMTNotificationTable(wrapper)
                mEventTable = SMTEventTable(wrapper)
                mInAppRulesTable = SMTInAppRulesTable(wrapper)
                mAppInboxTable = SMTAppInboxTable(wrapper)
                mNotificationTable.createTable()
                mEventTable.createTable()
                mInAppRulesTable.createTable()
                mAppInboxTable.createTable()
            }
        }
    }

    /**
     * Overridden method of SQLiteOpenHelper class
     * creates database  wrapper class, and the required tables
     */
    override fun onUpgrade(db: SQLiteDatabase?, oldVersion: Int, newVersion: Int) {
        var wrapper: SMTDataBaseWrapper
        db?.let { dataBase ->
            context.get()?.let {
                wrapper = SMTDataBaseWrapper.getInstance(it)
                wrapper.setDatabaseInstance(dataBase)
                mNotificationTable.upgradeTable(oldVersion, newVersion)
                mEventTable.upgradeTable(oldVersion, newVersion)
                mInAppRulesTable.upgradeTable(oldVersion, newVersion)
                mAppInboxTable.upgradeTable(oldVersion, newVersion)
            }
        }
    }

    /**
     * Exposed method to get Notification table reference
     * @return SMTNotificationTable reference
     */
    fun getNotificationTable(): SMTNotificationTable {
        return mNotificationTable
    }

    /**
     * Exposed method to get InApprule table reference
     * @return SMTInAppRulesTable reference
     */
    fun getInAppRuleTable(): SMTInAppRulesTable {
        return mInAppRulesTable
    }

    /**
     * Exposed method to get Event table reference
     * @return SMTEventTable reference
     */
    fun getEventTable(): SMTEventTable {
        return mEventTable
    }

    /**
     * Exposed method to get Event table reference
     * @return SMTEventTable reference
     */
    fun getInboxTable(): SMTAppInboxTable {
        return mAppInboxTable
    }

    override fun close() {
        INSTANCE = null
        mDb?.close()
        mDb = null
    }
}