package com.netcore.android.db

import android.content.ContentValues
import com.netcore.android.event.SMTNotificationSourceType
import com.netcore.android.inbox.utility.SMTInboxMessageStatus
import com.netcore.android.inbox.utility.SMTInboxMessageType
import com.netcore.android.logger.SMTLogger
import com.netcore.android.notification.SMTNotificationParser
import com.netcore.android.notification.models.SMTNotificationData

internal class SMTAppInboxTable(private val wrapper: SMTDataBaseWrapper) : SMTDBTable(wrapper) {

    private val TAG = SMTAppInboxTable::class.java.simpleName

    companion object {
        internal val STATUS_SENT = "sent"
        internal val STATUS_DELIVERED = "delivered"
        internal val STATUS_VIEWED = "viewed"
        internal val STATUS_CLICKED = "clicked"
        internal val STATUS_DISMISSED = "dismissed"
    }

    // App Inbox Table Columns names
    private val KEY_TR_ID = "tr_id"   // Column I (Primary Key)
    private val KEY_STATUS = "status" //Column II status of the message(sent/delivered/viewed/clicked/dismissed)
    private val KEY_PUBLISHED_DATE = "publishedDate" //Time stamp in long)
    private val KEY_PAYLOAD = "payload" //Column IV (Notification payload)
//    private val KEY_FILE_PATH = "filePath" //column store the already downloaded file path

    private val mTableName = "appInbox"

    override fun createTable() {
        executeQuery(getAppInboxTableCreateStatement())
    }

    private fun getAppInboxTableCreateStatement(): String {
        return ("CREATE TABLE IF NOT EXISTS " + mTableName
                + " ( "
                + KEY_TR_ID + " TEXT PRIMARY KEY, " // The inbox notification Transaction ID
                + KEY_STATUS + " TEXT NOT NULL, " // Inbox notification staus
                + KEY_PUBLISHED_DATE + " LONG NOT NULL, " // Inbox notification created time
                + KEY_PAYLOAD + " TEXT NOT NULL " // Inbox notification Payload
//                + KEY_FILE_PATH + " TEXT " // Inbox notification Payload
                + " ) ")
    }

    override fun upgradeTable(oldVersion: Int, newVersion: Int) {
        // if needs to do any action for updating the table
    }

    /**
     * Insert a single notification to the notification table
     * @param trid notification transaction id
     * @param payload notification payload
     * @param status notification status
     * @param timeStamp notification creation time
     */
    internal fun insertAppInboxMessage(trid: String, payload: String, status: String, timeStamp: Long) {
        val values = ContentValues()
        values.put(KEY_TR_ID, trid)
        values.put(KEY_PAYLOAD, payload)
        values.put(KEY_PUBLISHED_DATE, timeStamp)
        val updatedStatus = if (status == STATUS_SENT) {
            STATUS_DELIVERED
        } else {
            status
        }
        values.put(KEY_STATUS, updatedStatus)
        wrapper.insert(mTableName, null, values)
    }

    internal fun getInboxMessages(@SMTInboxMessageType.Type type: Int): MutableList<SMTNotificationData> {
        val notifList: MutableList<SMTNotificationData> = mutableListOf()
        val query = getInboxMessageQuery(type)

        val cursor = executeRawQuery(query)
        cursor?.let {
            if (it.count > 0) {
                if (it.moveToFirst()) {
                    do {

                        val payload = it.getString(it.getColumnIndex(KEY_PAYLOAD))
                        val notif = SMTNotificationParser().parse(payload, SMTNotificationSourceType.NOTIFICATION_SOURCE_PAMP)
                        notif?.mPayload = payload
                        notif?.mStatus = it.getString(it.getColumnIndex(KEY_STATUS))
//                        notif?.mMediaLocalPath = cursor.getString(cursor.getColumnIndex(KEY_FILE_PATH))
                        if (notif != null) {
                            notifList.add(notif)
                        }
                    } while (it.moveToNext())
                }
            }
        }
        cursor?.close()
        return notifList
    }

    internal fun getInboxMessageCount(@SMTInboxMessageType.Type messageType: Int): Int {
        val query = getInboxMessageQuery(messageType)

        val cursor = executeRawQuery(query)
        cursor?.let {
            return it.count
        }
        cursor?.close()
        return 0
    }

    /**
     * To find a notification with trId
     * @param trId notification trid
     * @return Retruns true if already notification present in Table else false
     */
    internal fun getInboxMessageById(trId: String): SMTNotificationData? {
        val selectClause = "SELECT * FROM $mTableName where $KEY_TR_ID = ?"
        var notif: SMTNotificationData? = null

        val cursor = executeRawQuery(selectClause, arrayOf(trId))
        cursor?.let {
            val payload = it.getString(it.getColumnIndex(KEY_PAYLOAD))
            notif = SMTNotificationParser().parse(payload, SMTNotificationSourceType.NOTIFICATION_SOURCE_PAMP)
            notif?.mPayload = payload
            notif?.mStatus = it.getString(it.getColumnIndex(KEY_STATUS))

        }
        cursor?.close()
        return notif
    }

    internal fun updateInboxMessageStatus(trId: String, @SMTInboxMessageStatus.Status status: Int): Boolean {
        var count = 0
        val statusString = getInboxStatus(status)
        val values = ContentValues()

        values.put(KEY_STATUS, statusString)
        val selection = "$KEY_TR_ID = ?"

        try {
            count = wrapper.update(mTableName, values, selection, arrayOf(trId))
        } catch (e: Exception) {
            SMTLogger.e(TAG, e.message.toString())
        }
        return count > 0
    }

    /**
     * Update notification status in notification table
     * @param trid - Notification ID, which to be updated
     * @param columnName - columname which to be updated
     * @param columnValue - value to be updated
     */
    internal fun updateInboxMessageById(trid: String, columnName: String, columnValue: String) {

        val values = ContentValues()

        values.put(columnName, columnValue)
        val selection = "$KEY_TR_ID = ?"

        wrapper.update(mTableName, values, selection, arrayOf(trid))
    }

    internal fun clearInboxTable() {
        wrapper.delete(mTableName, null, null)
    }

    private fun getInboxMessageQuery(@SMTInboxMessageType.Type type: Int): String {

        return when (type) {
            SMTInboxMessageType.Type.INBOX_MESSAGE -> "SELECT * FROM $mTableName WHERE $KEY_STATUS IS NOT '$STATUS_DISMISSED' ORDER BY $KEY_PUBLISHED_DATE DESC"
            SMTInboxMessageType.Type.READ_MESSAGE -> "SELECT * FROM $mTableName WHERE $KEY_STATUS IS $STATUS_CLICKED ORDER BY $KEY_PUBLISHED_DATE DESC"
            SMTInboxMessageType.Type.UNREAD_MESSAGE -> "SELECT * FROM $mTableName WHERE $KEY_STATUS IS NOT $STATUS_CLICKED AND $KEY_STATUS IS NOT $STATUS_DISMISSED ORDER BY $KEY_PUBLISHED_DATE DESC"
            SMTInboxMessageType.Type.ALL_MESSAGE -> "SELECT * FROM $mTableName ORDER BY $KEY_PUBLISHED_DATE DESC"
            else -> {
                "SELECT * FROM $mTableName ORDER BY $KEY_PUBLISHED_DATE DESC"
            }
        }
    }

    private fun getInboxStatus(@SMTInboxMessageStatus.Status status: Int): String {

        return when (status) {
            SMTInboxMessageStatus.Status.DELETED -> STATUS_DISMISSED
            SMTInboxMessageStatus.Status.READ -> STATUS_CLICKED
            SMTInboxMessageStatus.Status.VIEWED -> STATUS_VIEWED
            SMTInboxMessageStatus.Status.DELIVERED -> STATUS_DELIVERED
            else -> STATUS_DELIVERED
        }
    }

    fun updateInboxMessagePayload(notification: SMTNotificationData) {

        try {
            SMTLogger.internal(TAG, "${notification.mTrid} updated payload ${notification.mPayload}")
            val values = ContentValues()

            values.put(KEY_PAYLOAD, notification.mPayload)
            val selection = "$KEY_TR_ID = ?"

            wrapper.update(mTableName, values, selection, arrayOf(notification.mTrid))
        } catch (e: Exception) {
            SMTLogger.e(TAG, e.message.toString())
        }
    }
}