package com.netcore.android.notification.carousel

import android.app.NotificationManager
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.graphics.Bitmap
import android.os.Build
import android.os.Bundle
import android.text.TextUtils
import android.view.View
import android.widget.RemoteViews
import androidx.core.app.NotificationCompat
import com.netcore.android.R
import com.netcore.android.event.SMTEventRecorder
import com.netcore.android.logger.SMTLogger
import com.netcore.android.mediadownloader.SMTMediaDownloadManager
import com.netcore.android.notification.*
import com.netcore.android.notification.models.SMTCarouselItemData
import com.netcore.android.notification.models.SMTNotificationData
import com.netcore.android.utility.SMTCommonUtility
import java.lang.ref.WeakReference

internal class SMTCarouselLandscapePNGenerator constructor(val context: WeakReference<Context>) : SMTBaseNotificationGenerator() {

    private val TAG: String = SMTCarouselLandscapePNGenerator::class.java.simpleName

    private var currentStartIndex = 0

    private var carouselSetUp: SMTCarouselSetup? = null

    private var isImagesInCarousel = true

    /**
     * Handles received notification, builds carousel items and also displays the notificaiton
     * @param - notifModel - received notification payload
     */
    internal fun handle(notifModel: SMTNotificationData) {
        carouselSetUp = null
        notifModel.notificationId = SMTCommonUtility.getRandomId()
        notificationManager = context.get()?.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
        updateNotificationId(context.get()!!, notifModel)

        // Build the carousel item
        buildCarousel(notifModel)
    }

    /**
     * Starts downloading all carousel images if carousel has images
     * else simply show carousel with placeholder image
     * @param notifModel - Notification Payload
     */
    private fun buildCarousel(notifModel: SMTNotificationData) {
        var isImageInCarousel = false
        if (notifModel.mCarouselList != null && notifModel.mCarouselList!!.isNotEmpty()) {

            for (item in notifModel.mCarouselList!!) {
                if (!TextUtils.isEmpty(item.imgUrl)) {
                    isImageInCarousel = true
                    break
                }
            }
            // if images present download the images
            if (isImageInCarousel) {
                context.get()?.let {
                    SMTMediaDownloadManager().downloadMedia(it, notifModel, object : SMTMediaDownloadManager.MediaDownloadListener {
                        override fun onDownloadSuccess(notification: SMTNotificationData) {
                            initiateCarouselTransaction(notification)
                        }

                        override fun onDownloadFailed(notification: SMTNotificationData) {
                            initiateCarouselTransaction(notification)
                        }
                    })
                }
            } else {
                // else show carousel with placeholder images
                this.isImagesInCarousel = false
                initiateCarouselTransaction(notifModel)
            }
        }
    }

    /**
     * Initiate carousel and show notification
     */
    private fun initiateCarouselTransaction(notifModel: SMTNotificationData) {
        currentStartIndex = 0
        notifModel.mCarouselList?.let { item ->
            if (item.isNotEmpty()) {
                if (item.size == 1) {
                    prepareVariablesForCarouselAndShow(item[currentStartIndex], null, notifModel, carouselSetUp)
                } else {
                    prepareVariablesForCarouselAndShow(item[currentStartIndex], item[currentStartIndex + 1], notifModel, carouselSetUp)
                }
            } else {
                prepareVariablesForCarouselAndShow(null, null, notifModel, carouselSetUp)
            }
        }
    }

    /**
     * Builds carousel parcel and starts showing carousel
     * @param leftItem - left carousel item
     * @param rightItem - right carousel item
     * @param carouselSet - carousel parcel
     */
    private fun prepareVariablesForCarouselAndShow(leftItem: SMTCarouselItemData?, rightItem: SMTCarouselItemData?, notifModel: SMTNotificationData?, carouselSet: SMTCarouselSetup?) {

        context.get()?.let {
            val notifOption = SMTCommonUtility.getNotificationOptions(it)
            if (notifModel != null && carouselSet == null) {
                carouselSetUp = SMTCarouselSetup(
                        notifModel.mTrid,
                        notifModel.mNotificationType,
                        notifModel.mSource,
                        notifModel.mCarouselList,
                        notifModel.mTitle,
                        notifModel.mMessage,
                        notifModel.mTitle,
                        notifModel.mMessage,
                        notifModel.mSubtitle,
                        notifModel.mDeepLinkPath,
                        notifModel.mPNMeta,
                        notifModel.notificationId,
                        currentStartIndex,
                        notifOption.smallIconId,
                        notifOption.largeIconId,
                        notifOption.placeHolderIcon,
                        leftItem,
                        rightItem,
                        isImagesInCarousel,
                        notifModel.mCustomPayload,
                        notifModel.mSmtAttributePayload,
                        notifModel.mSound,
                        notifModel.mSoundFile,
                        notifModel.mChannelId
                )
            } else if (carouselSet != null) {
                carouselSet.leftItem = leftItem
                carouselSet.rightItem = rightItem
                carouselSet.currentStartIndex = currentStartIndex
                carouselSetUp = carouselSet
            }
            showCarousel(notifOption, notifModel)
        }
    }


    /**
     * Sets carousel items, creates remoteviews,
     * creates notification and displays in tray
     */
    private fun showCarousel(notifOption: SMTNotificationOptions, notifModel: SMTNotificationData?) {
        carouselSetUp?.let { setup ->
            if (setup?.carouselItems?.size != 0) {
                setup.carouselItems?.let {
                    if (it.isNotEmpty()) {
                        // setsup carousel title
                        setUpCarouselTitles()

                        val remoteViews = RemoteViews(context.get()?.packageName, R.layout.notification_carousel_landscape_layout)

                        remoteViews.let { rv ->
                            setUpLandscapeCarouselVisibilities(rv)
                            setUpCarouselLandscapeItems(rv, notifOption)
                            setPendingIntentsForLandscapeCarousel(rv)
                        }
                        var mBuilder: NotificationCompat.Builder? = null
                        context.get()?.let { _context ->
                            mBuilder = getNotificationBuilder(_context,
                                    setup.contentTitle ?: "",
                                    setup.contentText ?: "",
                                    setup.contentSubtitle ?: "",
                                    createPendingIntent(setup),
                                    setup)
                        }
                        if (mBuilder == null) return

                        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                            remoteViews.setViewVisibility(R.id.carousel_large_icon, View.GONE)
                            mBuilder!!.setStyle(NotificationCompat.DecoratedCustomViewStyle())
                            mBuilder!!.setLargeIcon(null)
                        }
                        val foregroundNote = mBuilder!!.build()

                        @Suppress("DEPRECATION")
                        foregroundNote.bigContentView = remoteViews

                        val smtNotifMgr = context.get()?.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
                        smtNotifMgr?.notify(setup.carouselNotificationId, foregroundNote)

                    } else {
                        SMTLogger.e(TAG, "Empty item array or of length less than 2")
                    }
                }
            } else {

                context.get()?.let {
                    if (notifModel != null) {
                        SMTSimplePNGenerator().handle(it, notifModel)
                    }
                }
            }
        }

    }

    private fun createPendingIntent(setup: SMTCarouselSetup): PendingIntent {
        // Set Notificaiton Action receiver
        val carouselIntent = Intent(context.get(), SMTPNActionReceiver::class.java)

        val bundle = Bundle()
        bundle.putString(SMTNotificationConstants.NOTIF_TR_ID_KEY, setup.trid)
        bundle.putString(SMTNotificationConstants.NOTIF_TYPE_KEY, setup.notifType)
        bundle.putInt(SMTNotificationConstants.CAROUSEL_ITEM_CLICKED_KEY, SMTNotificationConstants.CAROUSEL_OTHER_REGION_CLICKED)
        bundle.putParcelable(SMTNotificationConstants.CAROUSEL_SET_UP_KEY, carouselSetUp)

        carouselIntent.putExtras(bundle)

        return PendingIntent.getBroadcast(context.get(), SMTCommonUtility.getRandomId(), carouselIntent, PendingIntent.FLAG_ONE_SHOT)
    }

    /**
     * Fetch carousel item bitmap from local storage
     * @param item - Carousel item data
     * @return Bitmap - Bitmap
     */
    private fun getCarouselBitmap(item: SMTCarouselItemData, notifOption: SMTNotificationOptions): Bitmap? {
        var bitmap: Bitmap? = null
        if (!TextUtils.isEmpty(item.mMediaLocalPath)) {
            bitmap = SMTCommonUtility.loadImageFromLocalStorage(item.mMediaLocalPath!!)

            if (bitmap != null)
                return bitmap
        }
        context.get()?.let {
            bitmap = SMTCommonUtility.getBitmapFromResId(it, notifOption.placeHolderIcon)

            if (bitmap == null) {
                bitmap = SMTCommonUtility.getBitmapFromResId(it, notifOption.largeIconId)
            }
        }

        return bitmap
    }

    /**
     * set up the carousel item visibility of widgets
     * @param remoteView - remote view of which widgets visibility to be set
     */
    private fun setUpLandscapeCarouselVisibilities(remoteView: RemoteViews) {
        // if carousel items size less than 2 dont show left and right arrows
        if (carouselSetUp?.carouselItems?.size!! < 2) {
            remoteView.setViewVisibility(R.id.carousel_arrow_left, View.GONE)
            remoteView.setViewVisibility(R.id.carousel_arrow_right, View.GONE)
        } else {
            remoteView.setViewVisibility(R.id.carousel_arrow_left, View.VISIBLE)
            remoteView.setViewVisibility(R.id.carousel_arrow_right, View.VISIBLE)
        }

        if (TextUtils.isEmpty(carouselSetUp?.bigContentText)) {
            //remoteView.setViewVisibility(R.id.carousel_message, View.GONE)
        } else {
            remoteView.setViewVisibility(R.id.carousel_message, View.VISIBLE)
        }
        if (TextUtils.isEmpty(carouselSetUp?.contentTitle)) {
            remoteView.setViewVisibility(R.id.carousel_title, View.GONE)
        } else {
            remoteView.setViewVisibility(R.id.carousel_title, View.VISIBLE)
        }
        if (TextUtils.isEmpty(carouselSetUp?.leftItem?.imgTitle)) {
            remoteView.setViewVisibility(R.id.carousel_image_title, View.GONE)
        } else {
            remoteView.setViewVisibility(R.id.carousel_image_title, View.VISIBLE)
        }
        if (TextUtils.isEmpty(carouselSetUp?.leftItem?.imgMsg)) {
            remoteView.setViewVisibility(R.id.carousel_image_message, View.GONE)
        } else {
            remoteView.setViewVisibility(R.id.carousel_image_message, View.VISIBLE)
        }

        if (!isImagesInCarousel) {
            remoteView.setViewVisibility(R.id.carousel_image, View.GONE)
        } else {
            remoteView.setViewVisibility(R.id.carousel_image, View.VISIBLE)
        }
    }

    /**
     * Set up carousel titles
     */
    private fun setUpCarouselTitles() {
        if (TextUtils.isEmpty(carouselSetUp?.contentTitle)) {
            context.get()?.let {
                carouselSetUp?.contentTitle = (SMTCommonUtility.getApplicationName(it))
            }
        }

        if (carouselSetUp?.contentTitle == null)
            carouselSetUp?.contentTitle = ""
        if (carouselSetUp?.contentText == null)
            carouselSetUp?.contentText = ""
    }

    /**
     * setup carousel items - like image, text
     * @param remoteVIew - for which image and titles has to be set
     */
    private fun setUpCarouselLandscapeItems(remoteVIew: RemoteViews, notifOption: SMTNotificationOptions) {

        carouselSetUp?.leftItem?.let {
            remoteVIew.setImageViewBitmap(R.id.carousel_image, getCarouselBitmap(it, notifOption))
        }

        context.get()?.let {
            remoteVIew.setImageViewBitmap(R.id.carousel_large_icon, SMTCommonUtility.getBitmapFromResId(it, notifOption.smallIconId))
        }

        remoteVIew.setTextViewText(R.id.carousel_title, carouselSetUp?.bigContentTitle)
        remoteVIew.setTextViewText(R.id.carousel_message, carouselSetUp?.bigContentText)
        remoteVIew.setTextViewText(R.id.carousel_image_title, carouselSetUp?.leftItem?.imgTitle)
        remoteVIew.setTextViewText(R.id.carousel_image_message, carouselSetUp?.leftItem?.imgMsg)
    }

    /**
     * Sets the actionable intents to the remote view
     */
    private fun setPendingIntentsForLandscapeCarousel(remoteViews: RemoteViews) {
        val rightArrowPendingIntent = getPendingIntent(SMTNotificationConstants.CAROUSEL_RIGHT_ARROW_CLICKED)
        remoteViews.setOnClickPendingIntent(R.id.carousel_arrow_right, rightArrowPendingIntent)

        val leftArrowPendingIntent = getPendingIntent(SMTNotificationConstants.CAROUSEL_LEFT_ARROW_CLICKED)
        remoteViews.setOnClickPendingIntent(R.id.carousel_arrow_left, leftArrowPendingIntent)

        val leftItemPendingIntent = getPendingIntent(SMTNotificationConstants.CAROUSEL_LEFT_ITEM_CLICKED)
        remoteViews.setOnClickPendingIntent(R.id.carousel_layout_item, leftItemPendingIntent)
    }

    /**
     * Creates actionable pending intent as per event id
     * @param eventClicked - event id - fow which the action has to be set
     */
    private fun getPendingIntent(eventClicked: Int): PendingIntent {
        val carouselIntent = Intent(context.get(), SMTPNActionReceiver::class.java)
        val bundle = Bundle()
        val type = carouselSetUp?.notifType ?: SMTNotificationType.CAROUSEL_PORTRAIT.type
        bundle.putString(SMTNotificationConstants.NOTIF_TYPE_KEY, type)
        bundle.putInt(SMTNotificationConstants.CAROUSEL_ITEM_CLICKED_KEY, eventClicked)
        bundle.putParcelable(SMTNotificationConstants.CAROUSEL_SET_UP_KEY, carouselSetUp)
        carouselIntent.putExtras(bundle)
        return PendingIntent.getBroadcast(context.get(), SMTCommonUtility.getRandomId(), carouselIntent, PendingIntent.FLAG_ONE_SHOT)
    }

    /**
     * Clear the existing carousel notificaiton and its related info from file
     */
    private fun clearCarouselIfExists(): SMTCarouselLandscapePNGenerator {
        carouselSetUp?.let { setup ->
            setup.carouselItems?.let { list ->
                list.forEach { item ->
                    item.mMediaLocalPath?.let {
                        SMTCommonUtility.deleteFile(it)
                    }
                }
            }

            isImagesInCarousel = true
            currentStartIndex = 0

            val mNotifyManager = context.get()?.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
            mNotifyManager.cancel(setup.carouselNotificationId)
        }
        carouselSetUp = null
        return this
    }

    /**
     * Handles  click event
     * @param clickEvent - Click event id
     * @param setUp - carousel parcel
     */
    fun handleClickEvent(clickEvent: Int, setUp: SMTCarouselSetup) {
        carouselSetUp = null
        verifyAndSetUpVariables(setUp)

        when (clickEvent) {
            SMTNotificationConstants.CAROUSEL_LEFT_ARROW_CLICKED -> onLeftArrowClicked()
            SMTNotificationConstants.CAROUSEL_RIGHT_ARROW_CLICKED -> onRightArrowClicked()
            SMTNotificationConstants.CAROUSEL_LEFT_ITEM_CLICKED -> onLeftItemClicked(setUp)
            else -> onOtherRegionClicked(setUp)
        }
    }

    /**
     * Base on clickevent sets up the parecel
     * @param setUp - received carousel parcel
     */
    private fun verifyAndSetUpVariables(setUp: SMTCarouselSetup) {
        if (carouselSetUp == null) {
            isImagesInCarousel = setUp.isImagesInCarousel
            currentStartIndex = setUp.currentStartIndex

            carouselSetUp = setUp

        } /*else if (carouselSetUp != null && carouselNotificationId != setUp.carouselNotificationId) {
            carouselSetUp = null
            verifyAndSetUpVariables(setUp)
        }*/
    }

    /**
     * Handles Left Carousel item click
     */
    private fun onLeftItemClicked(setup: SMTCarouselSetup) {
        sendItemClickedBroadcast(context, carouselSetUp?.leftItem?.imgDeeplink ?: "", setup)
    }

    /**
     * Handles on click of other region of carousel
     */
    private fun onOtherRegionClicked(setup: SMTCarouselSetup) {
        try {
            sendItemClickedBroadcast(context, carouselSetUp?.deeplink ?: "", setup)
        } catch (e: Exception) {
            e.printStackTrace()
            SMTLogger.e(TAG, "Unable to send notification's pendingIntent")
        }

    }

    /**
     * Broadcasts the item clicked and clear the existing carousel
     * @param context App context
     * @param itemLink deeplink path
     */
    private fun sendItemClickedBroadcast(context: WeakReference<Context>, itemLink: String, setup: SMTCarouselSetup) {

        context.get()?.let {
            SMTCommonUtility.handleNotificationClick(it, itemLink, setup.customPayload)
            SMTEventRecorder.getInstance(it).recordNotificationClick(setup.trid
                    ?: "", setup.pnMeta,
                    itemLink, setup.source, setup.smtAttributePayload ?: HashMap())
        }



        clearCarouselIfExists()
    }

    /**
     * Handles carousel Left navigation
     */
    private fun onLeftArrowClicked() {
        carouselSetUp?.let { setup ->
            setup.carouselItems?.let {
                // If currentStartIndex is last carousel item
                if (currentStartIndex == 0) {
                    currentStartIndex = it.size - 1
                } else {
                    --currentStartIndex
                }
                prepareVariablesForCarouselAndShow(it[(currentStartIndex)], null, null, setup)
            }
        }

    }

    /**
     * Handles carousel Right navigation
     */
    private fun onRightArrowClicked() {
        carouselSetUp?.let { setup ->
            setup.carouselItems?.let {
                if (it.size > currentStartIndex) {
                    if (currentStartIndex == (it.size - 1)) {
                        currentStartIndex = 0
                    } else {
                        ++currentStartIndex
                    }
                    prepareVariablesForCarouselAndShow(it[currentStartIndex], null, null, setup)
                }
            }
        }
    }

    /**
     * Handle notification dismiss
     */
    override fun handleNotificationDismiss(context: Context?, extras: Bundle) {
        if (extras.containsKey(SMTNotificationConstants.NOTIFICATION_PARCEL)) {
            val parcel = extras.getParcelable(SMTNotificationConstants.NOTIFICATION_PARCEL) as? SMTCarouselSetup
            carouselSetUp = parcel
        }
        clearCarouselIfExists()
    }
}