package com.netcore.android.notification.audio

import android.app.Notification
import android.app.NotificationManager
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.os.Build
import android.os.Bundle
import android.view.View
import androidx.core.app.NotificationCompat
import android.widget.RemoteViews
import com.netcore.android.R
import com.netcore.android.event.SMTEventId
import com.netcore.android.event.SMTEventRecorder
import com.netcore.android.logger.SMTLogger
import com.netcore.android.mediadownloader.SMTMediaDownloadManager
import com.netcore.android.notification.SMTBaseNotificationGenerator
import com.netcore.android.notification.SMTNotificationConstants
import com.netcore.android.notification.SMTNotificationType
import com.netcore.android.notification.SMTPNActionReceiver
import com.netcore.android.notification.models.SMTNotificationData
import com.netcore.android.utility.SMTCommonUtility
import org.json.JSONObject
import java.lang.ref.WeakReference


/**
 * Handles Audio notification
 */
internal class SMTRichAudioPNGenerator private constructor() : SMTBaseNotificationGenerator() {

    private val TAG = SMTRichAudioPNGenerator::class.java.simpleName

    private lateinit var nBuilder: NotificationCompat.Builder

    companion object {

        @Volatile
        private var INSTANCE: SMTRichAudioPNGenerator? = null

        internal fun getInstance(): SMTRichAudioPNGenerator =
                INSTANCE ?: synchronized(SMTRichAudioPNGenerator::class.java) {
                    INSTANCE ?: buildInstance().also { INSTANCE = it }
                }

        private fun buildInstance(): SMTRichAudioPNGenerator {
            return SMTRichAudioPNGenerator()
        }

    }

    /**
     * Notification Action receiver content Pending Intent created
     * @param context - App context
     * @param audioParcel - Audio realted info
     * @return PendingIntent - receiver pending intent
     */
    private fun createPendingIntent(context: Context, audioParcel: SMTNotificationData): PendingIntent {
        val rand = SMTCommonUtility.getRandomId()

        audioParcel.action = SMTNotificationConstants.AUDIO_NOTIF_OTHER_REGION_CLICKED

        val bundle = Bundle()

        bundle.putString(SMTNotificationConstants.NOTIF_TYPE_KEY, SMTNotificationType.AUDIO.type)
        bundle.putString(SMTNotificationConstants.AUDIO_NOTIF_CLICKED_KEY, SMTNotificationConstants.AUDIO_NOTIF_OTHER_REGION_CLICKED)
        bundle.putParcelable(SMTNotificationConstants.NOTIFICATION_PARCEL, audioParcel)

        val launchIntent = Intent(context, SMTPNActionReceiver::class.java)
        launchIntent.putExtras(bundle)

        return PendingIntent.getBroadcast(context, rand, launchIntent, PendingIntent.FLAG_UPDATE_CURRENT)
    }

    /**
     * Creates a Custom Notifications that is usually used by music player apps.
     * @param [context] application context for associate the notification with.
     */
    internal fun handle(context: Context, notifModel: SMTNotificationData) {

        notifModel.notificationId = SMTCommonUtility.getRandomId()

        updateNotificationId(context, notifModel)
        /**
         * If media url is empty directly show notification
         * else download media and start music service and show notificaiton
         * it media download fails then also show notification directly
         */
        if (notifModel.mMediaUrl?.isEmpty() == true) {
            notifModel.isPlaying = false
            notifModel.action = SMTMusicService.ACTION_NO_MUSIC
            notifModel.mMediaLocalPath = ""
            showNotification(context, notifModel, true)
        } else {
            // Download Media file
            SMTMediaDownloadManager().downloadMedia(context, notifModel, object : SMTMediaDownloadManager.MediaDownloadListener {

                override fun onDownloadSuccess(notification: SMTNotificationData) {
                    // On success start Music service and show the notification
                    startMusicServiceAndShowNotification(context, notifModel)
                }

                override fun onDownloadFailed(notification: SMTNotificationData) {
                    notifModel.isPlaying = false
                    notifModel.action = SMTMusicService.ACTION_NO_MUSIC
                    notifModel.mMediaLocalPath = ""
                    showNotification(context, notifModel, true)
                }
            })
        }
    }

    /*private fun clearExistingNotificationDataAndService(context: Context, audioParcel: SMTNotificationData?) {
        */
    /**
     * Stops music service if its already running
     *//*
//        stopMusicService(context)

        */
    /**
     * Clear the existing notification
     *//*
        audioParcel?.let {
//            clearNotificationFromTray(context, it)
        }
    }*/

    /**
     * clears existing notification
     */
    private fun clearNotificationFromTray(context: Context, audioParcel: SMTNotificationData) {
        val mNotifyManager = context.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
        mNotifyManager.cancel(audioParcel.notificationId)
    }

    private fun deleteMediaFile(audioParcel: SMTNotificationData) {
        audioParcel.mMediaLocalPath?.let {
            SMTCommonUtility.deleteFile(it)
        }
    }

    /**
     * Stops music service
     */
    /*private fun stopMusicService(context: Context) {
        val serviceIntent = Intent(context, SMTMusicService::class.java)
//        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            serviceIntent.action = SMTMusicService.ACTION_STOP
//        } else {
            context.stopService(serviceIntent)
//        }
    }*/

    /**
     * Starts music service and also shows notficaiton in the tray
     */
    private fun startMusicServiceAndShowNotification(context: Context, notifModel: SMTNotificationData) {

        notifModel.isPlaying = false
        notifModel.action = SMTMusicService.ACTION_INIT

        val bundle = Bundle()
        bundle.putParcelable(SMTNotificationConstants.NOTIFICATION_PARCEL, notifModel)

        /*val serviceIntent = Intent(context, SMTMusicService::class.java)
        serviceIntent.putExtras(bundle)

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            context.startForegroundService(serviceIntent)
        } else {
            context.startService(serviceIntent)
        }*/

        showNotification(context, notifModel, true)

    }


    /**
     * Handles of creating remoteview, notification builder,
     * delete intent, content intent, and shows notification
     * @param context - App context
     * @param audioParcel - Audio parcel data
     */
    internal fun showNotification(context: Context, audioParcel: SMTNotificationData, showNotification: Boolean): Notification {

        notificationManager = context.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager

        val title = audioParcel.mTitle ?: ""
        val message = audioParcel.mMessage ?: ""
        val subtitle = audioParcel.mSubtitle ?: ""

        val remoteViews = createRemoteView(context, audioParcel)

        // Build the content of the notification
        nBuilder = getNotificationBuilder(context,
                title,
                message,
                subtitle,
                createPendingIntent(context, audioParcel),
                audioParcel)

        val notif = getNotificationInstance(remoteViews)
        if (showNotification) {
            notificationManager?.notify(audioParcel.notificationId, notif)
        }

        return notif
    }

    internal fun clearForegroundNotificationAndShowSimpleNotification(context: Context, audioParcel: SMTNotificationData) {
        notificationManager = context.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
        notificationManager?.cancel(audioParcel.notificationId)

        showNotification(context, audioParcel, true)
    }

    private fun notifyService(context: Context, audioParcel: SMTNotificationData) {
        val serviceIntent = Intent(context, SMTMusicService::class.java)
        serviceIntent.action = SMTMusicService.ACTION_PLAY_PAUSE
        audioParcel.action = SMTMusicService.ACTION_PLAY_PAUSE
        val bundle = Bundle()
        bundle.putParcelable(SMTNotificationConstants.NOTIFICATION_PARCEL, audioParcel)

        serviceIntent.putExtras(bundle)
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            context.startForegroundService(serviceIntent)
        } else {
            context.startService(serviceIntent)
        }
    }

    private fun getNotificationInstance(remoteViews: RemoteViews): Notification {
        // Notification through notification manager
        val notificationObj: Notification
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            remoteViews.setViewVisibility(R.id.audio_large_icon, View.GONE)
            nBuilder.setCustomBigContentView(remoteViews)
            nBuilder.setStyle(NotificationCompat.DecoratedCustomViewStyle())
            nBuilder.setLargeIcon(null)
            notificationObj = nBuilder.build()
        } else {
            notificationObj = nBuilder.build()
            @Suppress("DEPRECATION")
            notificationObj.bigContentView = remoteViews
        }
        return notificationObj
    }

    /**
     * Called when user clicked play/pause icon
     * or clicked on other region of the notification
     * @param context - App Context
     * @param extras - Bundle containing audio parcelable data
     */
    internal fun handleAudioClick(context: Context, extras: Bundle) {

        val audioParcel = (if (extras.containsKey(SMTNotificationConstants.NOTIFICATION_PARCEL)) {
            extras.get(SMTNotificationConstants.NOTIFICATION_PARCEL) as? SMTNotificationData
        } else {
            null
        }) ?: return

        val clickedKey = if (extras.containsKey(SMTNotificationConstants.AUDIO_NOTIF_CLICKED_KEY)) {
            extras.get(SMTNotificationConstants.AUDIO_NOTIF_CLICKED_KEY)
        } else {
            SMTMusicService.ACTION_PAUSE
        }
        when (clickedKey) {
            SMTMusicService.ACTION_PLAY -> {
                audioParcel.isPlaying = true
                // update play/pause button in notification
                notifyService(context, audioParcel)
            }
            SMTMusicService.ACTION_PAUSE -> {
                // update play/pause button in notification
                audioParcel.isPlaying = false
                notifyService(context, audioParcel)
            }
            SMTMusicService.ACTION_CLOSE -> {
                // update play/pause button in notification
                audioParcel.isPlaying = false
                dismissNotification(extras, context, true)
            }
            SMTNotificationConstants.AUDIO_NOTIF_OTHER_REGION_CLICKED -> {
                handleOtherRegionClicked(context, audioParcel)
                return
            }
        }
    }

    /**
     * Creates the remoteview as per Audio play status
     * @param context - app context
     * @param audioParcel - Audio parcel data
     */
    private fun createRemoteView(context: Context, audioParcel: SMTNotificationData): RemoteViews {
        val notifOptions = SMTCommonUtility.getNotificationOptions(context)
        // Inflate layout
        val remoteViews = RemoteViews(context.packageName, R.layout.notification_audio_layout)
        // set play / pause icon based on audio play status
        if (audioParcel.isPlaying == true) {
            remoteViews.setImageViewResource(R.id.audio_icon_play, R.drawable.ic_audio_pause)
        } else {
            remoteViews.setImageViewResource(R.id.audio_icon_play, R.drawable.ic_audio_play)
        }

        remoteViews.setImageViewResource(R.id.audio_icon_close, R.drawable.ic_audio_close)
        remoteViews.setImageViewResource(R.id.audio_large_icon, notifOptions.largeIconId)

        // if mMediaLocalPath is empty then hide the icons
        if (audioParcel.mMediaLocalPath?.isEmpty() == true) {
            remoteViews.setViewVisibility(R.id.audio_icon_play, View.GONE)
            remoteViews.setViewVisibility(R.id.audio_icon_close, View.GONE)
        }

        remoteViews.setTextViewText(R.id.audio_title, audioParcel.mTitle)
        remoteViews.setTextViewText(R.id.audio_message, audioParcel.mMessage)

        /**
         * Set listener to the Remote view
         */
        setListeners(remoteViews, context, audioParcel)

        return remoteViews
    }

    /**
     * Handle the Audio control buttons like play and pause.
     * @param [remoteView] remote view for big content.
     * @param [context] application context for associate the notification with.
     * @param [audioParcel] Notification Data
     */
    private fun setListeners(remoteView: RemoteViews, context: Context, audioParcel: SMTNotificationData) {
        /**
         * If music is playing then set listener for pause
         */
        if (audioParcel.isPlaying == true) {
            val pendingIntentPlay = getPendingIntent(context, SMTMusicService.ACTION_PAUSE, audioParcel)//PendingIntent.getBroadcast(context, CommonUtility.getRandomId(), intentPlay, PendingIntent.FLAG_UPDATE_CURRENT)
            remoteView.setOnClickPendingIntent(R.id.audio_icon_play, pendingIntentPlay)
        }
        /**
         * If music is playing then set listener for play
         */
        else {
            val pendingIntentPause = getPendingIntent(context, SMTMusicService.ACTION_PLAY, audioParcel)//PendingIntent.getBroadcast(context, CommonUtility.getRandomId(), intentPause, PendingIntent.FLAG_UPDATE_CURRENT)
            remoteView.setOnClickPendingIntent(R.id.audio_icon_play, pendingIntentPause)
        }
        val pendingIntentClose = getPendingIntent(context, SMTMusicService.ACTION_CLOSE, audioParcel)//PendingIntent.getBroadcast(context, CommonUtility.getRandomId(), intentClose, PendingIntent.FLAG_ONE_SHOT)
        remoteView.setOnClickPendingIntent(R.id.audio_icon_close, pendingIntentClose)
    }

    /**
     * Creates actionable pending intent as per event id
     * @param eventClicked - event id - fow which the action has to be set
     */
    private fun getPendingIntent(context: Context, eventClicked: String, audioParcel: SMTNotificationData): PendingIntent {

        val pendingIntent = Intent(context, SMTPNActionReceiver::class.java)

        val bundle = Bundle()

        bundle.putString(SMTNotificationConstants.NOTIF_TYPE_KEY, SMTNotificationType.AUDIO.type)
        bundle.putString(SMTNotificationConstants.AUDIO_NOTIF_CLICKED_KEY, eventClicked)
        bundle.putParcelable(SMTNotificationConstants.NOTIFICATION_PARCEL, audioParcel)

        pendingIntent.putExtras(bundle)

        return PendingIntent.getBroadcast(context, SMTCommonUtility.getRandomId(), pendingIntent, PendingIntent.FLAG_ONE_SHOT)
    }

    /**
     * Upon Audio notification clicked handle deeplinking
     * and stop the music service as well
     * @param context - App context
     * @param audioParcel - Audio parcel data
     */
    private fun handleOtherRegionClicked(context: Context, audioParcel: SMTNotificationData?) {
        try {
            audioParcel?.let {
                SMTCommonUtility.handleNotificationClick(context, it.mDeepLinkPath
                        ?: "", audioParcel.mCustomPayload)
                notifyStopService(context, audioParcel)
                deleteMediaFile(it)
                clearNotificationFromTray(context, it)
                recordAudioEvent(context, SMTEventId.EVENT_PN_CLICKED, it.mDeepLinkPath, it)
            }
        } catch (e: Exception) {
            e.printStackTrace()
            SMTLogger.e(TAG, "Unable to send notification's pendingIntent")
        }

    }

    /**
     * Handle Notification dismiss event like clearing heap and downloaded data
     * @param context - App context
     * @param extras - Audio parcel data
     */
    override fun handleNotificationDismiss(context: Context?, extras: Bundle) {
        context?.let {
            if (extras.containsKey(SMTNotificationConstants.NOTIFICATION_PARCEL)) {
                dismissNotification(extras, it, false)
            }
        }
    }

    internal fun dismissNotification(extras: Bundle, context: Context, isRecord: Boolean) {
        val notifData: SMTNotificationData? = extras.getParcelable(SMTNotificationConstants.NOTIFICATION_PARCEL)
        notifData?.let { data ->
            notifyStopService(context, data)
            deleteMediaFile(data)
            if (isRecord) {
                clearNotificationFromTray(context, data)
                recordAudioEvent(context, SMTEventId.EVENT_PN_DISMISSED, null, data)
            }
        }
    }

    /**
     * From Stop service the onDestroy of MusicService does not get called
     * So deleting the downloaded media manually
     * @param context - App context
     * @param audioParcel - Audio payload
     */
    private fun notifyStopService(context: Context, audioParcel: SMTNotificationData?) {
        audioParcel?.let {
            val serviceIntent = Intent(context, SMTMusicService::class.java)

            it.action = SMTMusicService.ACTION_STOP

            val bundle = Bundle()
            bundle.putParcelable(SMTNotificationConstants.NOTIFICATION_PARCEL, audioParcel)

            serviceIntent.putExtras(bundle)
            context.startService(serviceIntent)
        }
    }

    private fun recordAudioEvent(context: Context, eventId: Int, deeplink: String?, notifModel: SMTNotificationData) {
        when (eventId) {
            SMTEventId.EVENT_PN_CLICKED -> {
                SMTEventRecorder.getInstance(context).recordNotificationClick(notifModel.mTrid, notifModel.mPNMeta,
                        deeplink ?: "", notifModel.mSource, notifModel.mSmtAttributePayload
                        ?: HashMap())
            }

            SMTEventId.EVENT_PN_DISMISSED -> {
                SMTEventRecorder.getInstance(context).recordNotificationDismiss(notifModel.mTrid, notifModel.mPNMeta, notifModel.mSource)
            }
        }
    }
}