package com.netcore.android.notification

import android.content.Context
import android.util.Log
import com.netcore.android.SMTConfigConstants
import com.netcore.android.SMTEventParamKeys
import com.netcore.android.db.SMTDataBaseService
import com.netcore.android.event.SMTEventId
import com.netcore.android.event.SMTEventRecorder
import com.netcore.android.event.SMTEventType
import com.netcore.android.event.SMTNotificationSourceType
import com.netcore.android.logger.SMTLogger
import com.netcore.android.preference.SMTPreferenceConstants
import com.netcore.android.preference.SMTPreferenceHelper
import com.netcore.android.utility.SMTGWSource
import com.netcore.android.workmgr.SMTWorkerScheduler
import org.json.JSONObject
import java.lang.ref.WeakReference
import java.util.concurrent.TimeUnit

/**
 *  @author Netcore
 *
 *  This class is utility class for Notification.
 */
internal class SMTNotificationUtility private constructor() {

    companion object {
        private var INSTANCE: SMTNotificationUtility? = null
        private val TAG = SMTNotificationUtility::class.java.simpleName

        fun getInstance() = INSTANCE
                ?: synchronized(SMTNotificationUtility::class.java) {
                    INSTANCE
                            ?: buildInstance().also { INSTANCE = it }
                }

        private fun buildInstance(): SMTNotificationUtility {
            return SMTNotificationUtility()
        }
    }

    fun processOpenAndDeliverNotificationEvents(context: Context, notificationObject: JSONObject, @SMTNotificationSourceType.Source sourceType: Int, smtNotificationEvent: SMTNotificationEventType) {

        val payload = notificationObject.toString()
        val mNotificationModel = SMTNotificationParser().parse(payload, sourceType)
        if (mNotificationModel?.isFromSmartech == false) {
            return
        }
        // Insert Notification into Notification table
        mNotificationModel?.mTrid?.let {

            when (smtNotificationEvent) {

                SMTNotificationEventType.DELIVERY -> {

                    // If Notification already present in the DB, then don't process it again, just ignore it
                    if (!SMTDataBaseService.getInstance(WeakReference(context)).findNotificationWithId(it)) {
                        val dbService = SMTDataBaseService.getInstance(WeakReference(context))
                        dbService.insertPNToNotificationTable(it, payload, sourceType)

                        SMTEventRecorder.getInstance(context).recordNotificationDelivery(it, payload,
                                sourceType, mNotificationModel!!)
                    } else {
                        SMTLogger.i("Utility", "Notification delivery event sent to user for trId : $it")
                        return
                    }
                }

                SMTNotificationEventType.OPEN -> {
                    //Update DB record whether it has been clicked
                    if (!SMTDataBaseService.getInstance(WeakReference(context)).findNotificationReadStatusWithId(it)) {
                        SMTEventRecorder.getInstance(context).recordNotificationClick(it, mNotificationModel.mPNMeta,
                                mNotificationModel.mDeepLinkPath
                                        ?: "", mNotificationModel.mSource, mNotificationModel.mSmtAttributePayload
                                ?: HashMap())
                    } else {
                        SMTLogger.i("Utility", "Notification read event sent to user for trId : $it")
                        return
                    }
                }

                else -> {
                }
            }
            // Schedule event sync call
            SMTWorkerScheduler.getInstance().checkStatusAndScheduleEventWorker(context)
        }
    }

    fun checkIfNotificationListenerEnabled(context: Context): Int {
        val preferences = SMTPreferenceHelper.getAppPreferenceInstance(context, null)
        return preferences.getInt(SMTPreferenceConstants.SMT_MF_IS_NOTIFICATION_LISTENER_ENABLED, 0)
    }

    fun getDevicePushToken(context: Context): String {
        val preferences = SMTPreferenceHelper.getAppPreferenceInstance(context, null)
        return preferences.getString(SMTPreferenceConstants.PUSH_TOKEN_CURRENT, "")
    }

    @Synchronized
    fun setPushToken(context: Context?, token: String?) {
        if (!token.isNullOrEmpty()) {
            context?.let {
                val preference = SMTPreferenceHelper.getAppPreferenceInstance(it, null)
                val currentToken = preference.getString(SMTPreferenceConstants.PUSH_TOKEN_CURRENT, "")

                if (currentToken.isNotEmpty() && currentToken == token) {
                    SMTLogger.i(TAG, "Current Token and Old Token is same.")
                    return
                }

                val tokenEventIDType = when (currentToken.isNotEmpty()) {
                    true -> {
                        preference.setString(SMTPreferenceConstants.PUSH_TOKEN_OLD, currentToken)
                        preference.setString(SMTPreferenceConstants.FCM_PUSH_TOKEN_OLD, currentToken)
                        SMTEventId.EVENT_PN_TOKEN_REFRESHED
                    }
                    false -> {
                        preference.setString(SMTPreferenceConstants.PUSH_TOKEN_OLD, token)
                        preference.setString(SMTPreferenceConstants.FCM_PUSH_TOKEN_OLD, token)
                        SMTEventId.EVENT_PN_TOKEN_GENERATED
                    }
                }

                preference.setString(SMTPreferenceConstants.PUSH_TOKEN_CURRENT, token)
                preference.setString(SMTPreferenceConstants.FCM_PUSH_TOKEN_CURRENT, token)

                val payload = HashMap<String, Any>()
                payload[SMTEventParamKeys.SMT_GWSOURCE] = SMTGWSource.FCM.value

                SMTEventRecorder.getInstance(it).recordEvent(tokenEventIDType,
                        SMTEventId.getEventName(tokenEventIDType), payload, SMTEventType.EVENT_TYPE_SYSTEM)

                SMTLogger.internal(TAG, "Old Token: ${preference.getString(SMTPreferenceConstants.FCM_PUSH_TOKEN_OLD, "")}")
                SMTLogger.internal(TAG, "New Token: ${preference.getString(SMTPreferenceConstants.FCM_PUSH_TOKEN_CURRENT, "")}")
            }
        } else {
            SMTLogger.i(TAG, "Token is either null or empty.")
        }
    }

    internal fun handlingPNTokenGenerationEvent(context: Context) {
        val preference = SMTPreferenceHelper.getAppPreferenceInstance(context, null)
        val pnTokenTimestamp = preference.getLong(SMTPreferenceConstants.PN_TOKEN_TIMESTAMP, System.currentTimeMillis())
        val timestampDifference = System.currentTimeMillis() - pnTokenTimestamp
        val timeDifference: Long = TimeUnit.MILLISECONDS.toMinutes(timestampDifference)
        val tokenFrequency = preference.getInt(SMTPreferenceConstants.TOKEN_INTERVAL, SMTConfigConstants.DEFAULT_TOKEN_INTERVAL)
        val currentToken = preference.getString(SMTPreferenceConstants.PUSH_TOKEN_CURRENT, "")
        SMTLogger.internal("PNTokenGenerated", "PN token details $timeDifference $tokenFrequency $timestampDifference token:$currentToken")
        if (timeDifference > tokenFrequency && currentToken.isNotEmpty()) {
            val payload = HashMap<String, Any>()
            payload[SMTEventParamKeys.SMT_GWSOURCE] = SMTGWSource.FCM.value

            SMTEventRecorder.getInstance(context).recordEvent(SMTEventId.EVENT_PN_TOKEN_GENERATED,
                    SMTEventId.getEventName(SMTEventId.EVENT_PN_TOKEN_GENERATED), payload, SMTEventType.EVENT_TYPE_SYSTEM)
            SMTLogger.internal("PNTokenGenerated", "PN token generated event sent.")
        } else {
            SMTLogger.internal("PNTokenGenerated", "time interval didn't match")
        }

        preference.setLong(SMTPreferenceConstants.PN_TOKEN_TIMESTAMP, System.currentTimeMillis())
    }


    fun checkNotificationSource(payload: JSONObject?): Boolean {

        var value = try {
            payload?.let {
                if (it.has(SMTNotificationConstants.NOTIF_SOURCE_KEY)) {
                    (SMTNotificationConstants.NOTIF_SOURCE_VALUE.equals(it.optString(SMTNotificationConstants.NOTIF_SOURCE_KEY), true))
                } else false
            } ?: false
        } catch (e: Exception) {
            e.printStackTrace()
            false
        }
        return value
    }
}