package com.netcore.android.event


import android.content.Context
import android.os.Handler
import android.os.HandlerThread
import com.netcore.android.SMTConfigConstants
import com.netcore.android.db.SMTDataBaseService
import com.netcore.android.db.SMTEventTable
import com.netcore.android.logger.SMTLogger
import com.netcore.android.network.SMTEnumHttpMethodType
import com.netcore.android.network.models.SMTRequest
import com.netcore.android.notification.models.SMTEventPayload
import com.netcore.android.preference.SMTPreferenceConstants
import com.netcore.android.preference.SMTPreferenceHelper
import org.json.JSONArray
import org.json.JSONObject
import java.lang.ref.WeakReference

@Suppress("PrivatePropertyName")
/**
 * @Description
 *
 * Batch Processor handler
 * Handles batch processing api call, scheduling of next batch
 *
 * @param context app context
 *
 * @author Netcore
 * @version 1.0
 * @since 25-03-2019
 */
internal class SMTEventsBatchProcessor private constructor(val context: WeakReference<Context>) {

    private val TAG = SMTEventsBatchProcessor::class.java.simpleName
    private val lock = Any()

    companion object {
        @Volatile
        private var INSTANCE: SMTEventsBatchProcessor? = null
        private lateinit var sharedPreferences: SMTPreferenceHelper
        private lateinit var mHandler: Handler
        private lateinit var mHandlerThread: HandlerThread


        /**
         * Getting instance of the class
         */
        fun getInstance(context: Context): SMTEventsBatchProcessor =
                INSTANCE ?: synchronized(SMTEventsBatchProcessor::class.java) {
                    INSTANCE ?: buildInstance(context).also { INSTANCE = it }
                }

        private fun buildInstance(context: Context): SMTEventsBatchProcessor {
            sharedPreferences = SMTPreferenceHelper.getAppPreferenceInstance(context, null)
            mHandlerThread = HandlerThread("EventBatchProcessor_Thread")
            mHandlerThread.start()
            mHandler = Handler(mHandlerThread.looper)
            return SMTEventsBatchProcessor(WeakReference(context))
        }
    }

    /**
     * Create SMTRequest
     */
    fun getBatchProcessRequest(params: JSONArray): SMTRequest {
        val requestBuilder = SMTRequest.Builder()
                .setHttpMethod(SMTEnumHttpMethodType.POST)
                .setEndPoint(getBatchProcessingApiEndPoint())
                .setApiId(SMTRequest.SMTApiTypeID.BATCH_PROCESSING_API)
                .setParams(params)
                .setBaseUrl(sharedPreferences.getString(SMTPreferenceConstants.SMT_BASE_URL_TRACKAPPACT))
        //.setBaseUrl(sharedPreferences.getString(SMTPreferenceConstants.SMT_BASE_URL))
        return requestBuilder.build()
    }


    private fun getBatchProcessingApiEndPoint(): String {
        var urlParams = ""
        context.get()?.let {
            urlParams = SMTEventCommonDataDump(it).getURLParameters()
        }
        return "${SMTConfigConstants.TRACK_API_PATH}?$urlParams"
    }

    /**
     * Checks for more events if there is present in the event table for processing
     */
    fun checkIfMoreEventsPresentForBatchRequest(context: WeakReference<Context>, state: SMTWorkerType): Boolean {
        val batchSize = sharedPreferences.getInt(SMTPreferenceConstants.BATCH_SIZE)
        return when (state) {
            SMTWorkerType.EventWorker -> SMTDataBaseService.getInstance(context).checkIfMoreEventsPresentForBatchRequest(batchSize)
            SMTWorkerType.BackgroundSyncWorker -> SMTDataBaseService.getInstance(context).checkIfPendingFailedEventsBatchRequest(batchSize)
        }
    }

    /**
     * Prepares the batch request by fetching the events from DB
     * Creates the SMTRequest
     * Updates the DB with sync status to Inrpogress
     */
    fun prepareBatchRequest(context: WeakReference<Context>, state: SMTWorkerType): SMTEventPayload {
        var jsonArray: JSONArray
        synchronized(lock) {
            SMTLogger.v(TAG, "Creating batch request")
            jsonArray = JSONArray()
            //update the table to mark the processing the events in progress
            val arrayList = ArrayList<Int>()
            context.get()?.let {
                val batchSize = sharedPreferences.getInt(SMTPreferenceConstants.BATCH_SIZE)


                var batchMap = when (state) {
                    SMTWorkerType.EventWorker -> SMTDataBaseService.getInstance(context).getPendingEventsMapForBatchProcessing(batchSize, 0)
                    SMTWorkerType.BackgroundSyncWorker -> SMTDataBaseService.getInstance(context).getPendingFailedEventsMapForBatchProcessing(batchSize, 0)
                }

                // Add common data dump for batch , it should be first item in the array
                if (batchMap.size > 0) {
                    jsonArray.put(JSONObject(SMTEventCommonDataDump(it).getPayloadParams()))

                    try {
                        for ((key, value) in batchMap) {
                            jsonArray.put(JSONObject(value))
                            arrayList.add(key.toInt())
                        }
                    } catch (e: Exception) {
                        SMTLogger.e(TAG, "error while creating batch ---$e")
                    }

                    // Update the sync status of events
                    SMTDataBaseService.getInstance(context).updateEventProcessingStatus(arrayList.toTypedArray(),
                            SMTEventTable.KEY_EVENT_SYNC_STATUS, SMTEventSyncStatus.EVENT_SYNC_STATUS_INPROGRESS)
                }
            }
            SMTLogger.i(TAG, "Batch Details Size: ${jsonArray.length()} and Batch details: ${jsonArray.toString()}")
            return SMTEventPayload(jsonArray, arrayList.toTypedArray())
        }
    }
}

enum class SMTWorkerType {
    EventWorker, BackgroundSyncWorker
}