package com.netcore.android.event

import android.content.Context
import android.os.Handler
import android.os.HandlerThread
import android.util.Log
import androidx.work.WorkManager
import com.netcore.android.SMTWorkManagerConst
import com.netcore.android.db.SMTDataBaseService
import com.netcore.android.logger.SMTLogger
import com.netcore.android.preference.SMTPreferenceConstants
import com.netcore.android.preference.SMTPreferenceHelper
import com.netcore.android.utility.SMTCommonUtility
import com.netcore.android.workmgr.SMTWorkerScheduler
import java.lang.ref.WeakReference

@Suppress("PrivatePropertyName")
/**
 * @Description
 *
 * Batch Processor handler
 * Handles batch processing api call, scheduling of next batch
 *
 * @param context app context
 *
 * @author Netcore
 * @version 1.0
 * @since 25-03-2019
 */
internal class SMTEventBatchProcessor private constructor(val context: WeakReference<Context>) {

    private val TAG = SMTEventBatchProcessor::class.java.simpleName

    companion object {
        @Volatile
        private var INSTANCE: SMTEventBatchProcessor? = null
        private lateinit var sharedPreferences: SMTPreferenceHelper
        private lateinit var mHandler: Handler
        private lateinit var mHandlerThread: HandlerThread


        /**
         * Getting instance of the class
         */
        fun getInstance(context: Context): SMTEventBatchProcessor =
                INSTANCE ?: synchronized(SMTEventBatchProcessor::class.java) {
                    INSTANCE ?: buildInstance(context).also { INSTANCE = it }
                }

        private fun buildInstance(context: Context): SMTEventBatchProcessor {
            sharedPreferences = SMTPreferenceHelper.getAppPreferenceInstance(context, null)
            mHandlerThread = HandlerThread("EventBatchProcessor_Thread")
            mHandlerThread.start()
            mHandler = Handler(mHandlerThread.looper)
            return SMTEventBatchProcessor(WeakReference(context))
        }
    }

    /**
     * schedules event batch processing
     * @param scheduleTime time for scheduling
     * if scheduleTime is 0, then use stored batch interval time
     * else use it
     */
    private fun scheduleEventBatchProcessing(scheduleTime: Long) {

        val batchProcessRunner = getBatchProcessRunner()
        var batchInterval = if (scheduleTime == 0L) {
            sharedPreferences.getInt(SMTPreferenceConstants.BATCH_INTERVAL, 5) * 1000L // converting into mills
        } else {
            scheduleTime
        }

        // precaution check, if 0 seconds then changing it to 5 seconds
        if (batchInterval <= 0) {
            batchInterval = 5 * 1000L
        }
        SMTLogger.internal(TAG, "scheduling next batch process in $batchInterval milli seconds and thread name is ${Thread.currentThread().name}")
        mHandler.removeCallbacksAndMessages(null)
        mHandler.postDelayed(batchProcessRunner, batchInterval)
    }

    /**
     * calls Batch processing API
     * If there is any event then process the request
     * else schedule for next batch process api call
     * @param callApiInNewThread whether to make api call on new thread or not
     */
    internal fun makeEventBatchProcessingApiCall() {

        context.get()?.let {
            val workManager = WorkManager.getInstance(it)
            val workerStatus = SMTWorkerScheduler.getInstance().isWorkerRunning(workManager, SMTWorkManagerConst.SMT_EVENTSYNC_WORKER_TAG)
            val batchSize = sharedPreferences.getInt(SMTPreferenceConstants.BATCH_SIZE)
            val isMoreEventsPresent = SMTDataBaseService.getInstance(context).checkIfMoreEventsPresentForBatchRequest(batchSize)
            // Check SDK is active or not
            val isTrackingAllowed = SMTCommonUtility.checkIfTrackingAllowed(it)
            SMTLogger.internal(TAG, "worker status is running : $workerStatus  Events in DB: $isMoreEventsPresent tracking status: $isTrackingAllowed")
            if (!workerStatus && isMoreEventsPresent && isTrackingAllowed) {
                SMTWorkerScheduler.getInstance().scheduleEventWorker(it)
            }
        }
        scheduleEventBatchProcessing(0)
    }

    /**
     * If batch processing api is scheduled
     * if app is going BG then cancel the scheduled Batch processing
     * if app comes to FG then check if we need to process
     * batch process API immediately or needs to schedule
     */
    internal fun checkForBatchProcessing(isAppinFg: Boolean) {
        if (isAppinFg) {
            val appLastActiveTime = sharedPreferences.getLong(SMTPreferenceConstants.LAST_APP_ACTIVE_TIME_STAMP)
            val batchInterval = (sharedPreferences.getInt(SMTPreferenceConstants.BATCH_INTERVAL) * 1000L)
            // if batch interval elapsed then immediately call batch processing api

            if ((appLastActiveTime + batchInterval < System.currentTimeMillis())) {
                makeEventBatchProcessingApiCall()
            } else {
                // else schedule the next batch processing with difference time interval
                var nextBatchScheduleTime = appLastActiveTime + batchInterval - System.currentTimeMillis()
                if (nextBatchScheduleTime < 0) {
                    nextBatchScheduleTime = batchInterval
                }
                scheduleEventBatchProcessing(nextBatchScheduleTime)
            }

        } else {
            // if app going to BG then cancel the current scheduled task
            /**
             *  Uncomment below code to if you want to stop executor in background.
             *  However, we have handled this situation already
             */
            mHandler.removeCallbacksAndMessages(null)
        }
    }

    /**
     * Creates Batch Process runnable
     */
    private fun getBatchProcessRunner(): Runnable {
        return Runnable {
            makeEventBatchProcessingApiCall()
        }
    }

}