package com.netcore.android.workmgr

import android.content.Context
import androidx.work.Worker
import androidx.work.WorkerParameters
import com.netcore.android.SMTActivityLifecycleCallback
import com.netcore.android.db.SMTDataBaseService
import com.netcore.android.db.SMTEventTable
import com.netcore.android.event.SMTEventProcessor
import com.netcore.android.event.SMTEventSyncStatus
import com.netcore.android.event.SMTEventsBatchProcessor
import com.netcore.android.event.SMTWorkerType
import com.netcore.android.logger.SMTLogger
import com.netcore.android.network.SMTApiService
import com.netcore.android.network.SMTNetowrkUtil
import com.netcore.android.network.SMTNetworkManager
import com.netcore.android.network.models.SMTRequest
import com.netcore.android.network.models.SMTResponse
import com.netcore.android.notification.models.SMTEventPayload
import com.netcore.android.utility.SMTCommonUtility
import java.lang.ref.WeakReference

/**
 *  This worker is created to sync events in background.
 *
 *  @author Netcore
 *
 *  @Description Thiw worker will always run in background and only if EventSync worker is not running because
 *  to make sure that consistency between the batch of events processed.It will sync only Pending and FAILED events.
 *
 *  By default, It's a periodic worker which will run after every 15 minutes interval which can be changed from SMTWorkerScheduler class.
 *
 */

internal class BackgroundSyncWorker(context: Context, param: WorkerParameters) : Worker(context, param) {

    val TAG: String = BackgroundSyncWorker::class.java.simpleName

    lateinit var context: Context
    private lateinit var eventPayload: SMTEventPayload
    private lateinit var smtEventsBatchProcessor: SMTEventsBatchProcessor

    override fun doWork(): Result {
        try {
            context = applicationContext

            val eventWorkerRunningStatus = SMTWorkerScheduler.getInstance().isEventWorkerRunning(context)
            SMTLogger.v(TAG, "Event worker Status : $eventWorkerRunningStatus")
            val isAppInForeground = SMTActivityLifecycleCallback.getInstance().isAppInForeground()

            // SDK is inactive so cancel running existing workers
            if (!SMTCommonUtility.checkPanelAndSDKActiveStatus(context)) {
                SMTLogger.w(TAG, "SDK / Panel is inactive")
                SMTWorkerScheduler.getInstance().cancelBackgroundSyncWorker(context)
                return Result.failure()
            }

            if (!eventWorkerRunningStatus) {
                smtEventsBatchProcessor = SMTEventsBatchProcessor.getInstance(context)
                if (!isAppInForeground) {
                    initiateBatchExecution()
                } else {
                    // App is in Foreground
                    // Intentionally blank.
                }
            }
            return Result.success()
        } catch (e: Exception) {
            SMTLogger.v(TAG, e.localizedMessage)
        }
        return Result.success()
    }

    override fun onStopped() {
        super.onStopped()
        SMTLogger.v(TAG, "Background sync worker stopped")
    }

    private fun initiateBatchExecution() {
        processEvents()
    }

    private fun processEvents() {
        eventPayload = smtEventsBatchProcessor.prepareBatchRequest(WeakReference(applicationContext), SMTWorkerType.BackgroundSyncWorker)
        if (eventPayload.eventArray.length() <= 0) {
            SMTLogger.internal(TAG, "EventsArray size is 0")
            return
        }
        val smtRequest = smtEventsBatchProcessor.getBatchProcessRequest(eventPayload.eventArray)
        val smtResponse = SMTEventProcessor.getInstance().processEvents(smtRequest)
        smtResponse?.let {
            if (it.isSuccess) {
                onSuccess(eventPayload.idArray)
            } else onFailure(eventPayload.idArray)
            if (isMoreEventsPresents()) {
                processEvents()
            }
        } ?: onFailure(eventPayload.idArray)

    }

    private fun onSuccess(idArray: Array<Int>) {
        SMTDataBaseService.getInstance(WeakReference(context)).deleteEventsMultipleRows(idArray)
    }

    private fun isMoreEventsPresents() = smtEventsBatchProcessor.checkIfMoreEventsPresentForBatchRequest(WeakReference(context), SMTWorkerType.BackgroundSyncWorker)

    private fun onFailure(idArray: Array<Int>) {
        SMTDataBaseService.getInstance(WeakReference(context)).updateEventProcessingStatus(idArray, SMTEventTable.KEY_EVENT_SYNC_STATUS, SMTEventSyncStatus.EVENT_SYNC_STATUS_FAILED)
        SMTDataBaseService.getInstance(WeakReference(context)).updateFailedBatchPayload()
        SMTLogger.v(TAG, "Background Events sync failed.")
    }
}