package com.netcore.android.utility

import android.content.Context
import android.location.Location
import android.os.Build
import android.util.DisplayMetrics
import android.util.Log
import android.view.Surface
import android.view.WindowManager
import com.google.android.gms.common.GooglePlayServicesNotAvailableException
import com.google.android.gms.common.GooglePlayServicesRepairableException
import com.netcore.android.SMTConfigConstants
import com.netcore.android.listeners.SMTLocationCallback
import com.netcore.android.logger.SMTLogger
import com.netcore.android.network.SMTThreadPoolManager
import com.netcore.android.preference.SMTPreferenceConstants
import com.netcore.android.preference.SMTPreferenceHelper
import java.text.SimpleDateFormat
import java.util.*


/**
 * Copyright © 2019 Netcore. All rights reserved.
 *
 * Singleton class to provides device related Info
 *
 * @author Netcore
 * @version 1.0
 * @since 26-02-2019
 */
internal class SMTDeviceInfo(private val context: Context) /*: AdvertiseFetchListener*/ {

    private val TAG = SMTDeviceInfo::class.java.simpleName

    val osName = SMTConfigConstants.OS_NAME
    val osVersion = getosVersion()
    val deviceMake = getManufacturer().toLowerCase()
    val deviceModel = getModel().toLowerCase()
    val deviceWidth = getWidth(context).toString()
    val deviceHeight = getHeight(context).toString()
    val timeZone = SimpleDateFormat("ZZZZ", Locale.getDefault()).format(System.currentTimeMillis())
    val deviceLocale: String = Locale.getDefault().language.toLowerCase()
    //    var deviceUniqueId :String? = getUniqueId()
    var guid = ""
    var advertiserId: String? = null
    var orientation = ""
    var latitude = "0.0"
    var longitude = "0.0"

    /**
     * To get current location of the device.
     * */
    private lateinit var fusedLocationManager: SMTFusedLocationManager

    init {
        getLocation()
        generateUids()
        getUpdatedDeviceOrientation()
    }

    /**
     * Provides Unique ID for the deivce
     * First checks for ANDROID_ID,
     * If ANDROID_ID is not there then checks for ADID
     * If ADID is not there then generates GUID
     * @return String Device Unique ID
     */
    /*private fun getUniqueId(): String? {
//        deviceUniqueId = getAndroidId()
        guid = generateAndStoreGUID()
        if (SMTPreferenceHelper.getAppPreferenceInstance(context, null)
                        .getInt(SMTPreferenceConstants.SMT_MF_IS_USE_AD_ID) == 1) {
            getAdId()
        }

        return deviceUniqueId
    }*/

    private fun generateUids() {
        guid = generateAndStoreGUID()
        val preference = SMTPreferenceHelper.getAppPreferenceInstance(context, null)
        if (preference.getInt(SMTPreferenceConstants.SMT_MF_IS_USE_AD_ID) == 1) {
            getAdId()
        } else {
            advertiserId = ""
        }
    }

    /**
     * Provides ANDROID_ID
     * @return String android id
     */
    /*@SuppressLint("HardwareIds")
    private fun getAndroidId(): String? {
        return Settings.Secure.getString(context.contentResolver,
                Settings.Secure.ANDROID_ID)
    }*/

    /**
     * provides Random GUID
     */
    private fun generateAndStoreGUID(): String {
        return SMTCommonUtility.getStoredGUID(context)
    }

    /**
     * Provides ADID
     */
    private fun getAdId() {
        SMTThreadPoolManager.getIntance().execute {
            //Fetching ADID asynchronously
            // Device info is fetched in asynchronously
            var advertId: String? = null

            try {
                context.let {
                    val adIdClient = Class.forName("com.google.android.gms.ads.identifier.AdvertisingIdClient")

                    val getAdInfo = adIdClient.getMethod("getAdvertisingIdInfo", Context::class.java)

                    val adInfo = getAdInfo.invoke(null, context)

                    val getAdId = adInfo.javaClass.getMethod("getId")
                    advertId = getAdId.invoke(adInfo) as String
                }
            } catch (e: GooglePlayServicesNotAvailableException) {
                SMTLogger.e(TAG, "Error while fetching ADID error :-$e")
            } catch (e: GooglePlayServicesRepairableException) {
                SMTLogger.e(TAG, "Error while fetching ADID error :-$e")
            } catch (e: Exception) {
                SMTLogger.e(TAG, "Error while fetching ADID error :-$e")
            }
            advertiserId = advertId
        }
    }

    /**
     * Provides Device OS version
     * @return String OS version
     */
    private fun getosVersion(): String {
        return Build.VERSION.RELEASE
    }

    /**
     * Provides device manufaturer
     * @return String device manufaturer
     */
    private fun getManufacturer(): String {
        return Build.MANUFACTURER
    }

    /**
     * Provides device model
     * @return String Device Model
     */
    private fun getModel(): String {
        var model = Build.MODEL
        model = model.replace(getManufacturer(), "")
        return model
    }

    /**
     * Provides device display height
     * @param context - App context
     * @return Double - Display height
     */
    private fun getHeight(context: Context): Int {
        val wm = context.getSystemService(Context.WINDOW_SERVICE) as WindowManager
        val dm = DisplayMetrics()
        wm.defaultDisplay.getMetrics(dm)
        return dm.heightPixels + getNavigationBarHeight(context)
    }

    /**
     * Provides device display width
     * @param context - App context
     * @return Double - Display width
     */
    private fun getWidth(context: Context): Int {
        val wm = context.getSystemService(Context.WINDOW_SERVICE) as WindowManager
        val dm = DisplayMetrics()
        wm.defaultDisplay.getMetrics(dm)
        return dm.widthPixels
    }

    /**
     * Provides navigation bar size
     */
    private fun getNavigationBarHeight(context: Context): Int {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
            val wm = context.getSystemService(Context.WINDOW_SERVICE) as WindowManager
            val metrics = DisplayMetrics()
            wm.defaultDisplay.getMetrics(metrics)
            val usableHeight = metrics.heightPixels
            wm.defaultDisplay.getRealMetrics(metrics)
            val realHeight = metrics.heightPixels
            return if (realHeight > usableHeight)
                realHeight - usableHeight
            else
                0
        }
        return 0
    }


    /**
     * Round the number to nearest number
     * @param n - Number to be rounded off
     * @return Double - Rounded off number
     */
    private fun roundToNearestNumber(n: Double): Double {
        var result = n * 100
        result = Math.round(result).toDouble()
        result /= 100
        return result
    }

    /**
     * Provides device display density
     * @param context - App context
     * @return Int - Display density
     */
    /*private fun getDPI(context: Context): Int {
        val wm = context.getSystemService(Context.WINDOW_SERVICE) as WindowManager
        val dm = DisplayMetrics()
        wm.defaultDisplay.getMetrics(dm)
        return dm.densityDpi
    }*/

    internal fun getUpdatedDeviceOrientation(): String {
        orientation = getScreenOrientation()
        return orientation
    }

    /**
     * Method to fetch screen orientation like landscape or portrait
     */
    private fun getScreenOrientation(): String {
        val display = (context.getSystemService(Context.WINDOW_SERVICE) as WindowManager).defaultDisplay
        val orientation = display.rotation
        return if (orientation == Surface.ROTATION_90
                || orientation == Surface.ROTATION_270) {
            SMTConfigConstants.SCREEN_ORIENTATION_LANDSCAPE
        } else {
            SMTConfigConstants.SCREEN_ORIENTATION_PORTRAIT
        }
    }

    /**
     * To fetch location of user useing FusedLocationManager
     */
    private fun getLocation() {
        SMTThreadPoolManager.getIntance().execute {

            latitude = SMTPreferenceHelper.getAppPreferenceInstance(context, null).getString(SMTPreferenceConstants.SMT_LAST_KNOWN_LATITUDE)
            longitude = SMTPreferenceHelper.getAppPreferenceInstance(context, null).getString(SMTPreferenceConstants.SMT_LAST_KNOWN_LONGITUDE)

            // check for auto location fetch app permission
            if (SMTPreferenceHelper.getAppPreferenceInstance(context, null)
                            .getInt(SMTPreferenceConstants.SMT_MF_IS_AUTO_FETCH_LOCATION, 0) == 1) {
                // check if device api verion is greater than VERSION_CODES.M
                if (SMTCommonUtility.shouldCheckPermission()) {
                    if (SMTCommonUtility.isPermissionGranted(context, SMTConfigConstants.LOCATION_PERMISSION_MF_KEY)) {
                        fusedLocationManager = SMTFusedLocationManager(context, getLocationCallback())
                        fusedLocationManager.getLastLocation()
                    }
                } else {
                    fusedLocationManager = SMTFusedLocationManager(context, getLocationCallback())
                    fusedLocationManager.getLastLocation()
                }
            }
        }
    }

    /**
     * will receive users last known location
     */
    private fun getLocationCallback(): SMTLocationCallback {
        return object : SMTLocationCallback {
            override fun onLocationFetchSuccess(location: Location) {
                onLocationFetched(location)
            }

            override fun onLocationFetchFailed(e: Exception) {
                latitude = SMTPreferenceHelper.getAppPreferenceInstance(context, null).getString(SMTPreferenceConstants.SMT_LAST_KNOWN_LATITUDE)
                longitude = SMTPreferenceHelper.getAppPreferenceInstance(context, null).getString(SMTPreferenceConstants.SMT_LAST_KNOWN_LONGITUDE)
            }
        }
    }

    /**
     * Get Last Known location from FusedLocationManager
     * */
    internal fun onLocationFetched(location: Location) {
        SMTLogger.i(TAG, "Fetched Location LATITUDE [:] ${location.latitude}, LONGITUDE [:] ${location.longitude}")
        // set the current fetched location
        latitude = location.latitude.toString()
        longitude = location.longitude.toString()
        // Store the location for future use
        SMTPreferenceHelper.getAppPreferenceInstance(context, null).setString(SMTPreferenceConstants.SMT_LAST_KNOWN_LATITUDE, latitude)
        SMTPreferenceHelper.getAppPreferenceInstance(context, null).setString(SMTPreferenceConstants.SMT_LAST_KNOWN_LONGITUDE, longitude)
    }

    /**
     * Updates the location
     */
    internal fun updateLocation() {
        getLocation()
    }
}