package com.netcore.android.notification

import android.app.NotificationManager
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.os.Bundle
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import com.netcore.android.SMTConfigConstants
import com.netcore.android.event.SMTEventRecorder
import com.netcore.android.logger.SMTLogger
import com.netcore.android.notification.audio.SMTRichAudioPNGenerator
import com.netcore.android.notification.carousel.SMTCarouselLandscapePNGenerator
import com.netcore.android.notification.carousel.SMTCarouselPortraitPNGenerator
import com.netcore.android.notification.carousel.SMTCarouselSetup
import com.netcore.android.notification.gif.SMTRichGifPNGenerator
import com.netcore.android.notification.models.SMTNotificationData
import com.netcore.android.utility.SMTCommonUtility
import java.lang.ref.WeakReference


/**
 * Copyright © 2019 Netcore. All rights reserved.
 *
 * Handles the notification action receiver
 *
 * @author Netcore
 * @version 1.0
 * @since 26-02-2019
 */
internal class SMTPNActionReceiver : BroadcastReceiver() {

    private val TAG = SMTPNActionReceiver::class.java.simpleName

    override fun onReceive(context: Context?, intent: Intent?) {
        try {

            // Handles Notification dismiss event
            intent?.let { receivedIntent ->
                receivedIntent.action?.let { action ->
                    if (action == SMTNotificationConstants.NOTIF_ACTION_DISMISS) {
                        receivedIntent.extras?.let { bundle ->
                            clearAndUpdateDbOnNotificationDismiss(context, bundle)
                        }
                        return
                    }
                }
            }
            val extras = intent?.extras ?: return

            // Handles actions related to types of notification
            if (extras.containsKey(SMTNotificationConstants.NOTIF_TYPE_KEY)
                    && intent.action != SMTNotificationConstants.NOTIF_ACTIONS) {
                when (extras.get(SMTNotificationConstants.NOTIF_TYPE_KEY)) {
                    // Handle Carousel specific notification action
                    SMTNotificationType.CAROUSEL_PORTRAIT.type,
                    SMTNotificationType.CAROUSEL_LANDSCAPE.type -> {
                        context?.let {
                            // handles carousel notification action
                            handleCarouselNotificationAction(it, extras)
                        }
                        return
                    }
                    // Handle Audio specific notification action
                    SMTNotificationType.AUDIO.type -> {
                        context?.let {
                            broadcastPNClickedOrDismissed(it, extras, SMTConfigConstants.BROADCAST_EVENT_PN_CLICKED)
                            SMTRichAudioPNGenerator.getInstance().handleAudioClick(it, extras)
                            clearNotificationFromTray(context, extras)
                        }
                        return
                    }
                    // Handle Audio specific notification action
                    SMTNotificationType.BIG_IMAGE.type -> {
                        context?.let {
                            broadcastPNClickedOrDismissed(it, extras, SMTConfigConstants.BROADCAST_EVENT_PN_CLICKED)
                            SMTRichImagePNGenerator().handleImageClick(it, extras)
                            clearNotificationFromTray(context, extras)
                        }
                        return
                    }
                    // Handle Simple notification action
                    SMTNotificationType.SIMPLE.type -> {
                        context?.let {
                            broadcastPNClickedOrDismissed(it, extras, SMTConfigConstants.BROADCAST_EVENT_PN_CLICKED)
                            SMTSimplePNGenerator().handleContentClick(it, extras)
                            clearNotificationFromTray(it, extras)
                        }
                        return
                    }

                    // Handle GIF notification action
                    SMTNotificationType.GIF.type -> {
                        context?.let {
                            broadcastPNClickedOrDismissed(it, extras, SMTConfigConstants.BROADCAST_EVENT_PN_CLICKED)
                            SMTRichGifPNGenerator.getInstance(WeakReference(it)).handleContentClick(it, extras)
                        }
                        return
                    }
                }
            }
            // This action does not belong to Smartech, Dont proceed further
            if (!extras.containsKey(SMTNotificationConstants.NOTIF_ACTION)) {
                return
            }

            // handles action button click
            handleActionButtonClick(context, extras)

            SMTLogger.d(TAG, "handled notification: $extras")
        } catch (t: Throwable) {
            t.printStackTrace()
            SMTLogger.e(TAG, ": error handling notification $t")
        }
    }

    /**
     * Handles action button click
     * @param context - App context
     * @param extras - Extras passed through the pending intent
     */
    private fun handleActionButtonClick(context: Context?, extras: Bundle) {
        var deeplinkPath = ""
        var notifdata: SMTNotificationData? = null

        if (extras.containsKey(SMTNotificationConstants.NOTIF_ACTION_DEEPLINK_KEY)) {
            deeplinkPath = extras.get(SMTNotificationConstants.NOTIF_ACTION_DEEPLINK_KEY) as String
        }

        if (extras.containsKey(SMTNotificationConstants.NOTIFICATION_PARCEL)) {
            val parcel = extras.get(SMTNotificationConstants.NOTIFICATION_PARCEL)
            parcel?.let {

                //Update DB record whether it has been clicked
                notifdata = it as SMTNotificationData

                context?.let { _context ->
                    notifdata?.let { notif ->
                        SMTEventRecorder.getInstance(WeakReference(_context)).recordNotificationClick(notif.mTrid, notif.mPNMeta, deeplinkPath, notif.mSource, notif.mSmtAttributePayload
                                ?: HashMap())
                    }
                }

                clearNotificationFromTray(context, (notifdata?.notificationId ?: 0))

                extras.putString(SMTNotificationConstants.NOTIF_TYPE_KEY, it.mNotificationType)
                clearCache(context, extras, SMTConfigConstants.BROADCAST_EVENT_PN_CLICKED)
            }
        }

        context?.let {
            SMTCommonUtility.handleNotificationClick(it, deeplinkPath, notifdata?.mCustomPayload)
        }
    }

    /**
     * Clears notification from system tray
     * @param context - App context
     * @param notificationId - Id of the notification
     */
    private fun clearNotificationFromTray(context: Context?, notificationId: Int) {
        // Clear the notification from system tray

        val mNotifyManager = context?.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
        mNotifyManager.cancel(notificationId)
    }

    /**
     * Clears notification from system tray
     * @param context - App context
     * @param extras - Extras passed through the pending intent
     */
    private fun clearNotificationFromTray(context: Context?, extras: Bundle) {
        // Clear the notification from system tray

        var notifParcel = extras.getParcelable<SMTNotificationData>(SMTNotificationConstants.NOTIFICATION_PARCEL)

        var notifId: Int
        notifId = if (notifParcel != null) {
            notifParcel.notificationId
        } else {
            val carouselparcel = extras.getParcelable<SMTCarouselSetup>(SMTNotificationConstants.CAROUSEL_SET_UP_KEY)
            carouselparcel?.carouselNotificationId ?: -1
        }
        if (notifId != -1) {
            val mNotifyManager = context?.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
            mNotifyManager.cancel(notifId)
        }
    }


    /**
     * Handles clear action on notification dismiss
     * @param context - App context
     * @param extras - Extras passed through the pending intent
     */
    private fun clearAndUpdateDbOnNotificationDismiss(context: Context?, extras: Bundle) {

        if (extras.containsKey(SMTNotificationConstants.NOTIFICATION_PARCEL)) {
            val notifData = extras.get(SMTNotificationConstants.NOTIFICATION_PARCEL)

            val trid = if (notifData is SMTNotificationData) {
                notifData.mTrid
            } else {
                (notifData as SMTCarouselSetup).trid
            }
            val source = if (notifData is SMTNotificationData) {
                notifData.mSource
            } else {
                (notifData as SMTCarouselSetup).source
            }

            val pnMeta = if (notifData is SMTNotificationData) {
                notifData.mPNMeta
            } else {
                (notifData as SMTCarouselSetup).pnMeta
            }
            //Update DB record whether it has been dismissed
            context?.let {
                SMTEventRecorder.getInstance(WeakReference(it)).recordNotificationDismiss(trid!!, pnMeta, source)
            }
        }

        clearCache(context, extras, SMTConfigConstants.BROADCAST_EVENT_PN_DISMISSED)
    }

    private fun broadcastPNClickedOrDismissed(context: Context, extras: Bundle, event: String) {

        val intent = Intent(event)
        intent.putExtras(extras)

        // You can also include some extra data.
        LocalBroadcastManager.getInstance(context).sendBroadcast(intent)
    }

    private fun clearCache(context: Context?, extras: Bundle, event: String) {
        context?.let {
            broadcastPNClickedOrDismissed(it, extras, event)
            when (extras[SMTNotificationConstants.NOTIF_TYPE_KEY]) {
                SMTNotificationType.AUDIO.type -> SMTRichAudioPNGenerator.getInstance().handleNotificationDismiss(it, extras)
                SMTNotificationType.BIG_IMAGE.type -> SMTRichImagePNGenerator().handleNotificationDismiss(it, extras)
                SMTNotificationType.SIMPLE.type -> SMTSimplePNGenerator().handleNotificationDismiss(it, extras)
                SMTNotificationType.CAROUSEL_LANDSCAPE.type -> SMTCarouselLandscapePNGenerator(WeakReference(it)).handleNotificationDismiss(it, extras)
                SMTNotificationType.CAROUSEL_PORTRAIT.type -> SMTCarouselPortraitPNGenerator(WeakReference(it)).handleNotificationDismiss(it, extras)
                SMTNotificationType.GIF.type -> SMTRichGifPNGenerator.getInstance(WeakReference(context)).handleNotificationDismiss(it, extras)
            }
        }
    }

    /**
     * Handles carousel click events
     * @param context - App context
     * @param extras - Extras passed through the pending intent
     */
    private fun handleCarouselNotificationAction(context: Context, extras: Bundle) {
        val carouselEvent = extras.getInt(SMTNotificationConstants.CAROUSEL_ITEM_CLICKED_KEY)
        val carouselSetUp = extras.getParcelable(SMTNotificationConstants.CAROUSEL_SET_UP_KEY) as SMTCarouselSetup
        val type = extras.getString(SMTNotificationConstants.NOTIF_TYPE_KEY)

        if (carouselEvent > 0)
            if (type == SMTNotificationType.CAROUSEL_PORTRAIT.type) {
                SMTCarouselPortraitPNGenerator(WeakReference(context)).handleClickEvent(carouselEvent, carouselSetUp)
            } else {
                SMTCarouselLandscapePNGenerator(WeakReference(context)).handleClickEvent(carouselEvent, carouselSetUp)
            }
    }
}
