package com.netcore.android

import android.content.Context
import android.content.Intent
import android.content.SharedPreferences
import android.os.Bundle
import android.util.Log
import androidx.appcompat.app.AppCompatActivity
import com.google.android.gms.tasks.OnCompleteListener
import com.google.firebase.iid.FirebaseInstanceId
import com.netcore.android.event.SMTEventCommonDataDump
import com.netcore.android.event.SMTEventId
import com.netcore.android.event.SMTEventRecorder
import com.netcore.android.event.SMTEventType
import com.netcore.android.inbox.views.activity.InBoxActivity
import com.netcore.android.inbox.views.fragment.SMTInboxFragment
import com.netcore.android.logger.SMTLogger
import com.netcore.android.network.SMTEnumHttpMethodType
import com.netcore.android.network.SMTNetworkManager
import com.netcore.android.network.SMTRequestQueue
import com.netcore.android.network.SMTResponseListener
import com.netcore.android.network.models.SMTRequest
import com.netcore.android.network.models.SMTSdkInitializeResponse
import com.netcore.android.notification.SMTNotificationUtility
import com.netcore.android.preference.SMTPreferenceConstants
import com.netcore.android.preference.SMTPreferenceHelper
import com.netcore.android.utility.SMTCommonUtility
import com.netcore.android.utility.SMTInfo
import com.netcore.android.workmgr.SMTWorkerScheduler
import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject
import java.lang.ref.WeakReference
import java.util.*
import java.util.concurrent.TimeUnit


/**
 * @Description
 *
 * Helper class for Smartech
 * @param context app context
 * @param smtInfo SMTInfo reference
 * @param mPreferences PrefernceHelper reference
 * @param responseListener Api Response listener
 *
 * @author Netcore
 * @version 1.0
 * @since 14-03-2019
 */
internal class SmartechHelper(val context: WeakReference<Context>, private val smtInfo: SMTInfo,
                              private val mPreferences: SMTPreferenceHelper,
                              private val responseListener: SMTResponseListener) {

    private val TAG = SmartechHelper::class.java.simpleName

    /**
     * Initialise API call on app launch
     */
    internal fun makeInitializeApiCall(apiId: SMTRequest.SMTApiTypeID) {
        val requestBuilder = SMTRequest.Builder()
                .setHttpMethod(SMTEnumHttpMethodType.GET)
                .setBaseUrl(getInitApiUrl())
                .setEndPoint(getInitialiseApiEndPoint())
                .setApiId(apiId)
                .setResponseListener(responseListener)
        SMTNetworkManager.getInstance(SMTRequestQueue.getInstance()).processRequest(requestBuilder.build())
    }

    private fun getInitApiUrl(): String {
        return BuildConfig.BASE_URL
    }

    /**
     * create the Initialise api
     */
    private fun getInitialiseApiEndPoint(): String {
        val appId = mPreferences.getString(SMTPreferenceConstants.SMT_MF_APP_ID)
        var urlParams = context.get()?.let {
            SMTEventCommonDataDump(it).getURLParameters(false)
        } ?: ""
        return "appinit/$appId.json?$urlParams"
    }

    /**
     * Update smartech settings to prefernce
     * @param settings - smaretech setting POJO
     */
    internal fun updateSmartechConfigs(settings: SMTSdkInitializeResponse.SmartTechSettings) {
        mPreferences.setBoolean(SMTPreferenceConstants.IS_SMRATECH_SETTINGS_STORED, true)
        mPreferences.setInt(SMTPreferenceConstants.BATCH_INTERVAL, settings.batchInterval)
        mPreferences.setInt(SMTPreferenceConstants.BATCH_SIZE, settings.batchSize)
        mPreferences.setBoolean(SMTPreferenceConstants.IS_PUSH_AMP_ENABLED, settings.paEnabled)
        mPreferences.setInt(SMTPreferenceConstants.PUSH_AMP_INTERVAL, settings.paInterval)
        mPreferences.setBoolean(SMTPreferenceConstants.IS_FETCH_LOCATION_ENABLED, settings.fetchLocation)
        mPreferences.setBoolean(SMTPreferenceConstants.IS_PANEL_ACTIVE, settings.panelActive)
        mPreferences.setBoolean(SMTPreferenceConstants.IS_SDK_ACTIVE, settings.sdkActive)
        mPreferences.setInt(SMTPreferenceConstants.SESSION_INTERVAL, settings.sessionInterval)
        mPreferences.setString(SMTPreferenceConstants.SMT_BASE_URL, settings.baseUrl ?: "")
        mPreferences.setString(SMTPreferenceConstants.SMT_BASE_URL_TRACKAPPACT, settings.smartechURL?.trackAppActUrl
                ?: "")
        mPreferences.setString(SMTPreferenceConstants.SMT_BASE_URL_INAPP, settings.smartechURL?.inAppUrl
                ?: "")
        mPreferences.setString(SMTPreferenceConstants.SMT_BASE_URL_INBOX, settings.smartechURL?.inboxUrl
                ?: "")
        mPreferences.setString(SMTPreferenceConstants.SMT_BASE_URL_PUSHAMP, settings.smartechURL?.pushAmpUrl
                ?: "")
        mPreferences.setBoolean(SMTPreferenceConstants.IS_LOG_ENABLED, settings.debuglevel?.logEnabled
                ?: false)
        mPreferences.setInt(SMTPreferenceConstants.LOG_LEVEL, settings.debuglevel?.logLevel
                ?: 0)
        mPreferences.setString(SMTPreferenceConstants.GUIDS, settings.debuglevel?.guids?.toString()
                ?: "")
        SMTLogger.internal(TAG, "Smartech settings: ${settings.toString()}")
    }

    /**
     * Records App launch type like first or regular app launch
     * and also records App launch for the first time
     */
    private fun recordAppLaunchEvents(isFirstAppLaunch: Boolean) {
        // App regular launch event needs to be stored
        if (!isFirstAppLaunch) getRecorder().recordEvent(SMTEventId.EVENT_APP_LAUNCHED,
                SMTEventId.getEventName(SMTEventId.EVENT_APP_LAUNCHED), null, SMTEventType.EVENT_TYPE_SYSTEM)
    }

    /**
     * Provides Event Recorder reference
     */
    private fun getRecorder(): SMTEventRecorder {
        return SMTEventRecorder.getInstance(context)
    }

    /**
     * Calls push amp api
     * @param
     */
    internal fun makePushAmpApiCall() {
        context.get()?.let {
            if (SMTWorkerScheduler.getInstance().arePushampConditionsSatisfied(it)) {
                SMTWorkerScheduler.getInstance().schedulePushAmp(it)
            }
        }
    }

    /**
     *  This will be used to fetch existing token from the device
     *  and it will save the token to SharedPreferences.
     */
    internal fun fetchExistingToken() {

        if (SMTCommonUtility.isFCMAvailable()) {
            context.get()?.let {
                val preference = SMTPreferenceHelper.getAppPreferenceInstance(it, null)
                val currentToken = preference.getString(SMTPreferenceConstants.PUSH_TOKEN_CURRENT, "")

                if (currentToken.isNotEmpty()) {
                    SMTLogger.internal(TAG, "Token is present in preferences")
                    return
                }
                FirebaseInstanceId.getInstance().instanceId
                        .addOnCompleteListener(OnCompleteListener { task ->
                            if (!task.isSuccessful) {
                                SMTLogger.w(TAG, "getInstanceId failed")
                                return@OnCompleteListener
                            }

                            /* Get new Instance ID token */
                            val token = task.result?.token
                            // Saving token to preferences

                            token?.let { fcmToken ->
                                setDevicePushToken(fcmToken)
                                SMTLogger.i(TAG, "Existing token fetched successfully.")
                            }
                        })
            }
        }
    }

    /**
     *  This function will provide developer to manually
     *  provide token to SDK and it will be saved inside SDK's
     *  SharedPreferences.
     *  @param token device token
     */
    internal fun setDevicePushToken(token: String?) {
        context.get()?.let {
            SMTNotificationUtility.getInstance().setPushToken(it, token)
        }
    }

    /**
     * To stop background push amp job
     */
    internal fun stopAndRefreshPushAmpBgTask() {
        context.get()?.let {
            SMTWorkerScheduler.getInstance().cancelPushAmp(it)
        }
    }

    /**
     * checks if App or SDK is updated
     * if updated then record the app update event
     */
    /*private fun checkIfAppOrSdkIsUpdated() {
        if (mPreferences.getString(SMTPreferenceConstants.SMT_APP_VERSION) != smtInfo.mAppInfo?.appVersion
                || mPreferences.getString(SMTPreferenceConstants.SMT_SDK_VERSION) != smtInfo.mAppInfo?.smtSdkVersion) {
            recordAndStoreAppAndSdkVersion()
        }
    }*/

    /**
     *
     * Checks if any of the device details changed then update them in Preference
     */
    private fun checkAndUpdateDeviceDetails() {
        // If the app version is updated then record it
        context.get()?.let {
            if (SMTCommonUtility.checkIfDeviceDetailChanged(smtInfo, it)) {
                mPreferences.setString(SMTPreferenceConstants.SMT_OS_VERSION, smtInfo.mDeviceInfo?.osVersion
                        ?: "")
                mPreferences.setString(SMTPreferenceConstants.SMT_CARRIER, smtInfo.mNetworkInfo?.carrier
                        ?: "")
                mPreferences.setString(SMTPreferenceConstants.SMT_DEVICE_LOCALE, smtInfo.mDeviceInfo?.deviceLocale
                        ?: "")
                mPreferences.setString(SMTPreferenceConstants.SMT_ADID, smtInfo.mDeviceInfo?.advertiserId
                        ?: "")
                mPreferences.setString(SMTPreferenceConstants.SMT_TIMEZONE, smtInfo.mDeviceInfo?.timeZone
                        ?: "")
                SMTLogger.i(TAG, "recording device detail updated event")
                // Device details updated
                getRecorder().recordEvent(SMTEventId.EVENT_DEVICE_DETAILS_UPDATED,
                        SMTEventId.getEventName(SMTEventId.EVENT_DEVICE_DETAILS_UPDATED), null, SMTEventType.EVENT_TYPE_SYSTEM)
            }
        }
    }

    /**
     * Checks If current session is expired or not
     * by comparing the inactivity time with session interval
     * @return Boolean - True if session is expired else false
     */
    internal fun checkIfCurrentSessionExpired(): Boolean {
        val lastAppActiveTimeStamp = mPreferences.getLong(SMTPreferenceConstants.LAST_APP_ACTIVE_TIME_STAMP)
        SMTLogger.internal(TAG, "Session Interval ${mPreferences.getInt(SMTPreferenceConstants.SESSION_INTERVAL)}")
        val sessionInterval = TimeUnit.MINUTES.toMillis(mPreferences.getInt(SMTPreferenceConstants.SESSION_INTERVAL).toLong())

        // if there is no current session then create new session
        // else check for session lapse
        // use case - for the first app launch there wont be any session
        SMTLogger.internal(TAG, "Current session id: ${mPreferences.getLong(SMTPreferenceConstants.CURRENT_SESSION_ID)}")
        SMTLogger.internal(TAG, "Session Details: lastAppActiveTimeStamp: $lastAppActiveTimeStamp || currenttimemillis: ${System.currentTimeMillis()} || sessionInterval: $sessionInterval ")
        SMTLogger.internal(TAG, "Session calculation: ${(lastAppActiveTimeStamp - (System.currentTimeMillis() - sessionInterval) < 0)}")
        if (mPreferences.getLong(SMTPreferenceConstants.CURRENT_SESSION_ID) == 0L || (lastAppActiveTimeStamp - (System.currentTimeMillis() - sessionInterval) < 0)) {
            return true

        }
        return false
    }

    /**
     * It will drop attribute if userIdentity is not matching with previously stored Users identity
     * Call this method whenever app identity is changed
     */
    internal fun deletePNAttribution(userIdentity: String) {

        val appIdentity = mPreferences.getString(SMTPreferenceConstants.SMT_USER_IDENTITY, "").toLowerCase(Locale.getDefault())

        if (userIdentity != appIdentity) {
            mPreferences.setString(SMTPreferenceConstants.SMT_ATTRIBUTION_PARAMS, "")
        }
    }

    /**
     * Record the SDK init events
     * @param initEventTrackList - List of events to be tracked
     */
    internal fun recordInitAppEvents(initEventTrackList: ArrayList<Int>) {

        var isAppReinstalled = false
        var isAppInstallUpdate = false // used to check client is using AppInstallUpdate feature from smartech

        // check if app is reinstalled
        if (initEventTrackList.contains(SMTEventId.EVENT_APP_REINSTALLED)) {
            isAppReinstalled = checkAndRecordAppReinstall()
        }

        if (initEventTrackList.contains(SMTEventId.EVENT_APP_INSTALL_UPDATE_NETCORE) && !isAppReinstalled) {
            isAppInstallUpdate = true
            trackAppInstallUpdateBySmartech()
        }

        // if App is reinstalled then don't track App Install event
        if (initEventTrackList.contains(SMTEventId.EVENT_APP_INSTALLED) && !isAppReinstalled && !isAppInstallUpdate) {
            getRecorder().recordEvent(SMTEventId.EVENT_APP_INSTALLED,
                    SMTEventId.getEventName(SMTEventId.EVENT_APP_INSTALLED), null, SMTEventType.EVENT_TYPE_SYSTEM)
        }

        // Track app updated
        if (initEventTrackList.contains(SMTEventId.EVENT_APP_UPDATED) && !isAppInstallUpdate) {
            recordAndStoreAppAndSdkVersion()
        }

        // Track regular app launch
        if (initEventTrackList.contains(SMTEventId.EVENT_APP_LAUNCHED)) {
            recordAppLaunchEvents(false)
        }

        // Track device details updated
        if (initEventTrackList.contains(SMTEventId.EVENT_DEVICE_DETAILS_UPDATED)) {
            checkAndUpdateDeviceDetails()
        }

        // Track app location permission
        if (initEventTrackList.contains(SMTEventId.EVENT_LOCATION_FETCH_ENABLED)) {
            checkAndRecordLocationPermissionStatus()
        }

        // Track app notification permission
        if (initEventTrackList.contains(SMTEventId.EVENT_USER_ENABLED_PN)) {
            checkAndRecordNotificationPermissionStatus()
        }
    }

    /**
     * Records App Update event for App/SDK version updated
     * and also update the preference with current values
     */
    private fun recordAndStoreAppAndSdkVersion() {
        // Record App Update Event
        getRecorder().recordEvent(SMTEventId.EVENT_APP_UPDATED,
                SMTEventId.getEventName(SMTEventId.EVENT_APP_UPDATED), null, SMTEventType.EVENT_TYPE_SYSTEM)

        // Store updated App and SDK version into preference
        mPreferences.setString(SMTPreferenceConstants.SMT_APP_VERSION, smtInfo.mAppInfo?.appVersion
                ?: "")
        mPreferences.setString(SMTPreferenceConstants.SMT_SDK_VERSION, smtInfo.mAppInfo?.smtSdkVersion
                ?: "")
    }

    private fun trackAppInstallUpdateBySmartech() {

        // Tracking app install event
        val isAppInstalled = mPreferences.getBoolean(SMTPreferenceConstants.SMT_IS_APP_INSTALL_UPDATE_BY_SMARTECH, false)
        SMTLogger.internal(TAG, "Tracking app install.$isAppInstalled")

        if (!isAppInstalled) {
            getRecorder().recordEvent(SMTEventId.EVENT_APP_INSTALLED,
                    SMTEventId.getEventName(SMTEventId.EVENT_APP_INSTALLED), null, SMTEventType.EVENT_TYPE_SYSTEM)
            mPreferences.setBoolean(SMTPreferenceConstants.SMT_IS_APP_INSTALL_UPDATE_BY_SMARTECH, true)
        }

        // Tracking app update events
        try {
            val versionCode = smtInfo.mAppInfo?.appBuild?.toInt()
            val oldversionCode = mPreferences.getInt("version_code", 0)
            SMTLogger.internal(TAG, "App version code $versionCode  &&  $oldversionCode")
            if (oldversionCode != 0) {
                if ((versionCode as Int) > oldversionCode) {
                    recordAndStoreAppAndSdkVersion()
                }
            }
            mPreferences.setInt("version_code", versionCode as Int)
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    /**
     * checks if app is reinstalled
     * If it is then record the event
     */
    private fun checkAndRecordAppReinstall(): Boolean {

        if (mPreferences.getBoolean(SMTPreferenceConstants.IS_SMT_GUID_STORED_PREVIOUSLY, false)) {
            getRecorder().recordEvent(SMTEventId.EVENT_APP_REINSTALLED,
                    SMTEventId.getEventName(SMTEventId.EVENT_APP_REINSTALLED), null, SMTEventType.EVENT_TYPE_SYSTEM)
            mPreferences.setBoolean(SMTPreferenceConstants.SMT_IS_APP_INSTALL_UPDATE_BY_SMARTECH, true)
            return true
        }
        return false
    }

    /**
     * Records Location and Notification permission status if it is updated
     */
    internal fun recordLocationAndNotificationPermissionStatus() {

        // update notification and location
        checkAndRecordLocationPermissionStatus()
        checkAndRecordNotificationPermissionStatus()
    }

    /**
     * check and record if Location is enabled or disabled
     */
    private fun checkAndRecordLocationPermissionStatus() {
        context.get()?.let {
            val isLocationRecorded = mPreferences.getBoolean(SMTPreferenceConstants.SMT_IS_LOCATION_PERMISSION_AVAILABLE)
            val currentPermission = SMTCommonUtility.isPermissionGranted(it, SMTConfigConstants.LOCATION_PERMISSION_MF_KEY)

            // If permission stored then check if current status is different or not
            // if it is different then record it
            val storedPermissionStatus = mPreferences.getBoolean(SMTPreferenceConstants.SMT_LOCATION_PERMISSION)
            if (currentPermission != storedPermissionStatus || !isLocationRecorded) {
                recordLocationPermission(currentPermission)
                smtInfo.mDeviceInfo?.updateLocation()
            }

            mPreferences.setBoolean(SMTPreferenceConstants.SMT_IS_LOCATION_PERMISSION_AVAILABLE, true)
            mPreferences.setBoolean(SMTPreferenceConstants.SMT_LOCATION_PERMISSION, currentPermission)
        }
    }

    private fun recordLocationPermission(currentPermission: Boolean) {
        val eventId: Int
        val eventName: String
        if (currentPermission) {
            eventId = SMTEventId.EVENT_LOCATION_FETCH_ENABLED
            eventName = SMTEventId.getEventName(SMTEventId.EVENT_LOCATION_FETCH_ENABLED)
        } else {
            eventId = SMTEventId.EVENT_LOCATION_FETCH_DISABLED
            eventName = SMTEventId.getEventName(SMTEventId.EVENT_LOCATION_FETCH_DISABLED)
        }
        getRecorder().recordEvent(eventId,
                eventName, null, SMTEventType.EVENT_TYPE_SYSTEM)
    }

    /**
     * check and record if Location is enabled or disabled
     */
    private fun checkAndRecordNotificationPermissionStatus() {
        context.get()?.let {
            SMTCommonUtility.checkAndRecordNotificationPermissionStatus(it)
        }
    }

    /**
     * storing default config values into preference
     */
    internal fun storeDefaultSmartechSettings() {
        mPreferences.setBoolean(SMTPreferenceConstants.OPT_IN_OUT_IN_APP_MESSAGES, true)
        mPreferences.setBoolean(SMTPreferenceConstants.OPT_IN_OUT_PUSH_NOTIFICATION, true)
        mPreferences.setBoolean(SMTPreferenceConstants.OPT_IN_OUT_TRACKING, true)
        context.get()?.let {
            val currentLocationPermission = SMTCommonUtility.isPermissionGranted(it, SMTConfigConstants.LOCATION_PERMISSION_MF_KEY)
            mPreferences.setBoolean(SMTPreferenceConstants.SMT_LOCATION_PERMISSION, currentLocationPermission)
            val currentNotificationPermission = SMTCommonUtility.areNotificationsEnabled(it)
            mPreferences.setBoolean(SMTPreferenceConstants.SMT_NOTIFICATION_PERMISSION, currentNotificationPermission)
        }
    }

    internal fun handleInboxDisplay(containerId: Int, activity: AppCompatActivity) {
        if (containerId <= 0) {
            val intent = Intent(activity, InBoxActivity::class.java)
            activity.startActivity(intent)
        } else {
            addInBoxFragment(activity, containerId)
        }
    }

    private fun addInBoxFragment(activity: AppCompatActivity, containerId: Int) {
        val manager = activity.supportFragmentManager
        val inboxFragment = SMTInboxFragment()

        val bundle = Bundle()
        bundle.putBoolean(SMTConfigConstants.BUNDLE_KEY_INBOX_DISABLE_BACK, true)
        inboxFragment.arguments = bundle
        manager?.beginTransaction()?.add(containerId, inboxFragment, "InBox")?.commitAllowingStateLoss()
    }
}