package com.ncorti.ktfmt.gradle

import org.gradle.api.provider.Property

/** Gradle Extension to help you configure ktfmt-gradle */
@Suppress("UnnecessaryAbstractClass")
abstract class KtfmtExtension {
    init {
        maxWidth.convention(DEFAULT_MAX_WIDTH)
        blockIndent.convention(DEFAULT_BLOCK_INDENT)
        continuationIndent.convention(DEFAULT_CONTINUATION_INDENT)
        removeUnusedImports.convention(DEFAULT_REMOVE_UNUSED_IMPORTS)
        debuggingPrintOpsAfterFormatting.convention(DEFAULT_DEBUGGING_PRINT_OPTS)
        manageTrailingCommas.convention(DEFAULT_MANAGE_TRAILING_COMMAS)
        srcSetPathExclusionPattern.convention(DEFAULT_SRC_SET_PATH_EXCLUSION_PATTERN)
    }

    /** ktfmt breaks lines longer than maxWidth. Default 100. */
    abstract val maxWidth: Property<Int>

    /**
     * blockIndent is the size of the indent used when a new block is opened, in spaces.
     *
     * For example,
     * ```
     * fun f() {
     *   //
     * }
     * ```
     */
    abstract val blockIndent: Property<Int>

    /**
     * continuationIndent is the size of the indent used when a line is broken because it's too
     * long, in spaces.
     *
     * For example,
     * ```
     * val foo = bar(
     *     1)
     * ```
     */
    abstract val continuationIndent: Property<Int>

    /**
     * Automatically remove and insert trialing commas.
     *
     * Lists that cannot fit on one line will have trailing commas inserted. Lists that span
     * multiple lines will have them removed. Manually inserted trailing commas cannot be used as a
     * hint to force breaking lists to multiple lines.
     */
    abstract val manageTrailingCommas: Property<Boolean>

    /** Whether ktfmt should remove imports that are not used. */
    abstract val removeUnusedImports: Property<Boolean>

    /**
     * Regex to define what sourceSets paths should not be formatted
     *
     * For example the sourceSet "main" in the "build" folder would be excluded with the regex
     * "^(.*[\\/])?build([\\/].*)?$". This does not affect files inside a sourceSet. To exclude a
     * file inside a sourceSet use the include & exclude options on the ktfmt task.
     */
    abstract val srcSetPathExclusionPattern: Property<Regex>

    /**
     * Print the Ops generated by KotlinInputAstVisitor to help reason about formatting (i.e.,
     * newline) decisions
     */
    abstract val debuggingPrintOpsAfterFormatting: Property<Boolean>

    /** Sets the Google style (equivalent to set blockIndent to 2 and continuationIndent to 2). */
    @Suppress("MagicNumber")
    fun googleStyle() {
        blockIndent.set(2)
        continuationIndent.set(2)
        manageTrailingCommas.set(true)
    }

    /**
     * Sets the KotlinLang style. A format that attempts to reflect
     * https://kotlinlang.org/docs/coding-conventions.html.
     */
    @Suppress("MagicNumber")
    fun kotlinLangStyle() {
        blockIndent.set(4)
        continuationIndent.set(4)
        manageTrailingCommas.set(true)
    }

    internal fun toFormattingOptions(): FormattingOptionsBean =
        FormattingOptionsBean(
            maxWidth = maxWidth.get(),
            blockIndent = blockIndent.get(),
            continuationIndent = continuationIndent.get(),
            removeUnusedImports = removeUnusedImports.get(),
            manageTrailingCommas = manageTrailingCommas.get(),
            debuggingPrintOpsAfterFormatting = debuggingPrintOpsAfterFormatting.get(),
        )

    internal companion object {
        internal const val DEFAULT_MAX_WIDTH: Int = 100
        internal const val DEFAULT_BLOCK_INDENT: Int = 2
        internal const val DEFAULT_CONTINUATION_INDENT: Int = 4
        internal const val DEFAULT_REMOVE_UNUSED_IMPORTS: Boolean = true
        internal const val DEFAULT_DEBUGGING_PRINT_OPTS: Boolean = false
        internal const val DEFAULT_MANAGE_TRAILING_COMMAS: Boolean = false
        internal val DEFAULT_SRC_SET_PATH_EXCLUSION_PATTERN =
            Regex("^(.*[\\\\/])?build([\\\\/].*)?\$")
    }
}
