package com.ncorti.ktfmt.gradle

import com.ncorti.ktfmt.gradle.FormattingOptionsBean.Style.DROPBOX
import com.ncorti.ktfmt.gradle.FormattingOptionsBean.Style.FACEBOOK
import com.ncorti.ktfmt.gradle.FormattingOptionsBean.Style.GOOGLE
import javax.inject.Inject
import org.gradle.api.Project
import org.gradle.api.provider.Property

/** Gradle Extension to help you configure ktfmt-gradle */
@Suppress("UnnecessaryAbstractClass")
abstract class KtfmtExtension @Inject constructor(project: Project) {

    private val objects = project.objects

    internal var ktfmtStyle = FACEBOOK

    /** ktfmt breaks lines longer than maxWidth. Default 100. */
    val maxWidth: Property<Int> = objects.property(Int::class.java).convention(DEFAULT_MAX_WIDTH)

    /**
     * blockIndent is the size of the indent used when a new block is opened, in spaces.
     *
     * For example,
     * ```
     * fun f() {
     *   //
     * }
     * ```
     */
    val blockIndent: Property<Int> =
        objects.property(Int::class.java).convention(DEFAULT_BLOCK_INDENT)

    /**
     * continuationIndent is the size of the indent used when a line is broken because it's too
     * long, in spaces.
     *
     * For example,
     * ```
     * val foo = bar(
     *     1)
     * ```
     */
    val continuationIndent: Property<Int> =
        objects.property(Int::class.java).convention(DEFAULT_CONTINUATION_INDENT)

    /** Whether ktfmt should remove imports that are not used. */
    val removeUnusedImports: Property<Boolean> =
        objects.property(Boolean::class.java).convention(DEFAULT_REMOVE_UNUSED_IMPORTS)

    /**
     * Print the Ops generated by KotlinInputAstVisitor to help reason about formatting (i.e.,
     * newline) decisions
     */
    val debuggingPrintOpsAfterFormatting: Property<Boolean> =
        objects.property(Boolean::class.java).convention(DEFAULT_DEBUGGING_PRINT_OPTS)

    /** Enables --dropbox-style (equivalent to set blockIndent to 4 and continuationIndent to 4). */
    @Suppress("MagicNumber")
    fun dropboxStyle() {
        ktfmtStyle = DROPBOX
        blockIndent.set(4)
        continuationIndent.set(4)
    }

    /** Sets the Google style (equivalent to set blockIndent to 2 and continuationIndent to 2). */
    @Suppress("MagicNumber")
    fun googleStyle() {
        ktfmtStyle = GOOGLE
        blockIndent.set(2)
        continuationIndent.set(2)
    }

    /**
     * Sets the KotlinLang style. A format that attempts to reflect
     * https://kotlinlang.org/docs/coding-conventions.html.
     */
    @Suppress("MagicNumber")
    fun kotlinLangStyle() {
        ktfmtStyle = GOOGLE
        blockIndent.set(4)
        continuationIndent.set(4)
    }

    internal fun toBean(): FormattingOptionsBean =
        FormattingOptionsBean(
            style = ktfmtStyle,
            maxWidth = maxWidth.get(),
            blockIndent = blockIndent.get(),
            continuationIndent = continuationIndent.get(),
            removeUnusedImports = removeUnusedImports.get(),
            debuggingPrintOpsAfterFormatting = debuggingPrintOpsAfterFormatting.get()
        )

    internal companion object {
        internal const val DEFAULT_MAX_WIDTH: Int = 100
        internal const val DEFAULT_BLOCK_INDENT: Int = 2
        internal const val DEFAULT_CONTINUATION_INDENT: Int = 4
        internal const val DEFAULT_REMOVE_UNUSED_IMPORTS: Boolean = true
        internal const val DEFAULT_DEBUGGING_PRINT_OPTS: Boolean = false
    }
}
