/*
 * @(#) AudioCapture.java 2015. 1.
 *
 * Copyright 2015 Naver Corp. All rights Reserved.
 * Naver PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.naver.speech.clientapi;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

import android.media.AudioFormat;
import android.media.AudioRecord;
import android.media.MediaRecorder;

class AudioCapture {

	private final static String TAG = AudioCapture.class.getSimpleName();

	private static final int SAMPLING_FREQ = 16000;
	private static final int AUDIO_READ_SIZE_IN_SHORTS = SAMPLING_FREQ  / 10 * 2; // 200ms
	private static final int AUDIO_READ_SIZE_IN_BYTES  = AUDIO_READ_SIZE_IN_SHORTS * 2; // 200ms

	private AudioRecord audioRecord;
	private short[] audioBuffer     = new short[AUDIO_READ_SIZE_IN_SHORTS];
	private byte [] audioByteBuffer = new byte [AUDIO_READ_SIZE_IN_BYTES];
	private int recordedAudioBufferSize = 0;

	private boolean isAudioFileInput = false;
	private String  mAudioFilePath   = "";
	private File    mFile;
	private FileInputStream mInputStream;

	static final int OK    =  0;
	static final int ERROR = -1;

	void setAudioRecord(AudioRecord audioRecord) {
		this.audioRecord = audioRecord;
	}

	void beforeStart() throws IllegalArgumentException, IllegalStateException, NullPointerException, FileNotFoundException, Exception {
		recordedAudioBufferSize = 0;

		if(!isAudioFileInput) {
			audioRecord = new AudioRecord(MediaRecorder.AudioSource.VOICE_RECOGNITION,
					SAMPLING_FREQ, AudioFormat.CHANNEL_IN_MONO,
					AudioFormat.ENCODING_PCM_16BIT, 320000);

			audioRecord.startRecording();
		} else {
			mFile = new File(mAudioFilePath);
			mInputStream = new FileInputStream(mFile);
		}
	}

	void beforeFinish() throws IllegalStateException, IOException, Exception {
		if(!isAudioFileInput) {
			audioRecord.stop();
			audioRecord.release();
		} else {
			mInputStream.close();
		}
	}

	short[] record() throws AudioCaptureException, IOException, BufferUnderflowException, InterruptedException {

		if(!isAudioFileInput) {
			int readed = audioRecord.read(audioBuffer, 0, AUDIO_READ_SIZE_IN_SHORTS);
			if(readed < 0) {
				switch (readed) {
				case AudioRecord.ERROR:
					throw new AudioCaptureException("AudioRecord.read didn't work : AudioRecord.ERROR");
				case AudioRecord.ERROR_BAD_VALUE:
					throw new AudioCaptureException("AudioRecord.read didn't work : AudioRecord.ERROR_BAD_VALUE");
				case AudioRecord.ERROR_INVALID_OPERATION:
					throw new AudioCaptureException("AudioRecord.read didn't work : AudioRecord.ERROR_INVALID_OPERATION");
				default:
					throw new AudioCaptureException("AudioRecord.read didn't work : " + readed);
				}
			}

			recordedAudioBufferSize += readed;
		} else {
			if(mInputStream.available() > 0) {
				int readed = mInputStream.read(audioByteBuffer, 0, AUDIO_READ_SIZE_IN_BYTES);
				if(readed < 0) {
					throw new AudioCaptureException("FileInputStream.read didn't work");
				}
				ByteBuffer.wrap(audioByteBuffer).order(ByteOrder.LITTLE_ENDIAN).asShortBuffer().get(audioBuffer);

				recordedAudioBufferSize += readed / 2;
			} else {
				return null;
			}
		}
		return audioBuffer;
	}

	int getRecoredAudioBufferSize() {
		return recordedAudioBufferSize;
	}

	public boolean isAudioFileInput() {
		return isAudioFileInput;
	}

	public void setAudioFileInput(boolean isAudioFileInput) {
		this.isAudioFileInput = isAudioFileInput;
	}

	public String getAudioFilePath() {
		return mAudioFilePath;
	}

	public void setAudioFilePath(String audioFilePath) {
		this.mAudioFilePath = audioFilePath;
	}
}
