/*
 * @(#) SpeechRecognizer.java 2015. 1.
 *
 * Copyright 2015 Naver Corp. All rights Reserved.
 * Naver PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.naver.speech.clientapi;

import android.content.Context;

import com.naver.speech.clientapi.SpeechConfig.EndPointDetectType;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

/**
 * 이 클래스는 네이버 음성인식 서비스를 사용할 수 있도록 클라이언트 역할을 제공합니다.
 * 네이버 음성인식은 음성인식 서버와 클라이언트 간의 통신으로 구성되어 있고,
 * 클라이언트는 음성 입력을 받아 서버로 전송하고, 서버로부터 인식 결과를 받는 역할을 동시에 수행합니다.
 * 따라서 반드시 현재 어플리케이션이 android.permission.RECORD_AUDIO 와 android.permission.INTERNET 사용 권한을 가지고 있는지 확인해야 합니다.
 * <p>
 * 이 클래스는 실행 전 반드시 {@link SpeechRecognizer#initialize()} 메소드를 호출하여야 하고, 종료 후 반드시 {@link SpeechRecognizer#release()} 메소드를 호출하여야 합니다.
 * 이는 음성인식으로 인한 자원을 초기화하거나, 해제함으로써 다른 프로세스에 악영향을 주는 것을 방지하기 위함입니다.
 * 사용 시나리오에 따라서 달라질 수 있지만,
 * 적어도 {@link android.app.Activity#onStart()} 콜백이 호출될 때에는 {@link SpeechRecognizer#initialize()}를,
 * {@link android.app.Activity#onStop()} 콜백이 호출될 때에는 {@link SpeechRecognizer#release()}를 호출하는 것을 권장합니다.
 * <p>
 * 이 클래스는 {@link SpeechRecognizer#recognize(SpeechConfig)} 메소드를 호출함으로써 음성인식을 시작합니다.
 * 음성인식 수행 과정은 아래와 같은 스테이트머신에 기반합니다. (다이어그램 참고)
 * <p>
 * <img src="https://raw.githubusercontent.com/naver/naverspeech-sdk-android/gh-pages/resources/state_diagram.png" alt="ast state diagram">
 * <p>
 * 각 스테이트로 천이할 때 마다 콜백 메소드가 호출되며, {@link SpeechRecognitionListener} 를 상속받아서 콜백 메소드들을 오버라이딩 해야 합니다.
 * 이를 통해서 음성인식 시작, 중간 결과, 최종 결과 등 다양한 동작을 어플리케이션에 이용할 수 있습니다.
 *
 */
public class SpeechRecognizer {

	protected SpeechRecognitionListener speechRecognitionListener;

	private AudioCapture mAudioCapture;
    private Context context;
	private Boolean mIsGov;

	//////////////////////////////////////////////////////////////
	//
	// version of naverspeech-sdk-c
	private static final String CLIENT_LIB_VER = "1.1.16";
	
	//////////////////////////////////////////////////////////////
	//
	// Error code
	/**
	 * 네트워크 자원 초기화 오류
	 */
	public static final int ERROR_NETWORK_INITIALIZE   = 10;
	/**
	 * 네트워크 자원 해제 오류
	 */
	public static final int ERROR_NETWORK_FINALIZE     = 11;
	/**
	 * 네트워크 데이터 수신 오류
	 */
	public static final int ERROR_NETWORK_READ         = 12;
	/**
	 * 네트워크 데이터 전송 오류
	 */
	public static final int ERROR_NETWORK_WRITE        = 13;
	/**
	 * 인식 서버 오류
	 */
	public static final int ERROR_NETWORK_NACK         = 14;
	/**
	 * 전송/수신 패킷 오류
	 */
	public static final int ERROR_INVALID_PACKET       = 15;
	/**
	 * 오디오 자원 초기화 오류
	 *
	 * @see android.media.AudioRecord
	 */
	public static final int ERROR_AUDIO_INITIALIZE     = 20;
	/**
	 * 오디오 자원 해제 오류
	 *
	 * @see android.media.AudioRecord
	 */
	public static final int ERROR_AUDIO_FINIALIZE      = 21;
	/**
	 * 음성 입력(녹음) 오류
	 */
	public static final int ERROR_AUDIO_RECORD         = 22;
	/**
	 * 인증 권한 오류
	 */
	public static final int ERROR_SECURITY             = 30;
	/**
	 * 인식 결과 오류
	 */
	public static final int ERROR_INVALID_RESULT       = 40;
	/**
	 * 일정 시간 이상 인식 서버로 음성을 전송하지 못하거나, 인식 결과를 받지 못하였음
	 */
	public static final int ERROR_TIMEOUT              = 41;
	/**
	 * 클라이언트가 음성인식을 수행하고 있지 않는 상황에서, 특정 음성인식 관련 이벤트가 감지되었음
	 */
	public static final int ERROR_NO_CLIENT_RUNNING    = 42;
	/**
	 * 클라이언트 내부에 규정되어 있지 않은 이벤트가 감지되었음
	 */
	public static final int ERROR_UNKOWN_EVENT         = 50;
	/**
	 * 프로토콜 버전 오류
	 */
	public static final int ERROR_VERSION              = 60;
	/**
	 * 클라이언트 프로퍼티 오류
	 */
	public static final int ERROR_CLIENTINFO           = 61;
	/**
	 * 음성인식 서버의 가용 풀(pool) 부족
	 */
	public static final int ERROR_SERVER_POOL          = 62;
	/**
	 * 음성인식 서버 세션 만료
	 */
	public static final int ERROR_SESSION_EXPIRED      = 63;
	/**
	 * 음성 패킷 사이즈 초과
	 */
	public static final int ERROR_SPEECH_SIZE_EXCEEDED = 64;
	/**
	 * 인증 time stamp 불량
	 */
	public static final int ERROR_EXCEED_TIME_LIMIT    = 65;
	/**
	 * 올바른 Service Type이 아님
	 */
	public static final int ERROR_WRONG_SERVICE_TYPE   = 66;
	/**
	 * 올바른 Language Type이 아님
	 */
	public static final int ERROR_WRONG_LANGUAGE_TYPE  = 67;
	/**
	 * OpenAPI 인증 에러(Client ID 또는 AndroidManifest.xml의 package가 개발자센터에 등록한 값과 다름)
	 */
	public static final int ERROR_OPENAPI_AUTH         = 70;
	/**
	 * 정해진 Quota를 다 소진함
	 */
	public static final int ERROR_QUOTA_OVERFLOW       = 71;

	//////////////////////////////////////////////////////////////

	/**
	 * 음성인식 클라이언트 인스턴스를 생성하고, OpenAPI 인증 작업을 수행합니다.
	 * @param context 메인 액티비티의 context
	 * @param clientId 개발자센터에서 "내 애플리케이션"을 등록할 때 발급받은 Client ID
	 * @throws SpeechRecognitionException 예외가 발생하는 경우는 아래와 같습니다.<br>
	 * 1. context 파라미터가 올바른 메인 액티비티의 context가 아님<br>
	 * 2. AndroidManifest.xml에서 package 이름을 올바르게 등록하지 않았음<br>
	 * 3. package 이름을 올바르게 등록했지만 과도하게 긴 경우(256바이트 이하를 권장)<br>
	 * 4. clientId 파라미터가 null인 경우<br>
	 * 개발하면서 예외가 발생하지 않았다면 실서비스에서도 예외는 발생하지 않습니다. 개발 초기에만 주의하시면 됩니다.
	 */
	public SpeechRecognizer(Context context, String clientId) throws SpeechRecognitionException {
		this(context, clientId, false);
	}

	protected SpeechRecognizer(Context context, String clientId, boolean isGov) throws SpeechRecognitionException {
		this.mAudioCapture = new AudioCapture();
		this.context = context;
		this.mIsGov = isGov;

		String errMsg = setupJNI(CLIENT_LIB_VER, android.os.Build.MODEL, android.os.Build.VERSION.RELEASE, clientId, context, mIsGov);
		if (errMsg != null) {
			throw new SpeechRecognitionException(errMsg);
		}
	}

	/**
	 * 음성인식 자원을 초기화해줍니다. {@link android.app.Activity#onStart()} 콜백 시점에 호출되는 것을 권장합니다.
	 */
	public void initialize() {
		initializeJNI();
	}

	/**
	 * 음성인식 자원을 해제해줍니다. {@link android.app.Activity#onStop()} 콜백 시점에 호출되는 것을 권장합니다.
	 */
	public void release() {
		releaseJNI();
	}

	/**
	 * 음성인식의 각종 콜백 함수를 정의할 리스너를 등록합니다.
	 * @param callback 콜백 리스너
	 * @see SpeechRecognitionListener
	 */
	public void setSpeechRecognitionListener(SpeechRecognitionListener callback) {
		this.speechRecognitionListener = callback;
	}

	/**
	 * 음성인식의 EPD(end point detecion) 종류를 선택합니다.
	 * <br>
	 * 이 함수는 hybrid EPD 모드에서만 동작합니다. 그 외의 EPD 모드에서 호출할 경우, 동작하지 않고 false를 반환합니다.
	 * EPD 종류를 선택, 결정되면 {@link SpeechRecognizer#onEndPointDetectTypeSelected(int)} 콜백 함수가 호출됩니다.
	 * @param epdType EPD 종류(AUTO, MANUAL 중 하나를 선택)
	 * @return EPD 종류 선택 동작이 완료되면 true를 반환합니다.
	 * hybrid EPD 모드가 아닌 상태에서 호출하거나, 클라이언트가 음성인식을 수행 중이지 않는 상태에서 호출하면 false를 반환합니다.
	 * @see EndPointDetectType#AUTO
	 * @see EndPointDetectType#MANUAL
	 * @see EndPointDetectType#HYBRID
     */
	public boolean selectEPDTypeInHybrid(EndPointDetectType epdType) {
		return selectEPDTypeInHybridJNI(epdType.toInteger());
	}

	/**
	 * 이 함수를 호출하는 즉시 {@link SpeechRecognizer#onEndPointDetected}로 천이됩니다.
	 * <br>
	 * 즉, 이 함수를 호출하는 즉시 음성인식 서버로부터 최종 결과를 받은 후 종료됩니다.
	 * 네트워크 환경에 따라 최종 결과 수신 및 종료가 지연될 수 있습니다.
	 * @return {@link SpeechRecognizer#onEndPointDetected}로 천이가 완료되면 true를 반환합니다.
	 * 해당 동작 중에 오류가 발생하면, 동작을 완료하지 못하고 false를 반환합니다.
     */
	public boolean stop() {
		if(isRunning())
			return sendUserEPDJNI();
		else
			return false;
	}

	/**
	 * 이 함수를 호출하는 즉시 음성인식의 모든 동작을 즉시 정지하는 이벤트를 발생시킵니다.
	 * @return 즉시 정지 이벤트를 발생시킨 후, true를 반환합니다.
	 * 해당 동작 중에 오류가 발생하면, 이벤트를 발생시키지 못하고 false를 반환합니다.
	 */
	public boolean cancel() {
		return stopListeningJNI();
	}

	/**
	 * 현재 음성인식의 수행 여부를 반환합니다.
	 * @return 음성인식이 수행 중일 경우 true를 반환합니다.
	 */
	public boolean isRunning() {
		return isRunningJNI();
	}

	/**
	 * 음성인식을 시작하는 이벤트를 발생시킵니다.
	 * {@link SpeechRecognizer}는 시작 이벤트가 발생하면, 백그라운드로 오디오 자원 할당 및 네트워크 연결을 시도합니다.
	 * 그리고 이어서 발생하는 이벤트에 따라 콜백 함수를 호출하면서 음성인식을 수행하게 됩니다.
	 * 이 함수는 오직 시작 이벤트를 발생시키는 동작만 수행한 후, 종료됩니다.
	 * @param config 음성인식 서비스 종류, 언어 종류, EPD(end point detection)모드 종류 등 설정값을 포함
	 * @return 시작 이벤트를 발생시킨 후, true를 반환합니다.
	 * 해당 동작 중에 오류가 발생하면, 이벤트를 발생시키지 못하고 false를 반환합니다.
	 * @see SpeechRecognitionListener
	 *
	 * @throws SpeechRecognitionException config의 값이 null이거나 이미 음성인식이 동작 중임
	 */
	public boolean recognize(SpeechConfig config) throws SpeechRecognitionException {

		if (isRunning())
			throw new SpeechRecognitionException("Speech Recognizer is already running");

		if (config.getServiceType() == null)
			throw new SpeechRecognitionException("ServiceType is null");

		if (config.getLanguageType() == null)
			throw new SpeechRecognitionException("LanguageType is null");

		if (config.getEndPointDetectType() == null)
			throw new SpeechRecognitionException("LanguageType is null");

		if (config.getLanguageType() == null)
			throw new SpeechRecognitionException("LanguageType is null");

		if (config.getEndPointDetectType() == null)
			throw new SpeechRecognitionException("EndPointDetectType is null");

		if (config.getConnectionType() == SpeechConfig.ConnectionType.GRPC_SECURE) {
		    if (!this.setSSLOptions()) {
		         throw new SpeechRecognitionException("SSL options is incorrect");
            }
        }

		return startListeningJNI(config.getServiceType().toInteger(),
                config.getLanguageType().toInteger(),
                config.getCaptureType().toInteger(),
                config.getWakewordType().toInteger(),
                config.isQuestionDetection(),
                config.getEndPointDetectType().toInteger(),
                config.getConnectionType().toInteger(),
                config.getExtraInfo(),
                this.mIsGov
        );
	}

	private boolean setSSLOptions() {
	    if (context == null) {
	        return false;
        }

        Context ctx = this.context;
        String sslRootsFile = "roots.pem";
        InputStream in = null;

        try {
            in = ctx.getAssets().open(sslRootsFile);

            File outFile = new File(ctx.getExternalFilesDir(null), sslRootsFile);
            OutputStream out = new FileOutputStream(outFile);

            byte[] buffer = new byte[1024];
            int bytesRead;

            while ((bytesRead = in.read(buffer)) != -1) {
                out.write(buffer, 0, bytesRead);
            }

            in.close();
            out.close();

            configureSslRoots(outFile.getCanonicalPath());
        } catch (IOException e) {
            e.printStackTrace();

            return false;
        }

        return true;
    }

	// JNI reflection
	protected int startAudioRecording() {
		try {
			mAudioCapture.beforeStart();
		} catch (Exception e) {
			e.printStackTrace();
			return AudioCapture.ERROR;
		}
		return AudioCapture.OK;
	}

	protected int stopAudioRecording() {
		try {
			mAudioCapture.beforeFinish();
		} catch (Exception e) {
			e.printStackTrace();
			return AudioCapture.ERROR;
		}
		return AudioCapture.OK;
	}

	protected short[] record() {
		try {
			short[] speech = mAudioCapture.record();
			return speech;
		} catch (Exception e) {
			e.printStackTrace();
		}
		return null;
	}

	/**
	 * {@link SpeechRecognitionListener#onInactive()} 참고
	 */
	protected void onInactive() {
		if (null != speechRecognitionListener) {
			speechRecognitionListener.onInactive();
		}
	}

	/**
	 * {@link SpeechRecognitionListener#onReady()} 참고
	 */
	protected void onReady() {
		if (null != speechRecognitionListener) {
			speechRecognitionListener.onReady();
		}
	}

	/**
	 * {@link SpeechRecognitionListener#onRecord(short[])} 참고
	 * @param speech speech
	 */
	protected void onRecord(short[] speech) {
		if (null != speechRecognitionListener) {
			speechRecognitionListener.onRecord(speech);
		}
	}

	/**
	 * {@link SpeechRecognitionListener#onPartialResult(String)} 참고
	 * @param partialResult partial result
	 */
	protected void onPartialResult(String partialResult) {
		if (null != speechRecognitionListener) {
			speechRecognitionListener.onPartialResult(partialResult);
		}
	}

	/**
	 * {@link SpeechRecognitionListener#onEndPointDetected()} 참고
	 */
	protected void onEndPointDetected() {
		if (null != speechRecognitionListener) {
			speechRecognitionListener.onEndPointDetected();
		}
	}

	/**
	 * {@link SpeechRecognitionListener#onResult(SpeechRecognitionResult)}  참고
	 * @param finalResult final result
	 */
	protected void onResult(Object[] finalResult) {
		if (null != speechRecognitionListener) {
			SpeechRecognitionResult speechRecognitionResult = new SpeechRecognitionResult(finalResult);
			speechRecognitionListener.onResult(speechRecognitionResult);
		}
	}

	/**
	 * {@link SpeechRecognitionListener#onError(int)} 참고
	 * @param errorCode error code
	 */
	protected void onError(int errorCode) {
		if (null != speechRecognitionListener) {
			speechRecognitionListener.onError(errorCode);
		}
	}

	/**
	 * {@link SpeechRecognitionListener#onEndPointDetectTypeSelected(EndPointDetectType)} 참고
	 * @param epdType EPD(end point detection) type
     */
	protected void onEndPointDetectTypeSelected(int epdType) {
		if (null != speechRecognitionListener) {
			switch(epdType) {
			case 0:
				speechRecognitionListener.onEndPointDetectTypeSelected(EndPointDetectType.AUTO);
				break;
			case 1:
				speechRecognitionListener.onEndPointDetectTypeSelected(EndPointDetectType.MANUAL);
				break;
			default:
				break;
			}
		}
	}

	static {
		System.loadLibrary("naverspeech-sdk-c");
	}

	private native static String setupJNI(String client_lib_ver, String device, String os, String cliendId, Context context, boolean isGov);

	private native void initializeJNI();
	private native void releaseJNI();

	private native boolean selectEPDTypeInHybridJNI(int epdType);
	private native boolean sendUserEPDJNI();

	private native boolean startListeningJNI(int serviceType, int languageType, int captureType, int wakewordType, boolean questionDetection, int epdType, int connectionType, String extraInfo, boolean isGov);
	private native boolean stopListeningJNI();

	private native boolean isRunningJNI();

	public static native void configureSslRoots(String path);
}
