package com.nhn.android.naverlogin.util;


import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.ResolveInfo;
import android.net.Uri;
import android.support.customtabs.CustomTabsIntent;
import android.support.v4.app.FragmentActivity;
import android.support.v4.content.LocalBroadcastManager;
import android.util.Log;

import java.util.ArrayList;
import java.util.List;

import static android.support.customtabs.CustomTabsService.ACTION_CUSTOM_TABS_CONNECTION;

/**
 * 커스텀 탭 정보를 가져오는 매니저
 * Created by ksmail13 on 17. 7. 28.
 */
public final class CustomTabsManager {

    public static final String ACTION_NAVER_CUSTOM_TAB = "ACTION_NAVER_3RDPARTY_CUSTOM_TAB";

    private static final String CUSTOMTAB_DIALOG = "CUSTOMTAB_DIALOG";


    private final Context mContext;

    /**
     * Create CustomTabManager
     * @param base If manager will open CustomTabDialog then you must send {@link FragmentActivity}
     */
    public CustomTabsManager(Context base) {
        mContext = base;
    }

    private static final String STABLE_PACKAGE = "com.android.chrome";
    private static final String LOCAL_PACKAGE = "com.google.android.apps.chrome";

    /**
     * Find available CustomTab application packages
     *
     * @return a list of packages that support Custom Tabs.
     */
    public static List<PackageInfo> getCustomTabsPackages(Context context) {
        PackageManager pm = context.getPackageManager();
        // Get default VIEW intent handler.
        Intent activityIntent = new Intent(Intent.ACTION_VIEW, Uri.parse("http://www.example.com"));
        // Get all apps that can handle VIEW intents.
        List<ResolveInfo> resolvedActivityList = pm.queryIntentActivities(activityIntent, 0);
        List<PackageInfo> packagesSupportingCustomTabs = new ArrayList<>();
        for (ResolveInfo info : resolvedActivityList) {
            Intent serviceIntent = new Intent();
            serviceIntent.setAction(ACTION_CUSTOM_TABS_CONNECTION);
            serviceIntent.setPackage(info.activityInfo.packageName);
            Log.d("custom tab util", info.activityInfo.packageName);
            // Check if this package also resolves the Custom Tabs service.
            if (pm.resolveService(serviceIntent, 0) != null) {
                try {
                    ApplicationInfo ai = pm.getApplicationInfo(info.activityInfo.packageName, 0);
                    if (ai.enabled) {
                        packagesSupportingCustomTabs.add(pm.getPackageInfo(info.activityInfo.packageName, 0));
                    }
                } catch (PackageManager.NameNotFoundException e) {
                    e.printStackTrace();
                }
            }
        }
        return packagesSupportingCustomTabs;
    }

    /**
     * Check if chrome stable version in device
     *
     * @return True if chrome stable on device, False if not exist
     */
    private boolean chromeInstalledOrNot() {
        PackageManager pm = mContext.getPackageManager();
        try {
            pm.getPackageInfo(STABLE_PACKAGE, PackageManager.GET_ACTIVITIES);
            ApplicationInfo ai = pm.getApplicationInfo(STABLE_PACKAGE, 0);
            return ai.enabled;
        } catch (PackageManager.NameNotFoundException ignore) {
            // 현재 크롬이 설치되어 있는지 확인하기 위한 작업이기 때문에
            // 추가적인 예외처리가 필요하지 않습니다.
        }

        return false;
    }

    /**
     * 커스텀 탭을 사용할 수 있는지 확인한다.
     *
     * @return 사용가능한 커스텀탭 패키지가 있거나 크롬이 설치되어 있으며,
     * 현재 안드로이드 버전이 커스텀탭 지원 버전보다 높을 경우 true 아니면 false
     */
    public static boolean isCustomTabAvailable(Context context) {
        List<PackageInfo> customTabsPackages = getCustomTabsPackages(context);
        return customTabsPackages != null && customTabsPackages.size() > 0;
//		return true;
    }

    /**
     * 커스텀 탭을 이용해 url을 호출한다.
     * 커스텀 탭 사용가능 앱이 여러개 일경우 다이얼 로그를
     * @param packageName 커스텀 탭 패키지 명
     * @param url 실행할 url
     */
    public void launchUrl(String packageName, String url) {
        final CustomTabsIntent customTabsIntent = new CustomTabsIntent.Builder()
                .setShowTitle(true)
                .build();

        customTabsIntent.intent.setPackage(packageName);

        launchUrl(customTabsIntent, url);
    }

    /**
     * 커스텀 탭을 이용해 url을 호출한다.
     *
     * @param intent 커스텀탭 실행 옵션이 포함된 인텐트
     * @param url    실행할 url
     */
    public void launchUrl(CustomTabsIntent intent, String url) {
        intent.launchUrl(mContext, Uri.parse(url));
    }


    /**
     * 커스텀 탭 리스너를 지정합니다.<br />
     * 리스너는 <b>일회용</b>입니다. 사용에 주의하세요.
     *
     * @param listener 실행될 리스너 결과 값은 {@link Intent}로 주고 받습니다
     */
    public void setCustomTabListener(final CustomTabsListener listener) {
        final LocalBroadcastManager instance = LocalBroadcastManager.getInstance(mContext);

        instance.registerReceiver(new BroadcastReceiver() {
            @Override
            public void onReceive(Context context, Intent intent) {
                listener.onReceive(intent);
                instance.unregisterReceiver(this);
            }
        }, new IntentFilter(ACTION_NAVER_CUSTOM_TAB));
    }


    /**
     * 커스텀 탭의 결과 값을 리턴
     *
     * @param intent 리턴 값
     */
    public void sendCustomTabResult(Intent intent) {
        intent.setAction(ACTION_NAVER_CUSTOM_TAB);
        intent.setClass(mContext, CustomTabsManager.class);
        LocalBroadcastManager instance = LocalBroadcastManager.getInstance(mContext);
        instance.sendBroadcast(intent);
    }
}
