package com.nhn.android.naverlogin.ui;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.os.AsyncTask;
import android.os.Bundle;
import android.os.Message;
import android.support.annotation.NonNull;
import android.text.TextUtils;

import com.nhn.android.idp.common.logger.Logger;
import com.nhn.android.idp.common.util.DeviceAppInfo;
import com.nhn.android.naverlogin.OAuthLogin;
import com.nhn.android.naverlogin.OAuthLoginDefine;
import com.nhn.android.naverlogin.connection.OAuthLoginConnection;
import com.nhn.android.naverlogin.data.OAuthErrorCode;
import com.nhn.android.naverlogin.data.OAuthIntent;
import com.nhn.android.naverlogin.data.OAuthLoginBrowserIntentParam;
import com.nhn.android.naverlogin.data.OAuthLoginData;
import com.nhn.android.naverlogin.data.OAuthLoginPreferenceManager;
import com.nhn.android.naverlogin.data.OAuthResponse;
import com.nhn.android.naverlogin.util.CustomTabsListener;
import com.nhn.android.naverlogin.util.CustomTabsManager;
import com.nhn.android.oauth.R;

/**
 * refresh token 이 있는 경우 refresh token 으로 access token 을 가져옴
 * access token 가져오는 걸 실패하는 경우엔 로그인 창을 보여줌 (네이버앱을 통하거나, 직접 webview를 생성해서..)
 *
 * @author naver
 */
public class OAuthLoginActivity extends Activity {
	private static final String TAG = "OAuthLoginActivity";
	private static int REQUESTCODE_LOGIN = 100;
	private static int CUSTOMTAB_LOGIN = -1;

	// dialog
	private OAuthLoginDialogMng mDialogMng = new OAuthLoginDialogMng();


	private Context mContext;
	private OAuthLoginData mOAuthLoginData;
	private String mClientName;

	private boolean mIsLoginActivityStarted = false;
	private CustomTabsListener mCustomTabListener = new CustomTabsListener() {
		@Override
		public void onReceive(Intent intent) {
			if (intent == null) {
				intent = new Intent();
				intent.putExtra(OAuthIntent.EXTRA_OAUTH_ERROR_CODE
						, OAuthErrorCode.CLIENT_USER_CANCEL.getCode());
				intent.putExtra(OAuthIntent.EXTRA_OAUTH_ERROR_DESCRIPTION
						, OAuthErrorCode.CLIENT_USER_CANCEL.getDesc());
			}
			onActivityResult(CUSTOMTAB_LOGIN, RESULT_OK, intent);
		}
	};
	;

	private boolean initData(Bundle savedInstanceState) {

		mContext = OAuthLoginActivity.this;

		OAuthLoginPreferenceManager pref = new OAuthLoginPreferenceManager(mContext);
		String clientId = pref.getClientId();
		String clientSecret = pref.getClientSecret();
		String callbackUrl = pref.getCallbackUrl();
		String state = (null == savedInstanceState) ? null : savedInstanceState.getString("OAuthLoginData_state");

		mClientName = pref.getClientName();

		if (TextUtils.isEmpty(clientId)) {
			finishWithErrorResult(OAuthErrorCode.CLIENT_ERROR_NO_CLIENTID);
			return false;
		}
		if (TextUtils.isEmpty(clientSecret)) {
			finishWithErrorResult(OAuthErrorCode.CLIENT_ERROR_NO_CLIENTSECRET);
			return false;
		}
		if (TextUtils.isEmpty(mClientName)) {
			finishWithErrorResult(OAuthErrorCode.CLIENT_ERROR_NO_CLIENTNAME);
			return false;
		}
		if (TextUtils.isEmpty(callbackUrl)) {
			finishWithErrorResult(OAuthErrorCode.CLIENT_ERROR_NO_CALLBACKURL);
			return false;
		}

		mOAuthLoginData = new OAuthLoginData(clientId, clientSecret, callbackUrl, state);

		return true;
	}


	@Override
	protected void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);

		if (!Logger.isRealVersion()) {
			Logger.d(TAG, "onCreate()");
		}

		if (!initData(savedInstanceState)) {
			return;
		}

		if (null != savedInstanceState) {
			mIsLoginActivityStarted = savedInstanceState.getBoolean("IsLoginActivityStarted");
		}


		if (!mIsLoginActivityStarted) {
			mIsLoginActivityStarted = true;

			if (!Logger.isRealVersion()) {
				Logger.d(TAG, "onCreate() first");
			}
			runOnlyOnce();
		}

	}


	@Override
	protected void onRestoreInstanceState(Bundle savedInstanceState) {
		super.onRestoreInstanceState(savedInstanceState);

		if (!Logger.isRealVersion()) {
			Logger.d(TAG, "onRestoreInstanceState()");
		}

		if (null != savedInstanceState) {
			mIsLoginActivityStarted = savedInstanceState.getBoolean("IsLoginActivityStarted");
		}

	}

	@Override
	protected void onSaveInstanceState(Bundle outState) {
		super.onSaveInstanceState(outState);

		if (!Logger.isRealVersion()) {
			Logger.d(TAG, "Login onSaveInstanceState()");
		}

		outState.putBoolean("IsLoginActivityStarted", mIsLoginActivityStarted);
		outState.putString("OAuthLoginData_state", mOAuthLoginData.getInitState());

	}

	/**
	 * 각 상황별로 네이버 로그인을 진행할 액티비티를 실행
	 * @param mOAuthLoginData 네아로 메타 정보
	 */
	private void startLoginActivity(OAuthLoginData mOAuthLoginData) {
		if (!Logger.isRealVersion()) {
			Logger.d(TAG, "startLoginActivity()");
		}

		// https://oss.navercorp.com/id-mobilesdk/naverid-3rdparty-sdk-android/issues/39
		// 에 의해 비공개로 로직을 남겨둔다
		if (OAuthLoginDefine.LOGIN_BY_NAVERAPP_ONLY) {
			// Only Naver Login
			tryOAuthByNaverapp(mOAuthLoginData);
			return;
		}
		else if (OAuthLoginDefine.LOGIN_BY_CUSTOM_TAB_ONLY) {
			// Only Custom Tab Login
			tryOAuthByCustomTab(mOAuthLoginData);
			return;
		}
		else if (OAuthLoginDefine.LOGIN_BY_WEBVIEW_ONLY) {
			// Only WebView Login
			startLoginWebviewActivity(mOAuthLoginData);
			return;
		}
		// Test 용이 아닌 기본 로직은 아래의 else를 타게 된다
		if (!OAuthLoginDefine.LOGIN_BY_WEBVIEW_ONLY) {
			if (tryOAuthByNaverapp(mOAuthLoginData)) return;
			if (tryOAuthByCustomTab(mOAuthLoginData)) return;
		}
		// inapp browser 를 통해 기존 로그인 방식으로 로그인한다.
		startLoginWebviewActivity(mOAuthLoginData);
	}

	/**
	 * 네이버 앱으로 로그인 시도
	 * @param loginData 네아로 메타 정보
	 * @return 실행 여부
	 */
	private boolean tryOAuthByNaverapp(OAuthLoginData loginData) {
		try {

			Intent intent = newParamIntent(loginData.getClientId(), loginData.getInitState(), loginData.getCallbackUrl());
			intent.putExtra(OAuthLoginBrowserIntentParam.INTENT_PARAM_KEY_APP_NAME, mClientName);

			if (DeviceAppInfo.isIntentFilterExist(mContext, OAuthLoginDefine.NAVER_PACKAGE_NAME, OAuthLoginDefine.ACTION_OAUTH_LOGIN)) {

				if (!Logger.isRealVersion()) {
					Logger.d(TAG, "startLoginActivity() with naapp");
				}

				// 네이버앱에서 처리 가능한 intent
				intent.setPackage(OAuthLoginDefine.NAVER_PACKAGE_NAME);
				intent.setAction(OAuthLoginDefine.ACTION_OAUTH_LOGIN);
				startActivityForResult(intent, REQUESTCODE_LOGIN);
				return true;
			}

		} catch (Exception e) {
			e.printStackTrace();
		}
		return false;
	}

	/**
	 * 커스텀 탭으로 로그인 시도
	 * @param loginData 네아로 메타 정보
	 * @return 실행 여부
	 */
	private boolean tryOAuthByCustomTab(OAuthLoginData loginData) {
		if (!CustomTabsManager.isCustomTabAvailable(this)) return false;
		CustomTabsManager manager = new CustomTabsManager(this);
		manager.setCustomTabListener(mCustomTabListener);

		Intent intent = newParamIntent(OAuthCustomTabActivity.class, loginData.getClientId(), loginData.getInitState(), loginData.getCallbackUrl());
		intent.addFlags(Intent.FLAG_ACTIVITY_NO_ANIMATION);
		startActivityForResult(intent, CUSTOMTAB_LOGIN);
		return true;
	}

	/**
	 * login webview 를 실행함
	 * 네이버 앱도 없고 커스텀 탭도 불가능한 상황일 때 혹은 필요에 의해 웹뷰만 호출
	 */
	private void startLoginWebviewActivity(OAuthLoginData loginData) {
		if (!Logger.isRealVersion()) {
			Logger.d(TAG, "startLoginActivity() with webview");
		}

		startActivityForResult(newParamIntent(OAuthLoginInAppBrowserActivity.class
					, loginData.getClientId(), loginData.getInitState(), loginData.getCallbackUrl())
				, REQUESTCODE_LOGIN);
	}

	/**
	 * 네아로 로그인을 위한 파라메터 인텐트를 생성
	 * @param clientId 클라이언트 ID
	 * @param initState 요청 state값
	 * @param callbackUrl 콜백 url (패키지명)
	 * @return 생성된 인텐트
	 */
	@NonNull
	private Intent newParamIntent(String clientId, String initState, String callbackUrl) {
		return newParamIntent(null, clientId, initState, callbackUrl);
	}

	/**
	 * 네아로 로그인을 위한 파라메터 인텐트를 생성
	 * @param nextActivity 이동할 액티비티 클래스
	 * @param clientId 클라이언트 ID
	 * @param initState 요청 state
	 * @param callbackUrl 콜백 url 패키지명
	 * @return 생성된 인텐트
	 */
	@NonNull
	private Intent newParamIntent(Class<? extends Activity> nextActivity, String clientId, String initState, String callbackUrl) {
		Intent intent;
		if(nextActivity == null) {
			intent = new Intent();
		} else {
			intent = new Intent(this, nextActivity);
		}
		intent.putExtra(OAuthLoginBrowserIntentParam.INTENT_PARAM_KEY_CLIENT_ID, clientId);
		intent.putExtra(OAuthLoginBrowserIntentParam.INTENT_PARAM_KEY_CALLBACK_URL, callbackUrl);
		intent.putExtra(OAuthLoginBrowserIntentParam.INTENT_PARAM_KEY_STATE, initState);
		intent.putExtra(OAuthLoginBrowserIntentParam.INTENT_PARAM_KEY_OAUTH_SDK_VERSION, OAuthLoginDefine.VERSION);

		return intent;
	}

	@Override
	protected void onResume() {
		super.onResume();

		if (!Logger.isRealVersion()) {
			Logger.d(TAG, "onResume()");
		}
	}

	@Override
	public void onPause() {
		super.onPause();

		if (!Logger.isRealVersion()) {
			Logger.d(TAG, "onPause()");
		}
	}

	private void runOnlyOnce() {
		if (mOAuthLoginData == null) {
			finishWithErrorResult(OAuthErrorCode.CLIENT_ERROR_NO_CLIENTID);
			return;
		}

		startLoginActivity(mOAuthLoginData);

	}



	private void finishWithErrorResult(OAuthErrorCode errCode) {
		if (!Logger.isRealVersion()) {
			Logger.d("GILSUB", "Login finishWithErrorResult()");
		}

		Intent intent = new Intent();
		OAuthLoginPreferenceManager prefMng = new OAuthLoginPreferenceManager(mContext);

		prefMng.setLastErrorCode(errCode);
		prefMng.setLastErrorDesc(errCode.getDesc());

		intent.putExtra(OAuthIntent.EXTRA_OAUTH_ERROR_CODE, errCode.getCode());
		intent.putExtra(OAuthIntent.EXTRA_OAUTH_ERROR_DESCRIPTION, errCode.getDesc());

		setResult(RESULT_CANCELED, intent);
		finish();

		propagationResult(false);
	}

	@Override
	public void onActivityResult(int requestCode, int resultCode, Intent data) {
		super.onActivityResult(requestCode, resultCode, data);

		if (requestCode == CUSTOMTAB_LOGIN && resultCode == RESULT_CANCELED) {
			Logger.d(TAG, "activity call by customtab");
			return;
		}

		if (data == null) {
			finishWithErrorResult(OAuthErrorCode.CLIENT_USER_CANCEL);
			return;
		}

		String state = data.getStringExtra(OAuthIntent.EXTRA_OAUTH_STATE);
		String code = data.getStringExtra(OAuthIntent.EXTRA_OAUTH_CODE);
		String errorCode = data.getStringExtra(OAuthIntent.EXTRA_OAUTH_ERROR_CODE);
		String errorDesc = data.getStringExtra(OAuthIntent.EXTRA_OAUTH_ERROR_DESCRIPTION);

		mOAuthLoginData.setMiddleResult(code, state, errorCode, errorDesc);

		if (!Logger.isRealVersion()) {
			Logger.d("GILSUB", "Login onActivityResult()");
		}

		if (TextUtils.isEmpty(code)) {
			OAuthLoginPreferenceManager prefMng = new OAuthLoginPreferenceManager(mContext);

			prefMng.setLastErrorCode(OAuthErrorCode.fromString(errorCode));
			prefMng.setLastErrorDesc(errorDesc);

			setResult(RESULT_CANCELED, data);
			finish();

			propagationResult(false);
		} else {
			new GetAccessTokenTask().execute();
		}
	}


	private class GetAccessTokenTask extends AsyncTask<Void, Void, OAuthResponse> {
		@Override
		protected void onPreExecute() {
			try {
				mDialogMng.showProgressDlg(mContext, mContext.getString(R.string.naveroauthlogin_string_getting_token), null);
			} catch (Exception e) {
				// do nothing
			}
		}

		@Override
		protected OAuthResponse doInBackground(Void... params) {
			try {
				return OAuthLoginConnection.requestAccessToken(mContext,
						mOAuthLoginData.getClientId(),
						mOAuthLoginData.getClientSecret(),
						mOAuthLoginData.getState(), mOAuthLoginData.getCode());
			} catch (Exception e) {
				return new OAuthResponse(OAuthErrorCode.CLIENT_ERROR_CONNECTION_ERROR);
			}
		}

		protected void onPostExecute(OAuthResponse res) {
			try {
				mDialogMng.hideProgressDlg();
			} catch (Exception e) {
				// do nothing
			}

			try {
				Intent intent = new Intent();
				OAuthLoginPreferenceManager prefMng = new OAuthLoginPreferenceManager(mContext);

				if (res.isSuccess()) {
					prefMng.setAccessToken(res.getAccessToken());
					prefMng.setRefreshToken(res.getRefreshToken());
					prefMng.setExpiresAt(System.currentTimeMillis() / 1000 + res.getExpiresIn());
					prefMng.setTokenType(res.getTokenType());
					prefMng.setLastErrorCode(OAuthErrorCode.NONE);
					prefMng.setLastErrorDesc(OAuthErrorCode.NONE.getDesc());

					intent.putExtra(OAuthIntent.EXTRA_OAUTH_ACCESS_TOKEN, res.getAccessToken());
					intent.putExtra(OAuthIntent.EXTRA_OAUTH_REFRESH_TOKEN, res.getRefreshToken());
					intent.putExtra(OAuthIntent.EXTRA_OAUTH_EXPIRES_IN, res.getExpiresIn());
					intent.putExtra(OAuthIntent.EXTRA_OAUTH_TOKEN_TYPE, res.getTokenType());
					setResult(RESULT_OK, intent);
				} else {
					if (res.getErrorCode() == OAuthErrorCode.NONE) {
						finishWithErrorResult(OAuthErrorCode.CLIENT_USER_CANCEL);
						return;
					} else {
						prefMng.setLastErrorCode(res.getErrorCode());
						prefMng.setLastErrorDesc(res.getErrorDesc());
						intent.putExtra(OAuthIntent.EXTRA_OAUTH_ERROR_CODE, res.getErrorCode().getCode());
						intent.putExtra(OAuthIntent.EXTRA_OAUTH_ERROR_DESCRIPTION, res.getErrorDesc());
						setResult(RESULT_CANCELED, intent);
					}
				}
			} catch (Exception e) {
				e.printStackTrace();
			}
			finish();

			try {
				propagationResult(res.isSuccess());
			} catch (Exception e) {
				e.printStackTrace();
			}

		}

	}

	private void propagationResult(boolean b) {
		if (OAuthLogin.mOAuthLoginHandler != null) {
			Message msg = new Message();
			msg.what = b ? 1 : 0;
			OAuthLogin.mOAuthLoginHandler.sendMessage(msg);
		}


	}

}
