package com.nhn.android.naverlogin.ui;

import android.content.Intent;
import android.content.pm.PackageInfo;
import android.content.res.Configuration;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.RequiresApi;
import android.support.v4.app.Fragment;
import android.support.v4.app.FragmentActivity;
import android.support.v4.app.FragmentTransaction;
import android.support.v7.app.AlertDialog;

import com.nhn.android.idp.common.connection.NetworkState;
import com.nhn.android.idp.common.logger.Logger;
import com.nhn.android.idp.common.util.DeviceAppInfo;
import com.nhn.android.naverlogin.OAuthLogin;
import com.nhn.android.naverlogin.OAuthLoginDefine;
import com.nhn.android.naverlogin.connection.gen.OAuthQueryGenerator;
import com.nhn.android.naverlogin.data.OAuthErrorCode;
import com.nhn.android.naverlogin.data.OAuthIntent;
import com.nhn.android.naverlogin.data.OAuthLoginBrowserIntentParam;
import com.nhn.android.naverlogin.data.OAuthLoginData;
import com.nhn.android.naverlogin.ui.view.CustomTabDialogFragment;
import com.nhn.android.naverlogin.util.CustomTabsManager;

import java.util.List;

import static com.nhn.android.naverlogin.OAuthLoginDefine.CUSTOMTAB_AVAILABLE_VER;

/**
 * <p>커스텀 탭을 관리하는 activity</p>
 * Created by 김민규 on 6/13/17.
 *
 * @author 김민규 [minkyu.kim13@navercorp.com]
 */
@RequiresApi(api = CUSTOMTAB_AVAILABLE_VER)
public class OAuthCustomTabActivity extends FragmentActivity {

	public static final String TAG = OAuthCustomTabActivity.class.getSimpleName();
	public static final String SAVE_CUSTOM_TAB_OPEN = "isCustomTabOpen";
	private boolean mIsCustomTabOpen;

	private CustomTabsManager mCustomTabManager;
	private AlertDialog mAlertDialog;

	@Override
	protected void onCreate(@Nullable Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		if (savedInstanceState == null) {
			Logger.d(TAG, "Open Custom Tab Activity");
			mCustomTabManager = new CustomTabsManager(this);
			initIntentData();
		}

		if(savedInstanceState != null && savedInstanceState.getBoolean(SAVE_CUSTOM_TAB_OPEN, false)) {
			onNewIntent(getIntent());
		}
	}

	/**
	 * 로그인 요청시 넘어오는 파라메터를 정리한다.
	 */
	private void initIntentData()
	{
		Intent oauthParam = getIntent();
		if (oauthParam != null) {
			Logger.d(TAG, "read request");
			String clientId = oauthParam.getStringExtra(OAuthLoginBrowserIntentParam.INTENT_PARAM_KEY_CLIENT_ID);
			String callbackUrl = oauthParam.getStringExtra(OAuthLoginBrowserIntentParam.INTENT_PARAM_KEY_CALLBACK_URL);
			String state = oauthParam.getStringExtra(OAuthLoginBrowserIntentParam.INTENT_PARAM_KEY_STATE);

			String locale = DeviceAppInfo.getBaseInstance().getLocaleString(this);
			String network = NetworkState.getNetworkState(this);

			OAuthLoginData oauthLoginData = new OAuthLoginData(clientId, null, callbackUrl, state);
			String oauthUrl = new OAuthQueryGenerator().generateRequestCustomTabAuthorizationUrl(clientId, oauthLoginData.getInitState(), callbackUrl, locale, network, OAuthLoginDefine.VERSION);

			openCustomTab(oauthUrl);
		}
	}

	private void openCustomTab(final String oauthUrl) {
		List<PackageInfo> customTabsPackages = CustomTabsManager.getCustomTabsPackages(this);
		if(customTabsPackages.size() == 1) {
			mCustomTabManager.launchUrl(customTabsPackages.get(0).packageName, oauthUrl);
			return;
		}

		FragmentTransaction ft = getSupportFragmentManager().beginTransaction();
		Fragment prev = getSupportFragmentManager().findFragmentByTag(CustomTabDialogFragment.DIALOG_TAG);
		if(prev != null) {
			ft.remove(prev);
		}
		ft.addToBackStack(null);

		CustomTabDialogFragment fragment =
				CustomTabDialogFragment.newInstance(customTabsPackages);
		fragment.setPackageSelectListener(new CustomTabDialogFragment.OnPackageSelectListener() {
			@Override
			public void onPackageSelect(PackageInfo packageInfo) {
				if(packageInfo == null) {
					responseError(null, OAuthErrorCode.CLIENT_USER_CANCEL.getCode()
							, OAuthErrorCode.CLIENT_USER_CANCEL.getDesc());
					return ;
				}
				mCustomTabManager.launchUrl(packageInfo.packageName, oauthUrl);
			}
		});

		fragment.show(ft, CustomTabDialogFragment.DIALOG_TAG);
	}

	@Override
	protected void onResume() {
		super.onResume();
		if (mIsCustomTabOpen) {
			responseError(null, OAuthErrorCode.CLIENT_USER_CANCEL.getCode()
					, OAuthErrorCode.CLIENT_USER_CANCEL.getDesc());
		}

		mIsCustomTabOpen = true;

	}

	private void returnResult(@NonNull Intent data) {
		data.setAction(CustomTabsManager.ACTION_NAVER_CUSTOM_TAB);
		mCustomTabManager.sendCustomTabResult(data);
		setResult(RESULT_CANCELED);
		finish();
	}

	@Override
	protected void onPause() {
		super.onPause();
		if(mAlertDialog != null) {
			mAlertDialog.dismiss();
		}
	}

	/**
	 * lounchMode에서 singleTask로 설정한 후
	 * intent url을 통해 호출 할 경우 onCreate가 아닌 이 메서드가 호출 된다.
	 *
	 * @param intent intent url을 통해 넘어온 파라메터를 파싱하여 생성된 {@link Intent}
	 */
	@Override
	protected void onNewIntent(Intent intent) {
		Logger.d(TAG, "open by Intent url");
		String code = intent.getStringExtra("code");
		String state = intent.getStringExtra("state");
		String error = intent.getStringExtra("error");
		String errorDescription = OAuthWebviewUrlUtil.getDecodedString(intent.getStringExtra("error_description"));

		if (code != null || error != null) {
			responseResult(code, state, error, errorDescription);
		} else {
			responseError(state
					, OAuthErrorCode.CLIENT_ERROR_PARSING_FAIL.getCode()
					, OAuthErrorCode.CLIENT_ERROR_PARSING_FAIL.getDesc());
		}
	}

	/**
	 * 요청 후 Intent 파라메터로 넘어온 oauth결과를 저장한 Intent를 생성한다.
	 *
	 * @param code             결과 코드
	 * @param state            요청시 전달한 csrf 코드
	 * @param error            에러 코드
	 * @param errorDescription 에러 메시지
	 * @return 파라메터를 저장한 인텐트
	 */
	private void responseResult(String code, String state, String error, String errorDescription) {
		Intent resultIntent = new Intent();
		resultIntent.putExtra(OAuthIntent.EXTRA_OAUTH_CODE, code);
		resultIntent.putExtra(OAuthIntent.EXTRA_OAUTH_STATE, state);
		resultIntent.putExtra(OAuthIntent.EXTRA_OAUTH_ERROR_CODE, error);
		resultIntent.putExtra(OAuthIntent.EXTRA_OAUTH_ERROR_DESCRIPTION, errorDescription);

		returnResult(resultIntent);
	}

	/**
	 * 내부 앱 문제로 인해 에러가 발생할 때 리턴할 인텐트
	 *
	 * @param state state 값
	 * @param code 에러 코드
	 * @param desc 에러 메시지
	 * @return 에러 내용이 저장된 인텐트
	 */
	private void responseError(String state, String code, String desc) {
		Intent intent = new Intent();
		intent.putExtra(OAuthIntent.EXTRA_OAUTH_STATE, state);
		intent.putExtra(OAuthIntent.EXTRA_OAUTH_ERROR_CODE, code);
		intent.putExtra(OAuthIntent.EXTRA_OAUTH_ERROR_DESCRIPTION, desc);

		returnResult(intent);
	}

	@Override
	protected void onRestoreInstanceState(Bundle savedInstanceState) {
		super.onRestoreInstanceState(savedInstanceState);
		Logger.d(TAG, "load custom tab open state");
		mIsCustomTabOpen = savedInstanceState.getBoolean(SAVE_CUSTOM_TAB_OPEN, false);
	}

	@Override
	protected void onSaveInstanceState(Bundle outState) {
		super.onSaveInstanceState(outState);
		Logger.d(TAG, "save custom tab open state");
		outState.putBoolean(SAVE_CUSTOM_TAB_OPEN, mIsCustomTabOpen);
		mCustomTabManager = new CustomTabsManager(this);
	}

	@Override
	public void onConfigurationChanged(Configuration newConfig) {

		super.onConfigurationChanged(newConfig);
	}
}
