package com.nhn.android.idp.common.connection;

import android.content.Context;

import com.nhn.android.idp.common.connection.ResponseData.ResponseDataStat;
import com.nhn.android.idp.common.logger.Logger;
import com.nhn.android.idp.common.util.CookieUtil;
import com.nhn.android.idp.common.util.DeviceAppInfo;
import com.nhn.android.idp.common.util.HttpConnectionUtil;
import com.nhn.android.naverlogin.OAuthLoginDefine;

import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLHandshakeException;
import javax.net.ssl.SSLKeyException;
import javax.net.ssl.SSLPeerUnverifiedException;
import javax.net.ssl.SSLProtocolException;
/*
 * 서버 request시 사용할 수 있는 class
 * URL로 request하고 response를 리턴해주는 method 모음 
 * @author naver
 *
 */
public class CommonConnection {
	private static final String TAG = "CommonConnection";

	protected static HttpURLConnection 		mHttpUrlConnection = null;
	
	// when user-cancel, this is set.
	protected static boolean mCancel;

 
	/**
	 * URL 로 request 후 reponse로 ResponseData를 리턴
	 * @param context context
	 * @param strRequestUrl request url
	 * @param cookies cookie string 
	 * @param userAgent useragent
	 * @return : url request 로 얻은 response data를 리턴. response data는 content 와 status-code, cookie 로 구성됨
	 */
	public static ResponseData request(Context context, String strRequestUrl, String cookies, String userAgent) {
		return request(context, strRequestUrl, cookies, userAgent, false);
	}
	

	public static ResponseData request(Context context, String strRequestUrl, String cookies, String userAgent, String authHeader) {
		return request(context, strRequestUrl, cookies, userAgent, authHeader, false, OAuthLoginDefine.TIMEOUT);
	}
 
	public static ResponseData request(Context context, String strRequestUrl, String cookies, String userAgent, boolean httpClientIsolated) {
		return request(context, strRequestUrl, cookies, userAgent, null, httpClientIsolated, OAuthLoginDefine.TIMEOUT);
	}
	
	public static ResponseData request(Context context, String strRequestUrl, String cookies, String userAgent, String authHeader, boolean httpClientIsolated) {
		return request(context, strRequestUrl, cookies, userAgent, authHeader, httpClientIsolated, OAuthLoginDefine.TIMEOUT);
	}

	/*
	 * get 방식의 login 관련 request 를 해주는 메쏘드 
	 */
	public static ResponseData request(Context context, String strRequestUrl, String cookies, String userAgent, String authHeader, boolean httpClientIsolated, int timeout) {

		ResponseData res = new ResponseData();
		List<String> postCookies = new ArrayList<String>();
				
		HttpsURLConnection httpClient = null;
		// get 방식은 output 없음 
		//OutputStream outputStream = null;
		
		synchronized(CommonConnection.class) {
			
			if (httpClientIsolated) {
				
			} else {
				if (mHttpUrlConnection != null) {
					res.setResultCode(ResponseDataStat.BUSY, "HttpClient already in use.");
					return res;
				}
			}
			
			if (!Logger.isRealVersion()) {
				Logger.d(TAG, "request url : " + strRequestUrl);
			}
			
			if (strRequestUrl == null || strRequestUrl.length() == 0) {
				res.setResultCode(ResponseDataStat.URL_ERROR, "strRequestUrl is null");
				return res;  
			}

			try {
					
				// HttpClient 설정 
				if (httpClientIsolated) {
					if (userAgent != null && userAgent.length() > 0) {
						httpClient = getDefaultHttpsConnection("GET", strRequestUrl, userAgent, timeout);
					} else {
						httpClient = getDefaultHttpsConnection("GET", strRequestUrl, context, timeout);
					}
				} else {
					if (userAgent != null && userAgent.length() > 0) {
						mHttpUrlConnection = getDefaultHttpsConnection("GET", strRequestUrl, userAgent, timeout);
					} else {
						mHttpUrlConnection = getDefaultHttpsConnection("GET", strRequestUrl, context, timeout);
					}	
				}
			} catch (MalformedURLException e) {
				res.setResultCode(ResponseDataStat.URL_ERROR, "malformedUrl : " + e.getMessage());
				e.printStackTrace();
				return res;
			} catch (IOException e) {
				res.setResultCode(ResponseDataStat.CONNECTION_FAIL, "connection open fail : " + e.getMessage());
				e.printStackTrace();
				return res;
			} catch (Exception e) {
				res.setResultCode(ResponseDataStat.EXCEPTION_FAIL, "unknown fail : " + e.getMessage());
				Logger.e(TAG, "exception step : connection establishing");
				e.printStackTrace();
				return res;
			}
			
			mCancel = false;
		}
		
		// cookie 값 설정 
		if (cookies == null || cookies.length() == 0) {
			cookies = CookieUtil.getAllNidCookie();
			Logger.d(TAG, "request() --- request with naverCookie!");
			if (!Logger.isRealVersion()) {
				Logger.d(TAG, "request() --- " + cookies);
			}
		} else {
			Logger.d(TAG, "request() --- request with user Cookie!");
			if (!Logger.isRealVersion()) {
				Logger.d(TAG, "request() --- " + cookies);
			}
		}
				
		try {
			if (httpClientIsolated) {
				if (null != cookies && cookies.length() > 0) {
					httpClient.setRequestProperty("Cookie", cookies);
				}
				if (null != authHeader && authHeader.length() > 0) {
					httpClient.setRequestProperty("Authorization", authHeader);
				}
	
				int responseCode = httpClient.getResponseCode();
				Logger.i(TAG, "response status code:" + responseCode);
				
				postCookies = CookieUtil.getCookieUpperSDK23(httpClient.getHeaderFields());
				String contentType = HttpConnectionUtil.getCharsetFromContentTypeHeader(httpClient.getHeaderFields());			
				
				InputStream in = null;
				try {
					in = httpClient.getInputStream();
				} catch (IOException e) {
					e.printStackTrace();
					in = httpClient.getErrorStream();
				}
				
				res.setResponseData(responseCode, contentType, in, postCookies);
		        
			} else {
				if (null != cookies && cookies.length() > 0) {
					mHttpUrlConnection.setRequestProperty("Cookie", cookies);
				}
				if (null != authHeader && authHeader.length() > 0) {
					mHttpUrlConnection.setRequestProperty("Authorization", authHeader);
				}
	
				int responseCode = mHttpUrlConnection.getResponseCode();
				Logger.i(TAG, "response status code:" + responseCode);
				
				postCookies = CookieUtil.getCookieUpperSDK23(mHttpUrlConnection.getHeaderFields());
				String contentType = HttpConnectionUtil.getCharsetFromContentTypeHeader(mHttpUrlConnection.getHeaderFields());
				
				InputStream in = null;
				try {
					in = mHttpUrlConnection.getInputStream();
				} catch (IOException e) {
					e.printStackTrace();
					in = mHttpUrlConnection.getErrorStream();
				}

				res.setResponseData(responseCode, contentType, in, postCookies);
			}
		} catch (SSLPeerUnverifiedException e) {
			res.setResultCode(ResponseDataStat.NO_PEER_CERTIFICATE, "SSLPeerUnverifiedException : " + e.getMessage());
			e.printStackTrace();
		} catch (SSLProtocolException e) {
			res.setResultCode(ResponseDataStat.NO_PEER_CERTIFICATE, "SSLProtocolException : " + e.getMessage());
			e.printStackTrace();
		} catch (SSLKeyException e) {
			res.setResultCode(ResponseDataStat.NO_PEER_CERTIFICATE, "SSLKeyException : " + e.getMessage());
			e.printStackTrace();
		} catch (SSLHandshakeException e) {
			res.setResultCode(ResponseDataStat.NO_PEER_CERTIFICATE, "SSLHandshakeException : " + e.getMessage());
			e.printStackTrace();
		} catch (SSLException e) {
			res.setResultCode(ResponseDataStat.NO_PEER_CERTIFICATE, "SSLException : " + e.getMessage());
			e.printStackTrace();
		} catch (SocketTimeoutException e) {
			res.setResultCode(ResponseDataStat.CONNECTION_TIMEOUT, "SocketTimeoutException : " + e.getMessage());
			e.printStackTrace();
		} catch (SocketException e) {
			res.setResultCode(ResponseDataStat.CONNECTION_FAIL, "SocketException : " + e.getMessage());
			e.printStackTrace();
		} catch (IOException e) {
			res.setResultCode(ResponseDataStat.EXCEPTION_FAIL, "IOException : " + e.getMessage());
			e.printStackTrace();
		} catch (Exception e) {
			res.setResultCode(ResponseDataStat.EXCEPTION_FAIL, "Exception : " + e.getMessage());
			e.printStackTrace();
		}

		try {
			if (httpClientIsolated) {
				httpClient.disconnect();
			} else {
				mHttpUrlConnection.disconnect();
			}
		} catch (Exception e) {
			Logger.write(e);
		} finally {
			if (httpClientIsolated) {
				httpClient = null;
			} else {
				mHttpUrlConnection = null;
			}
		}
		
		if (mCancel) {
			ResponseData cc = new ResponseData();
			cc.setResultCode(ResponseDataStat.CANCEL, "User cancel");
			return cc;
		}
		
		try {
			CookieUtil.setCookie(strRequestUrl, postCookies);
		} catch (Exception e) {
			res.setResultCode(ResponseDataStat.FAIL, "setCookie() failed :" + e.getMessage());
			Logger.write(e);
		}
		return res;
	}

	/**
	 * 로그인 모듈 user-agent 가 설정된 http client 를 리턴
	 * @throws IOException 
	 * @throws MalformedURLException 
	 */
	public static HttpsURLConnection getDefaultHttpsConnection(String method, String url, Context context, int timeout) throws MalformedURLException, IOException {
		String useragent = DeviceAppInfo.getUserAgent(context);
		return getDefaultHttpsConnection(method, url, useragent, timeout);
	}
	
	private static HttpsURLConnection getDefaultHttpsConnection(String method, String url, String userAgent, int timeout) throws MalformedURLException, IOException {
		return (HttpsURLConnection) getDefaultHttpConnection(method, url, userAgent, timeout);
	}
	
	public static HttpURLConnection getDefaultHttpConnection(String method, String url, Context context, int timeout) throws MalformedURLException, IOException {
		String useragent = DeviceAppInfo.getUserAgent(context);
		return getDefaultHttpConnection(method, url, useragent, timeout);
	}
	
	private static HttpURLConnection getDefaultHttpConnection(String method, String url, String userAgent, int timeout) throws MalformedURLException, IOException {
		HttpURLConnection urlConn = (HttpURLConnection) (new URL(url)).openConnection();

		urlConn.setDefaultUseCaches(false);
		urlConn.setUseCaches(false);
		
		urlConn.setRequestMethod(method);
		urlConn.setRequestProperty("User-Agent", userAgent);
        // TODO 파일 다운로드시엔 다른거 사용해야함. urlConn.setRequestProperty("Content-Type", "Application/xml");
		
		urlConn.setReadTimeout(timeout);
		urlConn.setConnectTimeout(timeout);
		
		urlConn.setDoInput(true);
		
		if ("GET".equalsIgnoreCase(method)) {
			urlConn.setDoOutput(false);
		} else {
			urlConn.setDoOutput(true);
		}
		
		return urlConn;
	}
	public static boolean isBusy() {
		if (mHttpUrlConnection != null)
			return true;
		return false;
	}
	
	public static void cancel() {
		mCancel = true;
		if (mHttpUrlConnection != null) {
			Logger.e(TAG, "cancel() https-connection shutdown");
			mHttpUrlConnection.disconnect();
			mHttpUrlConnection = null;
		}
		// executor 는 cancel 안해줌 -> 동작이 복잡해짐
	}

}
