package com.nasserkhosravi.appcomponent.data


import com.nasserkhosravi.appcomponent.AppComponent
import com.nasserkhosravi.appcomponent.error.ErrorPublisher
import com.nasserkhosravi.appcomponent.error.HttpException

import io.reactivex.Observer
import io.reactivex.disposables.Disposable
import retrofit2.Response

/**
 * Created by Nasser Khosravi on 12/15/2017.
 */
abstract class RxRetrofitObserver<T> constructor(val appComponent: AppComponent) : Observer<Response<GeneralResponse<T>>> {

    private var disposable: Disposable? = null

    /**
     * used to tracking and identifying service in another classes
     */
    var requestId = -1

    constructor(appComponent: AppComponent,requestId: Int) : this(appComponent) {
        this.requestId = requestId
    }

    /**
     * calls if response is successful in #onNext
     */
    fun onSuccessful(response: T) {}

    /**
     * calls in firing service
     */
    override fun onSubscribe(d: Disposable) {
        this.disposable = d
        appComponent.getViewAccessor().progress.show()
    }

    /**
     * inspect result of service
     */
    override fun onNext(response: Response<GeneralResponse<T>>) {
        if (response.isSuccessful) {
            onSuccessful(response.body()!!.response)
        } else {
            throwHttpError()
        }
    }

    override fun onError(e: Throwable) {
        e.printStackTrace()
        ErrorPublisher.publishException(e, requestId)
        onComplete()
    }

    /**
     * calls either service completed or encountered to error
     */
    override fun onComplete() {
        if (disposable != null) {
            if (!disposable!!.isDisposed) {
                disposable!!.dispose()
            }
        }
        appComponent.getViewAccessor().progress.hide()
    }

    /**
     * moves execution to onError for having error in one place
     */
    protected fun throwHttpError() {
        onError(HttpException())
    }

}
